import json
import os
import time

import requests
import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_util import ProjectApi
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.software_package_util import find_software_package_by_name

from plugins.CSBS.common.constant import KARBOR_ROOT_NAME
from plugins.CSBS.common.constant import COMPONENT
from plugins.CSBS.common.iam_util import IamUtil
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common import constant
from plugins.CSBS.common.params_tool import ParamTool, VmsName
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.step_base import DeployBaseSubJob

logger.init("CSBS")
SYSTEM_TYPE = "CSBS-VBS"
RETRY_TIME_LIMIT = 5
PASSWD_CHANGE_MAX_BETWEEN = 90
INSECURE = False
DJMANAGER_DESCRIPTION = {
    "zh-cn": "考虑到对系统安全的影响，在登录节点时，不能使用root用户直接登录，"
             "需要由djmanager用户登录后通过执行命令 \"sudo su - root\"切换为root用户",
    "en-us": "To ensure system security, you cannot log in to the node as "
             "user root. You need to log in to the node as user djmanager "
             "and run the \"sudo su - root\" command to switch to user root"}
ROOT_DESCRIPTION = {
    "zh-cn": "使用该帐户登录karbor服务器进行运维操作，查看节点配置，"
             "查看日志，巡检和信息收集等。",
    "en-us": "Used to log in to an karbor server to perform O&M operations, "
             "view node configurations and logs, perform inspection, "
             "and collect information."}
DJMANAGER_RISK_MESSAGE = {"zh-cn": "djmanager", "en-us": "djmanager"}
ROOT_RISK_MESSAGE = {"zh-cn": "root", "en-us": "root"}
SLEEP_SECONDS = 300


class RegisterCMDB(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(RegisterCMDB, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.param_util = ParamUtil()
        self.service_name = "CSBS-VBS"
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.version = self.get_version()
        self.region_id = self.param_util.get_param_value(
            self.pod_id, "public", "region_id", "region0_id")
        self.cmdb_service_name = f"{self.service_name}_{self.region_id}"
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)
        self.iam_util = IamUtil(self.project_id, self.pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self._register_cmdb(pod_id)
        except HCCIException as err:
            logger.error("Failed executing RegisterCMDB, ErrMsg: {}".format(str(err)))
            return Message(500, err)
        except Exception as err:
            logger.error("Failed executing RegisterCMDB, ErrMsg: {}".format(err))
            return Message(500, HCCIException("640001", str(err)))

    def _register_cmdb(self, pod_id):
        csbs_cloud_service = self._get_csbs_cloud_service(pod_id)
        logger.info('Current csbs_cloud_service: {}'.format(csbs_cloud_service))
        logger.info('Start calling cmdb_util.set_cloud_service_info to execute RegisterCMDB ')
        self.cmdb_util.set_cloud_service_info(self.region_id, csbs_cloud_service)
        logger.info("CSBS-VBS Karbor scale set cloud service success")

        result = self.cmdb_util.get_cloud_service_info(self.region_id, self.service_name, self.version)
        logger.info('Calling get_cloud_service_info to get csbs info ,result:{}'.format(result))
        if not result:
            raise Exception('Failed Calling get_cloud_service_info to get csbs info')

        time.sleep(SLEEP_SECONDS)
        logger.info("Start register ManageOne OC.")
        try:
            if not self.register_unify_password():
                logger.error("Failed register ManageOne OC unify pass_word.")
                return Message(500, HCCIException("640009", "Failed register ManageOne OC"))
        except Exception as err:
            logger.error(f"Execute error:{str(err)}")
            return Message(500, HCCIException(640009, str(err)))
        logger.info("Succeed register ManageOne OC unify pass_word.")
        logger.info('Succeed executing RegisterCMDB')
        return Message(200)

    def _get_csbs_cloud_service(self, pod_id):
        # 设置云服务信息
        service_scale = ParamUtil().get_project_scales_by_project_id(self.project_id).get("scale")
        cipher_type = self.param_tool.get_cipher_type()
        if not cipher_type:
            raise Exception("Failed to obtain the cipher type.")
        # dmk_float_ip 用于fcu独立升级
        region_dmk_float_ip = self.param_util.get_param_value(pod_id, 'DMK', 'dmk_floatIp', 'dmk_floatIp')
        csbs_cloud_service = {"indexName": self.service_name,
                              "version": self.version,
                              "name": self.cmdb_service_name,
                              "extendInfos": [
                                  {
                                      "key": "CipherType",
                                      "value": cipher_type
                                  },
                                  {
                                      "key": "dmk_floatIp",
                                      "value": region_dmk_float_ip
                                  },
                                  {
                                      "key": "region_service_scale",
                                      "value": service_scale,
                                  }
                              ]
                              }
        return csbs_cloud_service

    def get_project_info(self):
        project_scence = {}
        project_api = ProjectApi()
        is_primary_region = project_api.get_project_info(self.project_id) \
            .get('is_primary_region')
        replication_dict = \
            project_api.get_project_ha_dr_status(self.project_id)
        project_scence['is_primary_region'] = is_primary_region
        project_scence['RegionConHA'] = replication_dict['RegionConHA']
        project_scence['GlobalConDR'] = replication_dict['GlobalConDR']
        project_scence['ExpansionFeature_GlobalConDR'] = \
            replication_dict['ExpansionFeature_GlobalConDR']
        return project_scence

    def get_vm_names(self):
        logger.info('Start getting vm names')
        project_scence = self.get_project_info()
        vm_names = [VmsName.KARBOR1, VmsName.KARBOR2, VmsName.KARBOR3]
        vm_names_ha = [VmsName.KARBOR_HA1, VmsName.KARBOR_HA2]
        # 从region无console
        if not project_scence.get('is_primary_region') and not project_scence.get('GlobalConDR'):
            csbs_vm_names = vm_names
            if project_scence.get('RegionConHA'):
                csbs_vm_names = vm_names_ha
        # 主备region
        else:
            csbs_vm_names = vm_names
            if project_scence.get('RegionConHA'):
                csbs_vm_names = vm_names_ha

        # 扩容管理面容灾场景
        logger.info('Succeed getting vm names, current vm names: {}'
                    .format(csbs_vm_names))
        return csbs_vm_names

    def get_vm_ips(self):
        logger.info('Start getting lld uploading Ips')
        service_ip01 = \
            self.param_util.get_value_from_cloudparam(self.pod_id,
                                                      COMPONENT,
                                                      "CSBS_Service1_ip")
        service_ip02 = \
            self.param_util.get_value_from_cloudparam(self.pod_id,
                                                      COMPONENT,
                                                      "CSBS_Service2_ip")
        service_ip03 = \
            self.param_util.get_value_from_cloudparam(self.pod_id,
                                                      COMPONENT,
                                                      "CSBS_Service3_ip")
        ips_list = [service_ip01, service_ip02, service_ip03]
        logger.info('lld uploading Ips: {}'.format(ips_list))
        return ips_list

    def handle_vm_names_ips(self):
        vm_names = self.get_vm_names()
        ips = self.get_vm_ips()
        # 未上传ip, 获取空字符串
        ips = [ip for ip in ips if ip]
        zip_vm_names_ips = list(zip(vm_names, ips))
        logger.info('zip vm names and ips :{}'.format(zip_vm_names_ips))
        return zip_vm_names_ips

    def get_version(self):
        logger.info("Start getting version from karbor package")
        is_cpu_arm = get_project_condition_boolean(self.project_id,
                                                   'manageARM')
        pkg_suffix = constant.PKG_SUFFIX_ARM if is_cpu_arm else constant.PKG_SUFFIX_X86
        pkg_path, file_name = find_software_package_by_name("OceanStor BCManager", pkg_suffix)
        pkg = os.path.join(pkg_path, file_name)
        if not file_name:
            raise HCCIException("640010")
        version = self.param_tool.get_pkg_version(pkg)
        if not version:
            raise Exception("Failed getting version from Karbor's package.")
        if version:
            logger.info('Succeed getting version:{}.'.format(version))
            return str(version)
        return ''

    def register_unify_password(self):
        iam_token = self.iam_util.get_iam_token(self.iam_util.get_iam_account_info())
        if not iam_token:
            logger.error("Get iam token failed.")
            return False
        header = {
            "Content-Type": "application/json",
            "X-Auth-Token": iam_token
        }
        oc_server_url = self.param_tool.get_oc_url()
        register_url = f"{oc_server_url}/rest/mounpwdservice/v1/account"
        register_info = self._build_unify_password_info()
        if not register_info:
            logger.error("Build unify pass_word info failed, so not register.")
            return False
        register_retry_times = 0
        logger.info(f"Register unify pass_word url:{register_url}.")
        while True:
            try:
                return self._check_register_unify_pwd_result(register_url, register_info, header)
            except Exception as err:
                logger.error(f"Register karbor service to mo unify pass_word occur exc, info: {err}")
            register_retry_times += 1
            if register_retry_times > RETRY_TIME_LIMIT:
                logger.error("Unify pass_word reach the maximum number of retries, so do not try again")
                return False
            time.sleep(30)

    @staticmethod
    def _check_register_unify_pwd_result(register_url, register_info, header):
        rsp = requests.request("POST", register_url, data=json.dumps(register_info), verify=INSECURE,
                               headers=header)
        logger.info(f"Register unify pass_word result code:{rsp.status_code}")
        if rsp.status_code <= 300 and rsp.content:
            rsp_content = json.loads(rsp.content)
        else:
            return False
        if rsp_content.get("code") == "00000000":
            logger.info("Register karbor service to mo unify pass_word success, "
                        "response code:{}".format(rsp_content.get("code")))
            return True
        logger.error("Register karbor service to mo unify pass_word failed, msg:{}".format(rsp.content))
        raise Exception("Register karbor service to mo unify pass_word failed")

    def _build_unify_password_info(self):
        """Construct information that needs to be registered to mo.
        :return:
        """
        node_list = self.karbor_util.get_karbor_node_list()
        account_list = list()

        for node in node_list:
            djmanager_account = self._build_account(node.user, node.user_pwd, node, self.region_id)
            root_account = self._build_account(node.root_name, node.root_pwd, node, self.region_id)
            account_list.append(djmanager_account)
            account_list.append(root_account)

        iam_account_data = self._build_iam_account_unify_password_info()
        account_list.extend(iam_account_data)
        register_info = {
            "componentName": SYSTEM_TYPE,
            "subComponents": [{"subComponentName": SYSTEM_TYPE,
                               "createdAccountList": account_list}]
        }
        return register_info

    def _build_account(self, username, password, node, region_id):
        expire_time_params = self._get_last_passwd_change_and_passwd_expires(node, username)
        last_passwd_change = expire_time_params.get("last_passwd_change")
        passwd_complexity = dict()
        if username == constant.KARBOR_ROOT_NAME:
            passwd_expires = -1
            account_description = json.dumps(ROOT_DESCRIPTION)
            risk_message = json.dumps(ROOT_RISK_MESSAGE)
            passwd_complexity = {"difok": 3, "minlen": 8}
        else:
            passwd_expires = expire_time_params.get("passwd_expires")
            account_description = json.dumps(DJMANAGER_DESCRIPTION)
            risk_message = json.dumps(DJMANAGER_RISK_MESSAGE)
            passwd_complexity["passwdChangeMaxBetweenTime"] = PASSWD_CHANGE_MAX_BETWEEN
        account = {
            "accountName": username,
            "modifyType": 1,
            "region": region_id,
            "ip": node.node_ip,
            "accountType": 1,
            "usedScene": "karbor",
            "accountDescription": account_description,
            "passwd": password,
            "oldPasswd": password,
            "lastPasswdChange": last_passwd_change,
            "passwdExpires": passwd_expires,
            "riskMessage": risk_message,
            "passwdComplexity": passwd_complexity,
            "urlModify": "/v1.5/unify_passwd/modify",
            "urlValid": "/v1.5/unify_passwd/check",
            "urlSyncStatus": "/v1.5/unify_passwd/tasks",
            "operationType": 0
        }
        return account

    def _get_last_passwd_change_and_passwd_expires(self, node, username):
        try:
            return self.__get_result_dict(node, username)
        except Exception as err:
            logger.error("Failed to get password expired time, err_msg: {}.".format(str(err)))
            raise Exception("Failed to get password expired time") from err

    def __get_result_dict(self, node, username):
        karbor_client = SshClient()
        ssh_client = karbor_client.get_ssh_client(node)
        chage_cmd = "chage -l {} ".format(username)
        awk_cmd = "awk -F ': ' {'print $2'}"
        cmds = {
            "last_passwd_change": '|'.join([chage_cmd, "grep 'Last password change'", awk_cmd]),
            "passwd_expires": '|'.join([chage_cmd, "grep 'Password expires'", awk_cmd])
        }
        result_dict = dict()
        for key, cmd in cmds.items():
            ret = karbor_client.ssh_exec_command_return(ssh_client, cmd)
            if not karbor_client.is_ssh_cmd_executed(ret):
                logger.error("Failed to execute cmd: {}.".format(cmd))
            time_expire_str = ret[0]
            if time_expire_str == "never":
                result_dict[key] = -1
            else:
                time_expire = time.mktime(time.strptime(time_expire_str, '%b %d, %Y'))
                result_dict[key] = int(round(time_expire * 1000))
        karbor_client.ssh_close(ssh_client)
        return result_dict

    def _build_iam_account_unify_password_info(self):
        def _build_data(username, pwd, _ip=None):
            return {
                "accountName": username,
                "passwd": pwd,
                "oldPasswd": pwd,
                "ip": _ip,
                "modifyType": 3,
                "region": self.region_id,
                "accountType": 5,
                "accountDescription":
                    "{\"zh-cn\": \"IAM鉴权帐号\","
                    "\"en-us\": \"IAM authentication account\"}",
                "passwdExpires": -1,
                "riskMessage":
                    "{\"zh-cn\": \"非必要场景不建议修改此密码\","
                    "\"en-us\": \"You are advised not to modify the "
                    "password when it is unnecessary.\"}",
                "passwdComplexity": {"difok": 3, "minlen": 8},
                "urlValid": "/v1.5/unify_passwd/check",
                "operationType": 0
            }

        karbor_float_ip = self.karbor_util.get_karbor_float_ip()
        op_service_account = self.iam_util.get_iam_account_info()
        op_service_account_data = _build_data(op_service_account.account_name,
                                              op_service_account.account_pwd,
                                              karbor_float_ip)
        op_scv_account = self.iam_util.get_iam_account_info(account_type="op_svc_csbs")
        op_scv_account_data = _build_data(op_scv_account.account_name,
                                          op_scv_account.account_pwd,
                                          karbor_float_ip)
        return [op_service_account_data, op_scv_account_data]
