import utils.common.log as logger
from plugins.CSBS.common.karborproxy.utils import KarborProxyOperation

from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.karborproxy.agent_proxy_param import AgentProxy
from plugins.CSBS.scripts.deploy.karborproxy.config.operation_ini import ReadConfig
from plugins.CSBS.scripts.deploy.karborproxy.config.pkg_param import PkgParam
from plugins.CSBS.common.karborproxy.project_set import project_preset
from plugins.CSBS.common.karborproxy.shell_cmd import ShellCmd
from plugins.CSBS.common.karborproxy.ssh_cmd import SShCmd
from plugins.CSBS.scripts.deploy.karborproxy.libs.proxy_threading import ProxyThreading


class AgentProxyBasicConfig(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        logger.init("AgentProxyBasicConfig")
        super(AgentProxyBasicConfig, self).__init__(project_id, pod_id, regionid_list)
        self.agent_proxy = AgentProxy(project_id, pod_id)
        self.pkg_param = PkgParam()
        self.proxy_util = KarborProxyOperation(project_id)
        self.ssh = SShCmd(project_id, pod_id)

    @project_preset
    def execute(self, project_id, pod_id, regionid_list=None):
        self._exec_install_agent()
        self._exec_config_agent_ntp()
        self._exec_config_agent_dns()
        self._exec_config_conn_zook()
        self._clear_files_and_dirs()

    def exec_thread(self, cmd):
        proxy_threads = []
        for proxy_node in self.proxy_util.get_proxy_node_list():
            proxy_t = ProxyThreading(self.ssh.ssh_exe_root_cmds, proxy_node, cmd)
            proxy_threads.append(proxy_t)
        for thread in proxy_threads:
            thread.start()
        for thread in proxy_threads:
            thread.join()
            if thread.exitcode:
                raise Exception(thread.exc_traceback)

    def _exec_install_agent(self):
        logger.info("Exec Searching for the KarborProxy Package")

        ReadConfig().set_ini(self.agent_proxy)
        agent_config_file = self.agent_proxy.get_agent_config_ini_path()
        self.ssh.put_file_to_all_roots(agent_config_file)

        file_path, file_name = self.pkg_param.find_agent_pkg(self.agent_proxy.is_arm())
        self.ssh.put_file_to_all_roots(file_path)
        logger.info("Exec install the KarborProxy")
        install_agent_cmd = ShellCmd.install_agent % file_name
        self.exec_thread(install_agent_cmd)

    def _exec_config_agent_ntp(self):
        logger.info("Exec Configuring NTP for the KarborProxy")
        dns_ips = self.agent_proxy.get_dns_ips()
        cmd = ShellCmd.config_ntp % (
            dns_ips[0], dns_ips[0], dns_ips[1], dns_ips[1])
        self.ssh.ssh_root_cmds_to_all_nodes(self.agent_proxy, cmd)

    def _exec_config_agent_dns(self):
        logger.info("Exec Configuring DNS for the KarborProxy")
        dns_ips = self.agent_proxy.get_dns_ips()
        cmd = ShellCmd.config_dns % (
            dns_ips[0], dns_ips[0], dns_ips[1], dns_ips[1])
        self.ssh.ssh_root_cmds_to_all_nodes(self.agent_proxy, cmd)

    def _exec_config_conn_zook(self):
        logger.info("Exec Configuring conn zook for the KarborProxy")
        cmd = "(ps -ef |grep agent_proxy |grep -v grep) || " \
              "(proxython /opt/huawei-data-protection/agentproxy/bin" \
              "/agent_proxy.py >/dev/null 2>/opt/huawei-data-protection" \
              "/agentproxy/logs/AgentProxy/AgentProxy.lle  & ) "
        self.agent_proxy.node_ip = self.agent_proxy.karbor_proxy_ip0
        self.ssh.ssh_root_cmds(self.agent_proxy, cmd)

    def _clear_files_and_dirs(self):
        self.pkg_param.remove_all_files_and_dirs_in_path(self.pkg_param.extract_karbor_proxy_path)
