import json

import requests
import utils.common.log as logger
from utils.business.manageone_util2 import ManageOneUtil2

from plugins.CSBS.common.iam_util import IamUtil
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.karborproxy.agent_proxy_param import AgentProxyDPAInfo
from plugins.CSBS.common.karborproxy.project_set import project_preset

logger.init("KarborProxy Register unibackup")


class RegisterUniBackup(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super().__init__(project_id, pod_id, regionid_list)
        self.agent_proxy = AgentProxyDPAInfo(project_id, pod_id)
        self.mo_util2 = ManageOneUtil2()
        self.iam_token = ""
        self.iam_util = IamUtil(project_id, pod_id)

    @project_preset
    def execute(self, project_id, pod_id, regionid_list=None):
        op_service_account = self.iam_util.get_iam_account_info()
        self.iam_token = self.iam_util.get_iam_token(op_service_account)
        body = self.build_external_driver_body(
            self.agent_proxy.region_id, self.agent_proxy.management_float_ip)
        self.add_external_driver(pod_id, body)

    def add_external_driver(self, pod_id, req_body):
        """
        添加驱动
        Parameters
        ----------
        pod_id
        req_body

        Returns
        -------

        """
        om_list = self.mo_util2.getMoOmInfo(pod_id)
        om_float_ip = om_list[0]
        headers = {
            'Content-Type': 'application/json;charset=utf8',
            'Accept': 'application/json',
            'X-Auth-Token': self.iam_token
        }
        url = "https://%s:26335/rest/system/v2/action/register" % om_float_ip

        rsp = requests.post(url, headers=headers,
                            data=json.dumps(req_body), verify=False)
        if rsp.status_code == 200:
            logger.info(
                "Node %s  add external driver succ" %
                self.agent_proxy.management_float_ip)
        else:
            logger.error(
                "Node %s add external driver failed" %
                self.agent_proxy.management_float_ip)
            raise Exception('add external driver failed')

    def build_external_driver_body(self, region_id, driver_ip):
        """
        构建驱动请求体。
        Parameters
        ----------
        region_id
        driver_ip

        Returns
        -------

        """
        req_body = {
            "ip": driver_ip,
            "systemType": "KarborProxy",
            "regionId": region_id,
            "systemVersion": self.agent_proxy.agent_assist_version,
            "connectParameters": {
                "restPort": 8088,
                "restAuthType": "IAM",
                "certs": ""
            },
            "implementsFunctions": [
                {
                    "feature": "UniBackup",
                    "urlList": [
                        {"action": "bkserver",
                         "actionUrl": "/v1/unity_backup/bkserver"},
                        {"action": "backupaction",
                         "actionUrl": "/v1/unity_backup/backupaction"},
                        {"action": "taskstatus",
                         "actionUrl": "/v1/unity_backup/taskstatus"},
                        {"action": "backuppolicy",
                         "actionUrl": "/v1/unity_backup/backuppolicy"},
                        {"action": "subsystem",
                         "actionUrl": "/v1/unity_backup/subsystem"}
                    ]
                },
                {
                    "feature": "Ping",
                    "urlList": [
                        {"action": "connectivity",
                         "actionUrl": "/v1/unity_backup/connectivity"}
                    ]
                }
            ]
        }
        return req_body

    @staticmethod
    def build_unibackup_body(region: str, node_ip: str) -> dict:
        """
        构建注册统一密码的请求体
        Parameters
        ----------
        region
        node_ip: 节点 IP


        Returns
        -------
        dody: dict
        """

        initial_body = {
            "region": region,
            "systemType": "AgentProxy",
            "ip": node_ip,
            "port": "8088",
            "backupType": ["full"],
            "tranProtocal": ["SFTP"],
            "actionJson": {
                "taskstatus": "/v1/unity_backup/taskstatus",
                "bkserver": "/v1/unity_backup/bkserver",
                "backupaction": "/v1/unity_backup/backupaction",
                "backuppolicy": "/v1/unity_backup/backuppolicy"
            }

        }
        return initial_body
