import json
import os

import utils.common.log as logger

from plugins.CSBS.common.util import ConfigIniFile
from plugins.CSBS.scripts.upgrade.ab_test.base import SSOMoMixin


class ProductManager(object):
    def __init__(self, context, http_client, vdc_name=None):
        self.context = context
        self.http_client = http_client
        self.pro_id = context.engine_id
        self.region_id = self.context.engine_info['region_id']
        self.region_type = self.context.engine_info['region_type']
        self.project_id = self.context.user_info['project_id']
        self.console_host = self.context.basic_mo_test.get('console_host')
        self.vdc_name = self.context.basic_mo_test["vdc1_name"] if not vdc_name else vdc_name
        self.published_conf = None

    def _do_product_online(self, service_type, product_id, product_name):
        if not self.product_action(product_id=product_id, action_type="online", process_ids=[]):
            err_msg = f"Failed to bring the product online, service type:{service_type}, " \
                      f"product id:{product_id}, product name:{product_name}."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"The product is successfully brought online, service type:{service_type}, "
                    f"product id:{product_id}, product name:{product_name}.")
        return True

    def _do_product_offline(self, service_type, product_id, product_name):
        if not self.product_action(product_id=product_id, action_type="offline", process_ids=[]):
            err_msg = f"Failed to bring the product offline, service type:{service_type}, " \
                      f"product id:{product_id}, product name:{product_name}."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"The product is successfully brought offline, service type:{service_type}, "
                    f"product id:{product_id}, product name:{product_name}.")
        return True

    def _do_modify_product_publish(self, service_type, product_name_en=None,
                                   is_default=True, is_sys_admin=True, is_publish=True):
        logger.info(f"Start to get product info list, service_type is {service_type}.")
        console_host, website = self._get_console_host_and_website(is_sys_admin)
        url = f"https://{console_host}/{website}/goku/rest/product/v3.0/products?" \
              f"limit=10&start=1&service_type={service_type.lower()}&region_id={self.region_id}"
        logger.info(f"Get product info request url:{url}.")
        _, body = self.http_client.get(url=url, verify=False)
        product_list = body.get("products", [])
        logger.info(f"Getted product info list are {product_list}.")
        tar_product_info = None
        for _product_info in product_list:
            cur_product_name = json.loads(_product_info.get("name", "{}"))
            if not self._check_product_name_is_expect(cur_product_name, product_name_en):
                continue
            if _product_info.get("service_type") == service_type and _product_info.get("is_default") is is_default:
                tar_product_info = _product_info
                break
        if not tar_product_info:
            raise Exception(f"Failed to obtain the target product information, the product may not be publish, "
                            f"service type: {service_type}, is default: {is_default}.")
        tar_product_id = tar_product_info.get("product_id", "")
        publish_vdc_id_list = self._get_publish_vdc_id_list(tar_product_info, tar_product_id, is_publish)
        if not self.product_action(product_id=tar_product_id, action_type="modify_publish", ids=publish_vdc_id_list,
                                   process_ids=[], is_sys_admin=True):
            raise Exception(f"Failed to publish the product, service type:{service_type}, product id:{tar_product_id}.")
        # 发布产品完成将服务状态写入配置文件
        if is_publish:
            self.write_published_product_type_to_conf(f"{service_type}:{product_name_en}")
        logger.info(f"Succeeded to publish the product, service type:{service_type}, product id:{tar_product_id}.")

    def _get_publish_vdc_id_list(self, tar_product_info, tar_product_id, is_publish=True):
        publish_scope_list = json.loads(tar_product_info.get("publish_scope", "{}"))
        vdc_id_list = []
        for scope in publish_scope_list:
            if not is_publish and scope.get("vdcName") == self.vdc_name:
                continue
            vdc_id_list.append(scope.get("vdcId"))
        if not vdc_id_list:
            # 修改发布范围时，对应产品至少应该已对1个租户VDC发布，获取为空，说明获取产品信息有误
            raise Exception(f"The obtained product information is incorrect, product information: {tar_product_info}.")
        if is_publish:
            test_vdc_id = self._get_product_vdc_id_by_vdc_name(tar_product_id, self.vdc_name)
            logger.info(f"The VDC name and VDC ID of the test tenant are {self.vdc_name} and {test_vdc_id}.")
            vdc_id_list.append(test_vdc_id)
        return list(set(vdc_id_list))

    def _get_product_vdc_id_by_vdc_name(self, product_id, vdc_name):
        url = f"https://{self.console_host}/moserviceaccesswebsite/goku/rest/product/moproductservice" \
              f"/v1/products/{product_id}/vdcs?limit=20&start=0"
        _, body = self.http_client.get(url=url, verify=False)
        vdc_list = body.get("vdcs", [])
        for vdc_info in vdc_list:
            if vdc_info.get("domain_name") == vdc_name:
                return vdc_info.get("id")
        raise Exception(f"Failed to obtain vdc id based on vdc name, product id:{product_id}, vdc name:{vdc_name}.")

    def get_product_list(self, service_type=None):
        logger.info(f"Start to get product info list, product_type is {service_type}.")
        url = f"https://{self.console_host}/moproductwebsite/goku/rest/product/v3.0/products?" \
              f"limit=8&start=1&service_type={service_type.lower()}&region_id={self.region_id}" \
              f"&project_id={self.project_id}"
        _, body = self.http_client.get(url=url, verify=False)
        logger.info(f"Getted product info list are {body}.")
        return body.get("products", [])

    def product_action(self, product_id, action_type, online_vdc_id=None, ids=None, process_ids=None,
                       is_sys_admin=False):
        logger.info(f"Start to exec product action, product_id:{product_id}, action_type:{action_type}, "
                    f"online_vdc_id:{online_vdc_id}, ids:{ids}, process_ids:{process_ids}.")
        console_host, website = self._get_console_host_and_website(is_sys_admin)
        url = f"https://{console_host}/{website}/goku/rest/product/v3.0/products/{product_id}/action"
        data = {"action_type": action_type}
        if ids:
            data["ids"] = ids
        if online_vdc_id:
            data["online_vdc_id"] = online_vdc_id
        if process_ids:
            data["process_ids"] = process_ids
        if action_type in ("offline", "unpublish"):
            data["process_ids"] = []
        _, body = self.http_client.put(url=url, data=json.dumps(data))
        if not body.get("result", ""):
            err_msg = f"Failed to execute product_action, action_type:{action_type}."
            logger.error(err_msg)
            raise Exception(err_msg)
        return True

    def _get_console_host_and_website(self, is_sys_admin):
        if is_sys_admin:
            console_host = self.context.basic_mo_test['manage_console_host']
            website = "moserviceaccesswebsite"
        else:
            console_host = self.context.basic_mo_test.get('console_host')
            website = "movdcwebsite"
        return console_host, website

    def _handing_product_unpublish_scene(self, service_type, product_name_en=None, product_name_zh=None,
                                         is_default=True):
        # 使用系统管理员登录
        SSOMoMixin(self.context, self.http_client, is_sys_admin=True)
        self._do_modify_product_publish(service_type, product_name_en, is_default, is_sys_admin=True)
        # 使用VDC用户登录
        SSOMoMixin(self.context, self.http_client)
        product_list = self.get_product_list(service_type)
        if not product_list:
            raise Exception(f"No product information is obtained, the service type:{service_type}.")
        product_id = None
        product_name = None
        online_status = None
        for product_info in product_list:
            cur_product_name = json.loads(product_info.get("name", "{}"))
            if product_info.get("is_default") is is_default and self._check_product_name_is_expect(
                    cur_product_name, product_name_en, product_name_zh):
                product_id = product_info.get("product_id")
                product_name = product_info.get("product_name")
                online_status = product_info.get("online_status")
                break
        if not product_id:
            raise Exception("Failed to obtain product information.")
        self._handing_product_approval_scene(service_type, product_id, product_name, online_status)
        return product_id

    def _handing_product_offline_scene(self, service_type, product_id, product_name):
        self._do_product_online(service_type, product_id, product_name)

    def _handing_product_approval_scene(self, service_type, product_id, product_name, online_status):
        if online_status.lower() == "online":
            self._do_product_offline(service_type, product_id, product_name)
        self._do_product_online(service_type, product_id, product_name)

    @staticmethod
    def _check_product_name_is_expect(cur_product_name, product_name_en=None, product_name_zh=None):
        cur_product_name_en = cur_product_name.get('en_US').replace(" ", "")
        cur_product_name_zh = cur_product_name.get('zh_CN').replace(" ", "")
        if product_name_en and product_name_en != cur_product_name_en:
            logger.warning(f"Current product en name: {cur_product_name_en}, target product en name:{product_name_en}.")
            return False
        if product_name_zh and product_name_zh != cur_product_name_zh:
            logger.warning(f"Current product zh name: {cur_product_name_zh}, target product zh name:{product_name_zh}.")
            return False
        return True

    def get_product_id(self, service_type, is_default=True, product_name_en=None, product_name_zh=None):
        logger.info(f"Start to get product id, service type is {service_type}, is_default:{is_default}.")
        product_list = self.get_product_list(service_type)
        for product_info in product_list:
            logger.info(f"The product info: {product_info}.")
            product_id = product_info.get("product_id")
            if not product_id:
                raise Exception("The product id is None, check whether the obtained product information is correct.")
            if product_info.get("region_id") != self.region_id or product_info.get("service_type") != service_type:
                continue
            is_default_vol = product_info.get("is_default")
            if is_default_vol is not is_default:
                continue
            product_name = json.loads(product_info.get("name", "{}"))
            if not self._check_product_name_is_expect(product_name, product_name_en, product_name_zh):
                continue
            online_status = product_info.get("online_status")
            approval_status = product_info.get("approval")
            if approval_status:
                logger.warn(f"The product approval status is {product_info['approval']}, so need to close approval.")
                self._handing_product_approval_scene(service_type, product_id, product_name, online_status)
            elif online_status != "online":
                logger.warn(f"The product of {service_type} is not online, need to online.")
                self._handing_product_offline_scene(service_type, product_id, product_name)
            elif product_info.get("publish_status") != "publish":
                logger.warn(f"The product of {service_type} is not publish, need to publish.")
                product_id = self._handing_product_unpublish_scene(service_type, product_name_en, product_name_zh,
                                                                   is_default)
            logger.info(f"Succeed to obtain the product id, obtained product id is: {product_id}, "
                        f"product info is: {product_info}.")
            return product_id
        else:
            # 服务未发布
            return self._handing_product_unpublish_scene(service_type, product_name_en, product_name_zh, is_default)

    def do_unpublish_product(self):
        published_product_list = self.read_published_product_type_from_conf()
        logger.info(f"Published product list:{published_product_list}.")
        # 使用系统管理员登录
        SSOMoMixin(self.context, self.http_client, is_sys_admin=True)
        for product_key in published_product_list:
            service_type,  product_name_en = product_key.strip().split(":")
            if not product_name_en or product_name_en == "None":
                product_name_en = None
            logger.info(f"Start to unpublish product by vdc name, vdc_name:{self.vdc_name}, "
                        f"service_type:{service_type}, product_name_en:{product_name_en}.")
            self._do_modify_product_publish(service_type, product_name_en, is_publish=False)
            logger.info(f"Succeeded to unpublish product by vdc name, service_type:{service_type}.")
        self._get_published_conf_file()
        self.published_conf.set_value_by_key_and_sub_key("published_product", self.vdc_name, "")
        SSOMoMixin(self.context, self.http_client)
        logger.info("All published product in test project has been unpublished.")

    def write_published_product_type_to_conf(self, product_key):
        self._get_published_conf_file()
        published_product_list = self.read_published_product_type_from_conf()
        published_product_list.append(product_key)
        published_product_list = list(set(published_product_list))
        published_product_str = ",".join(published_product_list)
        logger.info(f"The product type publish in the test project is: {published_product_str}.")
        ret = self.published_conf.set_value_by_key_and_sub_key("published_product", self.vdc_name,
                                                               published_product_str)
        if not ret:
            raise Exception("Failed to write the published product info to conf file.")

    def read_published_product_type_from_conf(self):
        self._get_published_conf_file()
        published_products = None
        try:
            published_products = self.published_conf.get_value_by_key_and_sub_key("published_product", self.vdc_name)
        except Exception as err:
            logger.warning(f"Failed to obtain the published product, vdc name: {self.vdc_name}, err_msg:{err}.")
        logger.info(f"Getted published product str:{published_products}.")
        if not published_products:
            return []
        return published_products.strip().split(",")

    @staticmethod
    def _get_published_product_conf_path():
        return os.path.join(os.path.dirname(os.path.realpath(__file__)),
                            "../../../../conf/upgrade/test_published_product.ini")

    def _get_published_conf_file(self):
        if self.published_conf:
            return
        conf_path = self._get_published_product_conf_path()
        self.published_conf = ConfigIniFile(conf_path)
