import os
from typing import Dict, List

import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.business.unified_pd_platform.account_query import AccountQuery
from utils.constant.path_constant import SOURCES_ROOT_DIR

from plugins.CSBS.common import model
from plugins.CSBS.common.constant import KARBOR_INSTALL_WORKDIR, KARBOR_USER_NAME, KARBOR_ROOT_NAME, MAN_MACHINE_ACCOUNT
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.constant import ENV_CONFIG_PATH
from plugins.CSBS.common.util import ConfigIniFile, auto_retry

logger.init("CSBS KarborUtil")


class KarborUtil(object):
    def __init__(self, project_id, pod_id):
        self.karbor_node_list = []
        self.ip_list = []
        self.project_id = project_id
        self.pod_id = pod_id
        self.service_name = "CSBS-VBS"
        self.param_util = ParamUtil()
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.ssh_client = SshClient()
        self.conf_file_path = None
        self.conf_file_obj = None

    def _get_karbor_ip_list_from_cloudparam(self):
        karbor_ip_list = []
        for ip_name in ["CSBS_Service1_ip", "CSBS_Service2_ip"]:
            karbor_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, ip_name)
            if not karbor_ip:
                raise Exception(f"Failed to obtain the ip of {ip_name}.")
            karbor_ip_list.append(karbor_ip)

        if not self.param_tool.is_csha_scene(self.project_id):
            karbor3_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "CSBS_Service3_ip")
            if not karbor3_ip:
                raise Exception("Failed to obtain the ip of CSBS_Service3_ip.")
            karbor_ip_list.append(karbor3_ip)
        return karbor_ip_list

    def get_csbs_node_info_from_cmdb(self):
        try:
            region_id = self.param_tool.get_region_id()
            nodes_info = ManageOneCmdbUtil(self.project_id, self.pod_id). \
                get_deploy_node_info(region_code=region_id,
                                     index_name=self.service_name)
            if not nodes_info:
                raise Exception("Failed to obtain csbs node info.")
            logger.info(f"Succeed to obtain {self.service_name} node info.")
            return nodes_info
        except Exception as err:
            logger.error(f"Failed to obtain csbs node info, err_msg:{err}.")
            raise Exception(f"Failed to obtain csbs node info, err_msg:{err}.") from err

    def _get_karbor_ip_list_from_cmdb(self):
        ip_list = []
        for node_info in self.get_csbs_node_info_from_cmdb():
            if not (node_info and node_info.get("name").
                    startswith("Service-CSBS")):
                continue
            for ip_dic in node_info.get("ipAddresses"):
                if ip_dic.get("netPlane") == "external_om":
                    ip_list.append(ip_dic.get("ip"))
        self._set_karbor_ip_list_to_conf(ip_list)
        return ip_list

    def get_karbor_ip_list(self):
        if self.ip_list:
            return self.ip_list
        karbor_ip_list = []
        if self.param_tool.is_install_csbs(self.project_id):
            karbor_ip_list = self._get_karbor_ip_list_from_cloudparam()
        elif self.param_tool.is_installed_csbs(self.project_id):
            karbor_ip_list = self._get_karbor_ip_list_from_conf()
            logger.info(f"Obtained the karbor ip list from the configuration file, karbor_ip_list:{karbor_ip_list}.")
            if not karbor_ip_list:
                logger.info("Obtained the karbor ip list from the configuration file is empty, "
                            "and then get karbor ip list from cmdb.")
                karbor_ip_list = self._get_karbor_ip_list_from_cmdb()
        if not karbor_ip_list:
            raise Exception("Failed to obtain the karbor ips.")
        karbor_ip_list.sort()
        self.ip_list = karbor_ip_list
        return self.ip_list

    def get_karbor_float_ip(self):
        float_ip = ""
        if self.param_tool.is_modification_region_con_ha(self.project_id):
            float_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "CSBS_Float_IP1")
        elif self.param_tool.is_install_csbs(self.project_id):
            float_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "CSBS_Float_ip1")
        elif self.param_tool.is_installed_csbs(self.project_id):
            nodes_info = self.get_csbs_node_info_from_cmdb()
            for node_info in nodes_info:
                if node_info.get("name").startswith("Service-CSBS"):
                    float_ip = \
                        node_info.get("floatIpAddresses")[0].get("floatIp")
                    break
        if not float_ip:
            raise Exception("Failed to obtain the karbor float ip.")
        return str(float_ip)

    def get_karbor_internal_float_ip(self):
        float_ip = self.param_util.get_value_from_cloudparam(
            self.pod_id, self.service_name, "CSBS_Float_ip2")
        if not float_ip:
            raise Exception(f"Failed to obtain the karbor internal float ip.")
        return float_ip

    def get_karbor_user_info(self, ip_list: List[str]) -> Dict[str, model.NodeUser]:
        users = {}
        if self.param_tool.is_install_csbs(self.project_id):
            user_info = self._get_install_scene_user_info()
            for ip in ip_list:
                users[ip] = user_info
            return users

        users = self._get_upgrade_scene_user_info()
        for ip in ip_list:
            if not users.get(ip):
                raise Exception(f"Failed to get user info on node {ip}, please check OC -> unite password.")
        return users

    def _get_install_scene_user_info(self):
        karbor_user_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, "karbor_user_name")
        if not karbor_user_name:
            karbor_user_name = KARBOR_USER_NAME

        karbor_root_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, "CBS_root_account")
        if not karbor_root_name:
            karbor_root_name = KARBOR_ROOT_NAME
        user_pwd = self.param_util.get_value_from_cloudparam(
            self.pod_id, self.service_name, "karbor_user_passwd")
        if not user_pwd:
            raise Exception("Failed to obtain the pwd of djmanager.")

        root_pwd = self.param_util.get_value_from_cloudparam(
            self.pod_id, self.service_name, "karbor_root_passwd")
        if not root_pwd:
            raise Exception("Failed to obtain the pwd of karbor root.")
        return model.NodeUser(user_name=karbor_user_name,
                              user_pwd=user_pwd,
                              root_name=karbor_root_name,
                              root_pwd=root_pwd)

    def _get_upgrade_scene_user_info(self):
        """
        根据统一密码平台查询账号、密码
        :rtype: dict
        """
        region_id = self.param_tool.get_region_id()
        if not region_id:
            raise Exception("Failed to obtain region id.")
        account_list = [(KARBOR_USER_NAME, MAN_MACHINE_ACCOUNT), (KARBOR_ROOT_NAME, MAN_MACHINE_ACCOUNT)]
        account_query = AccountQuery(self.project_id, self.pod_id)
        account_info_list = account_query.get_account_info(region_id, account_list, used_scene="",
                                                           component_name="CSBS-VBS")
        if not account_info_list:
            raise Exception("Failed to get account info.")
        users: Dict[str, Dict] = {}
        user_infos: Dict[str, model.NodeUser] = {}
        for account_info in account_info_list:
            ip = account_info.get('ip')
            users.setdefault(ip, {})
            account_name = account_info.get('accountName')
            users[ip][account_name] = account_info.get('newPasswd')
            if users[ip].get(KARBOR_ROOT_NAME) and users[ip].get(KARBOR_USER_NAME):
                user_infos[ip] = model.NodeUser(user_name=KARBOR_USER_NAME,
                                                user_pwd=users[ip].get(KARBOR_USER_NAME),
                                                root_name=KARBOR_ROOT_NAME,
                                                root_pwd=users[ip].get(KARBOR_ROOT_NAME))
        return user_infos

    @auto_retry(max_retry_times=6, delay_time=60)
    def get_karbor_node_list(self):
        if self.karbor_node_list:
            return self.karbor_node_list
        ip_list = self.get_karbor_ip_list()
        user_infos = self.get_karbor_user_info(ip_list)
        karbor_nodes_list = []
        for karbor_ip in ip_list:
            user_info = user_infos.get(karbor_ip)
            cbs_node = model.Node(
                node_ip=karbor_ip,
                user=user_info.user_name,
                user_pwd=user_info.user_pwd,
                root_name=user_info.root_name,
                root_pwd=user_info.root_pwd,
                extra="")
            karbor_nodes_list.append(cbs_node)
        if not karbor_nodes_list:
            raise Exception("Failed to get karbor node list.")
        self.karbor_node_list = karbor_nodes_list
        return karbor_nodes_list

    def get_karbor_version(self):
        karbor_client = self.ssh_client.get_ssh_client(self.get_karbor_node_list()[0])
        karbor_version = dict()
        cmds_dict = {
            "short_version": "grep 'SHORT_VERSION' /opt/huawei/dj/cfg/dj.version | cut -d '=' -f 2",
            "full_version": "grep 'FULL_VERSION' /opt/huawei/dj/cfg/dj.version | cut -d ' ' -f 5"
        }
        for key, cmd in cmds_dict.items():
            result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                raise Exception("Failed to execute cmd cat /opt/huawei/dj/cfg/dj.version.")
            karbor_version[key] = result[0]
        self.ssh_client.ssh_close(karbor_client)
        return karbor_version

    @staticmethod
    def get_installer_workdir():
        return KARBOR_INSTALL_WORKDIR

    def change_meter_switch(self, tar_status="on"):
        map_dict = {"on": "true", "off": "false"}
        if tar_status not in map_dict:
            raise Exception("The tar_status attr should be 'on' or 'off'.")

        logger.info(f"Ssh to karbor node and turn {tar_status} meter switch.")
        for node in self.get_karbor_node_list():
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"source /opt/huawei/dj/inst/utils.sh;set_features --support_sdr_meter {map_dict[tar_status]}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to turn {tar_status} meter switch on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to turn {tar_status} meter switch.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def change_cipher_type(self, cipher_type="generalCipher"):
        if cipher_type not in ("generalCipher", "SMCompatible", "SMOnly"):
            raise Exception("Unsupported cipher type.")

        logger.info("Ssh to karbor node and change cipher type.")
        for node in self.get_karbor_node_list()[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            self.ssh_client.ssh_send_command_expect(karbor_client, "sudo -u openstack /bin/bash", "$")
            cmd = f"source /opt/huawei/dj/inst/utils.sh;switch_cipher_type --cipher_type {cipher_type}"
            result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to change cipher type on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info("Succeed to change cipher type.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def get_cipher_type(self):
        logger.info("Ssh to karbor node and get cipher type.")
        exec_result = None
        for node in self.get_karbor_node_list()[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = "source /opt/huawei/dj/inst/utils.sh;get_cipher_type"
            exec_result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get cipher type on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(
                f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception(
                "Failed to obtain the cipher type from karbor node.")

        result_str = "::".join(exec_result)
        if "generalCipher" in result_str:
            return "generalCipher"
        elif "SMCompatible" in result_str:
            return "SMCompatible"
        elif "SMOnly" in result_str:
            return "SMOnly"
        else:
            raise Exception(
                "Failed to obtain the cipher type from karbor node.")

    def set_karbor_endpoint(self, option, endpoint):
        logger.info(f"Ssh to karbor node and config karbor endpoint, "
                    f"option: {option}, endpoint: {endpoint}.")
        for node in self.get_karbor_node_list()[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"source /opt/huawei/dj/inst/utils.sh;set_karbor_endpoints {option} {endpoint}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to set_karbor_endpoint on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to set_karbor_endpoint, "
                        f"option: {option}, endpoint: {endpoint}.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def set_karbor_account(self, account_name, account_pwd):
        logger.info(f"Ssh to karbor node and config karbor account, "
                    f"account name: {account_name}.")

        karbor_client = self.ssh_client.get_ssh_client(
            self.get_karbor_node_list()[0])
        self.ssh_client.ssh_send_command_expect(
            karbor_client, "source /opt/huawei/dj/inst/utils.sh;set_karbor_account", "user name", 30)
        self.ssh_client.ssh_send_command_expect(
            karbor_client, account_name, "password", 30)
        result = self.ssh_client.ssh_send_command_expect(
            karbor_client, account_pwd, "successfully", 30)
        logger.info(result)
        if self.ssh_client.failed_to_return(result, "successfully", karbor_client):
            raise Exception("Set karbor account failed.")
        return True

    def get_install_type(self):
        logger.info("Ssh to karbor node and get install type.")
        exec_result = None
        cmd = "grep 'profile =' /opt/huawei/dj/cfg/sys.ini | cut -d '=' -f 2"
        for node in self.get_karbor_node_list()[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            exec_result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get cipher type on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(
                f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception(
                "Failed to obtain the cipher type from karbor node.")
        logger.info(f"Succeed to get install type {exec_result[0].strip()} on karbor node.")
        return exec_result[0].strip()

    def get_hostname(self, ip_key):
        karbor_list = self.get_karbor_ip_list()
        # noinspection PyBroadException
        try:
            return f"karbor{(karbor_list.index(ip_key) + 1)}"
        except Exception:
            return f"karbor{(len(karbor_list) + 1)}"

    def get_karbor_ip_netmask(self):
        karbor_ip_netmask = self.param_util.get_value_from_cloudparam(self.pod_id, "CSBS-VBS",
                                                                      "CSBS_Service_subnet_mask")
        if karbor_ip_netmask:
            logger.info(f"Succeed to get_karbor_ip_netmask, {karbor_ip_netmask}")
            return karbor_ip_netmask
        raise Exception("Get karbor netmask failed.")

    def set_price_rate_switch(self, target="true"):
        """This function is used to modify the price rate switch on the Karbor

        node. If the switch is on, rate_params will be added into the
        request params when Karbor places an order, otherwise the rate_params
        will not be added.
        """
        if target not in ("true", "false"):
            raise Exception("Value of target should be true or false.")
        node = self.get_karbor_node_list()[0]
        cmd = f"source /opt/huawei/dj/inst/utils.sh;set_features --sc_price_rate_switch {target}"
        logger.info(f"Start to set the sc_price_rate_witch to be {target} on the node of {node.node_ip}.")
        self.ssh_client.exec_cmd_can_retry(node, cmd, timeout=120)
        logger.info(f"Set the sc_price_rate_witch to be {target} on the node of {node.node_ip} successfully.")

    def set_ebackup_server(self, ebackup_management_ip, ebackup_az, pwd):
        get_cmd = f"source /opt/huawei/dj/inst/utils.sh;get_ebackup_server | grep -w {ebackup_management_ip}"
        set_cmd = "source /opt/huawei/dj/inst/utils.sh;" \
                  f"set_ebackup_server --op add --ebackup_url {ebackup_management_ip} --az {ebackup_az}"
        ssh_client = None
        try:
            ssh_client = self.ssh_client.get_ssh_client_user_su_root(self.get_karbor_node_list()[0])
            ret = self.ssh_client.ssh_exec_command_return(ssh_client, get_cmd)
            # 当前eBackup已经对接
            if self.ssh_client.is_ssh_cmd_executed(ret):
                logger.info("Check eBackup server is exists.")
                ret = self.ssh_client.ssh_send_command_expect(ssh_client, set_cmd, expect_str="successfully",
                                                              timeout=30)
                logger.info(f"Result of set_ebackup_server: {ret}.")
            else:
                self.ssh_client.ssh_send_command_expect(ssh_client, set_cmd, expect_str="password", timeout=30)
                cert_except = "Are you sure to download the ca cert"
                self.ssh_client.ssh_send_command_expect(ssh_client, pwd, cert_except, 30)
                ret = self.ssh_client.ssh_send_command_expect(ssh_client, "y", "last cmd: 0", 30)
                logger.info(f"Result of download ca: {ret}.")
        except Exception as ex:
            logger.error(f"Set ebackup az failed! err_msg: {ex}.")
            return False
        finally:
            if ssh_client:
                self.ssh_client.ssh_close(ssh_client)
        return True

    def _init_conf_file_obj(self):
        if not self.conf_file_obj:
            self.conf_file_path = os.path.join(SOURCES_ROOT_DIR, ENV_CONFIG_PATH)
            self.conf_file_obj = ConfigIniFile(self.conf_file_path)

    def _set_karbor_ip_list_to_conf(self, ip_list):
        if not ip_list:
            return
        if not isinstance(ip_list, (list, tuple)):
            raise Exception("The parameter of ip_list type must be list or tuple, "
                            f"ip_list is {ip_list}, type is {type(ip_list)}.")
        karbor_ip_list_str = ','.join(ip_list)
        self._init_conf_file_obj()
        ret = self.conf_file_obj.set_value_by_key_and_sub_key("karbor", "karbor_ip_list", karbor_ip_list_str)
        if not ret:
            raise Exception(f"Failed to save karbor_ip_list to conf file({self.conf_file_path}), "
                            f"key:karbor_ip_list, value:{karbor_ip_list_str}.")

    def _get_karbor_ip_list_from_conf(self):
        self._init_conf_file_obj()
        karbor_ip_list_str = self.conf_file_obj.get_value_by_key_and_sub_key("karbor", "karbor_ip_list")
        return karbor_ip_list_str.strip().split(",") if karbor_ip_list_str else []
