import os

import utils.common.log as logger
from utils.common.check_result import CheckResult
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.common.util import ConfigIniFile

logger.init("CSBS-VBS")


class PreCheckVersion(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(PreCheckVersion, self).__init__(project_id, pod_id,
                                              regionid_list)
        self.karbor_util = KarborUtil(project_id, pod_id)
        self.param_tool = ParamsTools(project_id)
        self.karbor_operation = KarborOperation(self.project_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform installation & configuration.
        """
        self._init_cipher_type_param()
        karbor_version = self.karbor_util.get_karbor_version().get("short_version")
        if not karbor_version:
            raise Exception("Failed to obtain the karbor version.")

        # 8.2.0之前版本不支持密码算法切换
        if karbor_version < "8.2.0":
            logger.error("CSBS-VBS 8.2.0 or later supports cipher type switchover. The current version is earlier "
                         f"and does not support cipher type switchover. The current version is {karbor_version}.")
            return Message(500, HCCIException(645034, karbor_version))

        logger.info(f"Succeed to check the CSBS-VBS  karbor version, "
                    f"current version is {karbor_version}.")
        return Message(200, check_results=[CheckResult(status="success")])

    def _init_cipher_type_param(self):
        env_config_path = self.param_tool.get_env_config_path()
        conf_env_file = ConfigIniFile(env_config_path)
        # 切换算法前，从karbor获取的cipher_type为cur_cipher_type
        karbor_cipher_type = self.karbor_operation.get_cipher_type()
        ret = conf_env_file.set_value_by_key_and_sub_key("karbor", "cur_cipher_type",
                                                         karbor_cipher_type)
        if not ret:
            raise Exception("Failed to save the current cipher type.")
        tar_cipher_type = self.param_tool.get_tar_cipher_type()
        ret = conf_env_file.set_value_by_key_and_sub_key("karbor", "tar_cipher_type",
                                                         tar_cipher_type)
        if not ret:
            raise Exception("Failed to save the target cipher type.")
