import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil

from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.common.upgrade.params import ParamsTools


logger.init("CSBS-VBS")


class UpdateCipherTypeToCMDB(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(UpdateCipherTypeToCMDB, self).__init__(project_id, pod_id,
                                                     regionid_list)
        self.service_name = "CSBS-VBS"
        self.cipher_type = "CipherType"
        self.karbor_operation = KarborOperation(self.project_id)
        self.param_tool = ParamsTools(project_id)
        self.cmdb_util = ManageOneCmdbUtil(project_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform installation & configuration.
        """
        karbor_cipher_type = self.karbor_operation.get_cipher_type()
        logger.info(f"Succeeded to obtain the cipher type from the karbor node. "
                    f"The cipher type is {karbor_cipher_type}.")
        try:
            self._update_cipher_type_to_cmdb(karbor_cipher_type)
        except Exception as err:
            logger.error(f"Failed to update karbor cipher type to cmdb, "
                         f"err_msg:{err}.")
            return Message(500, HCCIException(645042))
        logger.info(f"Succeed to update cipher type to cmdb, "
                    f"{self.cipher_type} is: {karbor_cipher_type}.")
        return Message(200)

    def _update_cipher_type_to_cmdb(self, karbor_cipher_type):
        # 从CMDB获取注册信息
        cmdb_service_name = f"{self.service_name}_{self.region_id}"
        logger.info(f"Start to obtain cloud service info from CMDB, "
                    f"cmdb service name:{cmdb_service_name}.")
        service_info_list = self.cmdb_util.get_cloud_service_info(
            region_code=self.region_id,
            index_name=self.service_name,
            name=cmdb_service_name)
        if not service_info_list:
            raise Exception("Failed to obtain cloud service info from CMDB.")
        logger.info(f"The cloud service information obtained "
                    f"from the CMDB is: {service_info_list}.")

        # 组织注册数据
        cloud_service_info = dict()
        for service_info in service_info_list:
            if not service_info.get("name") == cmdb_service_name:
                continue
            cloud_service_info["indexName"] = service_info.get("indexName")
            cloud_service_info["version"] = service_info.get("version")
            cloud_service_info["name"] = service_info.get("name")
            extend_info_list = []
            for extend_info in service_info.get("extendInfos"):
                if extend_info.get("key") != self.cipher_type:
                    extend_info_list.append(extend_info)
            extend_info_list.append({"key": self.cipher_type,
                                     "value": karbor_cipher_type})
            cloud_service_info["extendInfos"] = extend_info_list

        # 注册CMDB
        logger.info(f"Start to register cloud service info to CMDB, "
                    f"cloud service info is: {cloud_service_info}.")
        self.cmdb_util.set_cloud_service_info(
            region_code=self.region_id,
            cloud_service_info=cloud_service_info)
