import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil

from plugins.CSBS.common.step_base import UpgradeBaseSubJob
from plugins.CSBS.common.karborproxy.utils import KarborProxyOperation

logger.init("KarborProxy")


class ProxyUpdateCipherTypeToCMDB(UpgradeBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(ProxyUpdateCipherTypeToCMDB, self).__init__(project_id, pod_id, regionid_list)
        self.service_name = "CSBS-VBS"
        self.karborproxy_operation = KarborProxyOperation(project_id)
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform installation & configuration.
        """
        karborproxy_cipher_type = self.karborproxy_operation.get_karborproxy_cur_cipher_type()
        logger.info(f"Succeeded to obtain the cipher type from the KarborProxy node. "
                    f"The cipher type is {karborproxy_cipher_type}.")
        try:
            self._update_karborproxy_cipher_type_to_cmdb(karborproxy_cipher_type)
        except Exception as err:
            logger.error(f"Failed to update KarborProxy cipher type to cmdb, err_msg:{err}.")
            return Message(500, HCCIException(645048))
        logger.info(f"Succeed to update cipher type to cmdb, "
                    f"CipherType is: {karborproxy_cipher_type}.")
        return Message(200)

    def _update_karborproxy_cipher_type_to_cmdb(self, karborproxy_cipher_type):
        # 从CMDB获取注册信息
        cmdb_service_name = f"{self.service_name}_{self.region_id}"
        logger.info(f"Start to obtain cloud service info from CMDB, "
                    f"cmdb service name:{cmdb_service_name}.")
        service_info_list = self.cmdb_util.get_cloud_service_info(
            region_code=self.region_id,
            index_name=self.service_name,
            name=cmdb_service_name)
        if not service_info_list:
            raise Exception("Failed to obtain cloud service info from CMDB.")
        logger.info(f"The cloud service information obtained "
                    f"from the CMDB is: {service_info_list}.")

        csbs_vbs_version = service_info_list[0].get("version")
        if not csbs_vbs_version:
            raise Exception(f"Failed to obtain the {self.service_name} version.")

        extend_infos = [
            {
                "key": "ProxyCipherType",
                "value": karborproxy_cipher_type
            }
        ]
        # 更新CMDB
        logger.info(f"Start to register cloud service info to CMDB, "
                    f"cloud service info is: {service_info_list}.")
        self.cmdb_util.update_cloud_service_extend_infos(self.region_id, self.service_name, extend_infos,
                                                         version=csbs_vbs_version, name=cmdb_service_name)
