import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.business.vm_util import PublicServiceOM
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karbor.cbs_installer.cbs_create_vm import CbsCreateVM

logger.init("CSBS")


class CreateVM(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(CreateVM, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.pub_om = PublicServiceOM()
        self.params = ParamUtil()
        self.cbs_create_vm = CbsCreateVM(project_id, pod_id)
        self.network_name = self.cbs_create_vm.network_name

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self.__create_vm(pod_id)
            return Message(200)
        except HCCIException as err:
            logger.error("execute error:%s" % str(err))
            error = str(err)
        except Exception as err:
            logger.error("execute error:%s" % str(err))
            error = str(err)
        return Message(500, HCCIException(640001, error))

    def rollback(self, project_id, pod_id, regionid_list=None):
        try:
            self.__delete_vm(pod_id)
        except HCCIException as err:
            logger.error("execute error:%s" % str(err))
            return Message(500, err)
        except Exception as err:
            logger.error("execute error:%s" % str(err))
            return Message(500, HCCIException(640001, str(err)))
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        self.rollback(project_id, pod_id, regionid_list)
        return self.execute(project_id, pod_id)

    def __delete_vm(self, pod_id):
        csbs_float_ip = self._get_csbs_ip(pod_id)

        if not self.cbs_create_vm.rollback():
            raise HCCIException("640001", "Delete VM failed! Please check logs.")

        # 创建浮动ip对应的port，把ip占用删除
        self.pub_om.delete_port_by_name(pod_id, "CSBS_float_ip01", csbs_float_ip[0], self.network_name)
        self.pub_om.delete_port_by_name(pod_id, "CSBS_float_ip02", csbs_float_ip[1], self.network_name)

    def __create_vm(self, pod_id):
        csbs_float_ip = self._get_csbs_ip(pod_id)

        if not self.cbs_create_vm.install(do_rollback=False):
            raise HCCIException("640001", "Create VM failed! Please check logs.")

        # 创建浮动ip对应的port，把ip占用
        self.pub_om.create_port(pod_id, "CSBS_float_ip01", csbs_float_ip[0], self.network_name)
        self.pub_om.create_port(pod_id, "CSBS_float_ip02", csbs_float_ip[1], self.network_name)

    def _get_csbs_ip(self, pod_id):
        csbs_float_ip01 = self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS", "CSBS_Float_IP1")
        if not csbs_float_ip01:
            raise HCCIException("640003", "CSBS_Float_IP1")

        csbs_float_ip02 = self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS", "CSBS_Float_IP2")
        if not csbs_float_ip02:
            raise HCCIException("640003", "CSBS_Float_IP2")
        return [csbs_float_ip01, csbs_float_ip02]
