# -*- coding:utf-8 -*-
import utils.common.log as logger

from utils.DBAdapter.DBConnector import BaseOps
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.message import RESULT_CODE
from utils.common.ssh_util2 import Ssh

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.param_check.os_param_check import check_pwd_will_expired
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.step_base import BaseStep
from plugins.CSBS.common.upgrade.params_checker import ParamsChecker

logger.init("CSBS")


class TestPreCheck(BaseStep):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(TestPreCheck, self).__init__(project_id, pod_id)
        self.param_util = ParamUtil()
        self.ssh_client = SshClient()
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)
        self.karbor_node_list = self.karbor_util.get_karbor_node_list()
        self.param_checker = ParamsChecker(self.project_id)
        self.region_type = self._get_region_type()
        self.karbor_ip = self._get_karbor_ip()
        self.os_username, self.os_pwd = self._get_karbor_os_user_info()

    def execute(self, project_id, pod_id, *args, **kwargs):
        try:
            result = self._check_input_param_login_ability()
            if result != RESULT_CODE['SUCCESS']:
                return Message(RESULT_CODE['ERROR'], HCCIException(result, self.os_username))
            self.param_checker.check_karbor_djmanager_login_ability(self.karbor_node_list)
            self.param_checker.check_karbor_root_login_ability(self.karbor_node_list)
        except HCCIException as err:
            logger.error(f"Execute error:{str(err)}.")
            return Message(RESULT_CODE['ERROR'], err)
        except Exception as err:
            logger.error(f"Execute error:{str(err)}")
            return Message(RESULT_CODE['ERROR'], f"Failed to check karbor params, err_msg:{str(err)}.")
        logger.info("Succeed to check karbor params.")
        return Message(RESULT_CODE['SUCCESS'])

    def _check_input_param_login_ability(self):
        karbor_client = None
        try:
            karbor_client = Ssh.ssh_create_client(
                self.karbor_ip, self.os_username, self.os_pwd)
            if "Current password".lower() in karbor_client.get("prompt", "").lower():
                logger.warning(f"The {self.os_username} account password has expired, please check.")
                return 640131
            elif check_pwd_will_expired(self.ssh_client, karbor_client, self.os_username):
                logger.warning(f"Node {self.karbor_ip}, User {self.os_username} password will "
                               "expire, please change and retry.")
                return 640129
            return 200
        except Exception as e:
            if str(e).find("Bad authentication") != -1:
                logger.warning(f"The {self.os_username} account password has expired, please check.")
                return 640131
            logger.error("Failed to login to Karbor node: "
                         f"IP {self.karbor_ip}, User {self.os_username}, error: {str(e)}.")
            return 640132
        finally:
            try:
                if karbor_client:
                    Ssh.ssh_close(karbor_client)
            except Exception as e:
                logger.error(f"Failed to close ssh client: IP {self.karbor_ip}, error: {str(e)}.")

    def _get_region_type(self):
        return BaseOps().get_region_type(self.project_id)

    def _get_karbor_ip(self):
        karbor_ip = self.param_util.get_value_from_cloudparam(
            self.pod_id, 'AcceptanceTest',
            f'{self.region_type}_karbor_control_host')
        if not karbor_ip:
            raise Exception("Failed to obtain karbor ip.")
        return karbor_ip

    def _get_karbor_os_user_info(self):
        user_name = self.param_util.get_value_from_cloudparam(
            self.pod_id, 'AcceptanceTest',
            f'{self.region_type}_karbor_control_username')
        if not user_name:
            raise Exception("Failed to obtain karbor node username.")
        user_pwd = self.param_util.get_value_from_cloudparam(
            self.pod_id, 'AcceptanceTest',
            f'{self.region_type}_karbor_control_password')
        if not user_pwd:
            raise Exception("Failed to obtain karbor node pwd.")
        return user_name, user_pwd
