import configparser
import importlib
import os
import re
import sys
from urllib.parse import quote

from utils.business.manageone_util import ManageOneUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common import log as logger
from utils.common.message import Message
from utils.constant.path_constant import SOURCES_ROOT_DIR
from plugins.AcceptanceTest_Upgrade.basic_fs_test.common.ecs_basic_sys.ecs_base import ECSBase
from plugins.CSBS.common.http_client import HttpClient
from plugins.CSBS.common.upgrade.constant import TASK_FILE, ENV_CONFIG_PATH
from plugins.CSBS.common.upgrade.models import LoginParam
from plugins.CSBS.common.util import auto_retry, ConfigIniFile

logger.init("CSBS-VBS")


class SSOMoMixin(object):
    """Mainly used for login related operations."""

    def __init__(self, context, http_client, is_sys_admin=False):
        self.headers = {
            "Accept": "*/*",
            "X-Language": "zh-cn",
            "X-Requested-With": "XMLHttpRequest",
            "Connection": "keep-alive",
            "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                          "AppleWebKit/537.36 (KHTML, like Gecko) "
                          "Chrome/59.0.3071.104 Safari/537.36",
            "Content-Type": "application/json; charset=UTF-8"
        }
        self.basic_header = {
            "Content-Type": "application/x-www-form-urlencoded; charset=UTF-8",
            "Accept": "*/*",
            "Cache-Control": "no-cache"
        }
        self.http_client = http_client
        self.context = context
        self.oc_float_ip = self._get_oc_float_ip()
        self.region_id = self.context.engine_info.get('region_id')
        self.is_sys_admin = is_sys_admin
        self.login_param = self.get_login_param(self.is_sys_admin)
        self.login_console()

    def _get_oc_float_ip(self):
        mo_util2 = ManageOneUtil2()
        om_list = mo_util2.get_mo_om_info(self.context.engine_id)
        if not om_list:
            err_msg = "Failed to obtain mo om info from get_mo_om_info method"
            logger.error(err_msg)
            raise Exception(err_msg)
        return om_list[0]

    def get_login_param(self, is_sys_admin=False):
        if is_sys_admin:
            console_host = self.context.basic_mo_test['manage_console_host']
            iam_host = self.context.basic_mo_test['manage_iam_host']
            web_site = "moserviceaccesswebsite"
            username = self.context.basic_mo_test['sc_fcu_sys_username']
            pwd = self.context.basic_mo_test['sc_user_new_password']
        else:
            console_host = self.context.basic_mo_test['console_host']
            iam_host = self.context.basic_mo_test["iam_host"]
            web_site = "motenantconsolehomewebsite"
            username = self.context.basic_mo_test['vdc1_vsm_user1']
            pwd = self.context.basic_mo_test['sc_user_new_password']
        return LoginParam(console_host=console_host, iam_host=iam_host, web_site=web_site, username=username, pwd=pwd)

    @property
    def is_b2b(self):
        return (self.context.basic_mo_test['manage_console_host'] !=
                self.context.basic_mo_test['console_host'])

    @staticmethod
    def is_redirect(http_code):
        return http_code == 302

    def user_authentication(self):
        tenant_auth_url = f"https://{self.oc_float_ip}:31943/mounisso/v1/cas/sc/tenantAuth"
        encrypt_init_pwd = ManageOneUtil().get_login_pwd(self.login_param.iam_host, self.login_param.pwd)
        quote_pwd = quote(encrypt_init_pwd)
        req_body = (
            f"userName={self.login_param.username}&password={quote_pwd}&verifyCode=&locale=zh-cn&"
            f"service=https%253A%252F%252F{self.login_param.console_host}%252F{self.login_param.web_site}%252F"
        )
        _, rsp_body = self.http_client.post(
            tenant_auth_url, data=req_body, verify=False,
            headers={
                "Content-Type": "application/x-www-form-urlencoded; charset=UTF-8",
                "Accept": "*/*"
            }
        )
        third_login_url = rsp_body.get("url", "")
        if third_login_url and 'loginView.html' in third_login_url:
            third_login_url = rsp_body.get('cancelUrl', "")
        if not third_login_url:
            raise Exception("Third login url is not found.")
        if "authui" in third_login_url:
            self.basic_header.update(
                {"Accept": "text/html,application/xhtml+xml,"
                           "application/xml;q=0.9,image/webp,*/*;q=0.8"})
        logger.info("Get auth url successfully.")
        return third_login_url

    def generate_st_tgt(self):
        third_login_url = self.user_authentication()
        resp, _ = self.http_client.get(third_login_url,
                                       allow_redirects=False,
                                       headers=self.basic_header)
        if self.is_redirect(resp.status_code):
            # cookie: TGT id;
            # note: TGT(global session identifier) - Indicates that the user has logged in.
            self.headers['global_tgt'] = resp.headers.get('set-cookie')
            # location(redirect url): ST - Business system certification mark
            return resp.headers.get('Location')
        return ""

    def generate_j_session_id(self):
        service_url = self.generate_st_tgt()
        if not service_url:
            raise Exception('Ticket url is not found.')

        resp, _ = self.http_client.get(service_url,
                                       allow_redirects=False,
                                       headers=self.headers)
        if not self.is_redirect(resp.status_code):
            raise Exception("The http request is not redirect.")

        ticket_cookie = resp.headers.get('set-cookie')
        if self.is_sys_admin:
            j_session_id = re.search(r'J_SESSION_ID={(?P<sid>.*?)};', ticket_cookie).group('sid')
            if not j_session_id:
                err_msg = "Failed to obtain j_session_id."
                logger.error(err_msg)
                raise Exception(err_msg)
            j_session_id_str = '{' + j_session_id + '}'
            self.headers['Cookie'] = (
                f"theme=default; SID=Set2; MO_J_SESSION_ID={j_session_id_str}; "
                f"J_SESSION_ID={j_session_id_str}; locale=zh-cn; browserCheckResult=A; "
                f"serviceOperationMgrSelectedRegion={self.region_id}"
            )
        else:
            self.headers['Cookie'] = ticket_cookie
            service_url = f"https://{self.login_param.console_host}/{self.login_param.web_site}/"
            self.http_client.get(
                service_url, allow_redirects=False,
                headers={
                    'Cookie': ticket_cookie,
                    'Content-Type': 'application/x-www-form-urlencoded; '
                                    'charset=UTF-8',
                    'Accept': 'text/html,application/xhtml+xml,'
                              'application/xml;q=0.9,image/webp,*/*;q=0.8'
                }
            )

    @auto_retry(max_retry_times=5, delay_time=180)
    def generate_cftk(self):
        self.generate_j_session_id()
        me_url = f"https://{self.login_param.console_host}/{self.login_param.web_site}/rest/me?" \
                 "salt=0.4070987459363351"
        _, rsp_body = self.http_client.get(me_url,
                                           allow_redirects=False,
                                           headers=self.headers)
        agency_id = rsp_body.get('id')
        if not agency_id:
            raise Exception("Failed to obtain AgencyId.")
        if self.is_sys_admin:
            self.headers['Cookie'] = f"{self.headers.get('Cookie')}; AgencyId={agency_id}"
        self.headers['cftk'] = rsp_body.get('cftk')
        self.headers['ProjectName'] = rsp_body.get('region')
        self.headers['AgencyId'] = rsp_body.get('id')
        self.headers["region"] = self.region_id

        user_info = {
            'project_id': rsp_body.get('projectId'),
            'user_id': rsp_body.get('userId'),
            'region_id': self.region_id
        }
        self.context.user_info = user_info

    def switch_project(self):
        if self.is_sys_admin:
            return
        agency_id = self.headers.get('AgencyId')
        if not agency_id:
            raise Exception("Failed to obtain agency_id.")
        projects_url = (
            f"https://{self.login_param.console_host}/{self.login_param.web_site}/rest/vdc/v3.1/users/"
            f"{agency_id}/projects?start=0&limit=100&rel_projects=true"
        )
        _, rsp_body = self.http_client.get(projects_url, allow_redirects=False, headers=self.headers)
        projects = rsp_body.get('projects', [])
        for project in projects:
            region_temp = project.get('regions')[0]
            if region_temp.get('region_id') == self.headers.get('region'):
                self.context.user_info['project_id'] = project.get('id')
                self.headers['ProjectName'] = project.get('name')

    def login_console(self):
        tmp_retries = self.http_client.retries
        self.http_client.retries = 0
        try:
            self._get_cftk_and_switch_project()
        except Exception as err:
            logger.trace()
            logger.error(f'Login cf2 console failed, msg: {err}. Retry login vpc console.')
            self.retry_login_vpc_console()

        self.headers['Referer'] = f"https://{self.login_param.console_host}/{self.login_param.web_site}/"
        self.http_client.headers = self.headers
        self.http_client.retries = tmp_retries
        logger.info("Login mo console success.")

    def _get_cftk_and_switch_project(self):
        self.generate_cftk()
        self.switch_project()

    def retry_login_vpc_console(self):
        self.headers['Cookie'] = self.headers.get('global_tgt')
        login_url = (
            "https://{host}/authui/login?service="
            "https%3A%2F%2F{service_host}%2Fvpc%2F%3FagencyId%3D"
            "{agencyid}%26region"
            "%3D{Region}%26locale%3Dzh-cn".format(
                host=self.context.basic_mo_test.get('iam_host'),
                service_host=self.context.basic_mo_test.get('console_host'),
                agencyid=self.headers.get('AgencyId'),
                Region=self.headers.get('ProjectName'))
        )
        rsp, _ = self.http_client.get(login_url, allow_redirects=False,
                                      headers=self.headers)
        vpc_location = rsp.headers.get('Location')

        rsp, _ = self.http_client.get(vpc_location, allow_redirects=False,
                                      headers=self.headers)
        if not self.is_b2b:
            vpc_j_session_id = rsp.headers.get('set-cookie').split(
                'J_SESSION_ID=')[1].split('; Version=1; Path=/; Secure;')[0]
            self.headers['Cookie'] += ';J_SESSION_ID=' + vpc_j_session_id
        else:
            vpc_j_session_id = rsp.headers.get('set-cookie').split(
                'Path=/; Secure, J_SESSION_ID=')[1].split(
                ';Path=/;Secure;HttpOnly')[0]
            self.headers['Cookie'] = (
                "theme=default; SID=Set2; agencyID={agencyid}; "
                "J_SESSION_ID={j_session_id}; locale=zh-cn; "
                "browserCheckResult=A".format(
                    agencyid=self.headers.get('AgencyId'),
                    j_session_id=vpc_j_session_id)
            )

        vpc_me_url = (
            "https://{host}/vpc/rest/me?salt=0.8809960674639905".format(
                host=self.context.basic_mo_test.get('console_host'))
        )
        _, rsp_body = self.http_client.get(vpc_me_url, allow_redirects=False,
                                           headers=self.headers)
        self.headers['cftk'] = rsp_body.get('cftk')


class ExecuteMixin(object):
    """Mainly used for task execution related operations."""

    def __init__(self, context, http_client):
        self.tasks = self.get_tasks()
        self.test_job_name = ""
        self.context = context
        self.http_client = http_client
        super().__init__(context=self.context, http_client=self.http_client)

    def get_tasks(self):
        conf = configparser.ConfigParser()
        task_file = os.path.join(SOURCES_ROOT_DIR, TASK_FILE)
        conf.read(task_file)
        return dict(conf.items(self.test_job_name))

    def execute_test_job_tasks(self):
        if not self.tasks:
            logger.error('The job is not configured with tasks.')
            return

        for task_name, task_uri in self.tasks.items():
            logger.info("Start execute task: {task_name}.".format(
                task_name=task_name))
            module_name, _, class_fun_name = task_uri.rpartition('.')
            class_name = class_fun_name.split(':')[0]
            fun_name = class_fun_name.split(':')[1]

            importlib.import_module(module_name)
            cls = getattr(sys.modules.get(module_name), class_name, None)
            if not cls:
                raise Exception('Class({class_name}) cannot '
                                'be found.'.format(class_name=class_name))

            task_instance = cls(self.context, self.http_client)
            fun = getattr(task_instance, fun_name, None)
            if not fun:
                raise Exception('Fun({fun}) cannot be found.'.format(fun=fun))
            if not callable(fun):
                raise Exception('Fun({fun}) not is callable '
                                'object.'.format(fun=fun))
            fun()
            logger.info('Execute task({}) success.'.format(task_name))


class Context(object):
    """AB test context, mainly used to transfer parameters."""

    def __init__(self, engine_id):
        self.conf_file_path = None
        self.conf_file_obj = None
        self.engine_id = engine_id
        self.engine_info = ProjectApi().get_project_info(engine_id)
        self.region_id = self._get_region_id()
        self.region_type = self._get_region_type()
        self.basic_mo_test = self._wrap_get_service_cloud_param('Basic_MO_Test')
        self.basic_fs_test = self._wrap_get_service_cloud_param('Basic_FS_Test')
        self.az_id = None
        self.flavor_id = None
        self.image_id = None
        self._get_test_params()

    def _get_region_id(self):
        region_id = self.engine_info.get('region_id')
        if not region_id:
            raise Exception("Get region_id failed.")
        return region_id

    def _get_region_type(self):
        region_type = self.engine_info.get('region_type')
        if not region_type:
            raise Exception("Get region_type failed.")
        return region_type

    def _get_test_params(self):
        self._get_test_param_from_conf()
        if not self.az_id:
            self._get_preset_servers_info()

    @auto_retry(max_retry_times=5, delay_time=60)
    def _get_preset_servers_info(self):
        logger.info("Start to obtain the preconfigured VM information.")
        ecs_base = ECSBase(self.engine_id, self.region_id, self.region_type)
        servers_info = ecs_base.query_preset_servers()
        for server in servers_info:
            if not server:
                continue
            test_vm_name = server.get("name")
            logger.info(f"The obtained preconfigured VM name is {test_vm_name}.")
            self.az_id = server.get("OS-EXT-AZ:availability_zone")
            self._set_test_param_to_conf("az_id", self.az_id)
            self.flavor_id = server.get("flavor", {}).get("id")
            self._set_test_param_to_conf("flavor_id", self.flavor_id)
            self.image_id = server.get("image", {}).get("id")
            self._set_test_param_to_conf("image_id", self.image_id)
            break
        if not self.az_id:
            err_msg = "Failed to obtain the az_id."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Get preset server info success, az_id:{self.az_id}, "
                    f"flavor_id:{self.flavor_id}, image_id:{self.image_id}.")

    def _init_conf_file_obj(self):
        if not self.conf_file_obj:
            self.conf_file_path = os.path.join(SOURCES_ROOT_DIR, ENV_CONFIG_PATH)
            self.conf_file_obj = ConfigIniFile(self.conf_file_path)

    def _set_test_param_to_conf(self, sub_key, value):
        self._init_conf_file_obj()
        ret = self.conf_file_obj.set_value_by_key_and_sub_key("UpgradeTest", sub_key, value)
        if not ret:
            raise Exception(f"Failed to save upgrade test param to conf file({self.conf_file_path}), "
                            f"key:{sub_key}, value:{value}.")

    def _get_test_param_from_conf(self):
        self._init_conf_file_obj()
        test_params = self.conf_file_obj.get_params_dict_by_key_name("UpgradeTest")
        for key, attr_value in test_params.items():
            setattr(self, key, attr_value)

    def _wrap_get_service_cloud_param(self, service_name):
        params = ParamUtil().get_service_cloud_param(self.engine_id, service_name, self.region_id)
        if not isinstance(params, dict):
            return params

        return {key.replace(f'{self.region_type}_', '').replace(
            f'{self.region_type.lower()}_', ''
        ): value for key, value in params.items()}


class BaseTest(ExecuteMixin, SSOMoMixin):
    def __init__(self, engine_id):
        self.engine_id = engine_id
        self.context = Context(engine_id)
        self.http_client = HttpClient()
        self.http_client.login_fun = self.login_console
        super().__init__(context=self.context, http_client=self.http_client)

    def pre_check(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform resource pre-check before installation,
        this interface is called by the execute interface,
        The tool framework does not directly call this interface.
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform installation & configuration.
        """
        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform job job failed rollback.
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform job failed retry.
        """
        self.rollback(project_id, pod_id, regionid_list)
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Check before task execution.
        """
        return Message(200)
