# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import json
import base64

from utils.business.param_util import ParamUtil
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.DBAdapter.DBConnector import BaseOps
from plugins.eBackup.common.ebackup_rest import EbackupRest
from plugins.eBackup.common.util import SshTool
from plugins.eBackup.common.model import SshInfo

logger.init("Start to register eBackup physical server to ManageOne CMDB")


class AddDeviceToOC(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super(AddDeviceToOC, self).__init__(project_id, pod_id)
        self.class_name = 'SYS_X86Server'
        self.project_id = project_id
        self.pod_id = pod_id
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.params = ParamUtil()
        self.device_infos = []
        self.rest_api = EbackupRest()
        self.__config_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.region_id = self.__config_dict.get('region_id')
        self.bmc_info = None
        self.db_obj = BaseOps()

    @staticmethod
    def check_response(response):
        status_code = str(response.status_code)
        logger.info(f"status code: {status_code}")
        # 2 is the correct response of redfish url request
        if status_code[0] != "2":
            content = str(response.content)
            raise Exception(f"status code [{status_code}] error, content [{content}]")

    def pre_check(self, project_id, pod_id):
        """Plug-in internal interface.

        Perform resource pre-check before installation,
        this interface is called by the execute interface,
        The tool framework does not directly call this interface.
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)

    def get_uuid(self, ebk_node_ip):
        cmd = "dmidecode -s system-uuid"
        ssh_info = SshInfo(ebk_node_ip, "root", self.__config_dict.get('eBackup_os_root_password'))
        ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=False)
        result = SshTool.ssh_send_comand_return(ssh_client, cmd)
        if not result:
            raise Exception(f"Failed to get uuid")
        # 最后三位是ssh函数所返回'\n'，故删除最后2位
        uuid = result[0][:-2]
        return uuid

    def main_handle(self):
        externalom_ip_lower = self.__config_dict['datamover_externalom_iplist'].lower()
        ip_list = externalom_ip_lower.split(',')
        logger.info("Start to add node to ManageOne")
        self.device_infos.clear()
        self.bmc_info = self.db_obj.get_ebackup_install_info(self.pod_id)
        for index, host in enumerate(self.bmc_info):
            bmc_ip = host.get('bmc_ip', '')
            ebackup_ip = ip_list[index]
            name = f"eBackup {ebackup_ip}"
            bmc_usr_name = host.get('bmc_name', '')
            bmc_passwd = host.get('bmc_passwd', '')
            uuid = self.get_uuid(ip_list[index])
            redfish_url = f"https://{bmc_ip}/redfish/v1/Systems/1"
            auth_str = bmc_usr_name + ":" + bmc_passwd
            base64_str = base64.b64encode(auth_str.encode()).decode()
            headers = {
                "Content-Type": 'application/json;charset=UTF-8',
                "Authorization": 'Basic ' + base64_str
            }
            res = self._get_request(redfish_url, headers)
            sn = res.get('SerialNumber')
            device_info = {
                "instance": {
                    "name": name,
                    "ipAddress": bmc_ip,
                    "nativeId": bmc_ip,
                    "hostType": "ebackup",
                    "sn": sn,
                    "mgmtIp": ebackup_ip,
                    "deviceType": "rack",
                    "managedStatus": "in-use",
                    "productName": "eBackup_physical",
                    "hostNativeId": uuid,
                    "rackName": '01'
                },
                "mainProtocol": {
                    "protocolType": "SYS_RedfishProtocol",
                    "protocol": {
                        "userName": bmc_usr_name,
                        "password": bmc_passwd,
                        "restPort": 443
                    }
                }
            }
            self.device_infos.append(device_info)

        self.cmdb_util.batch_add_devices(self.region_id, self.class_name, self.device_infos)
        logger.info('Register eBackup physical server to ManageOne CMDB successfully.')
        return Message(200)

    def execute(self, project_id, pod_id):
        """Plug-in internal interface.

        Perform installation & configuration.
        :param project_id:
        :param pod_id:
        :return:Message object
        """
        try:
            return self.main_handle()
        except HCCIException as err:
            logger.error("Failed to register eBackup physical server to ManageOne CMDB, err_msg:{}.".format(str(err)))
            return Message(500, err)
        except Exception as err:
            logger.error("Failed to execute AddDeviceToOC, err_msg:{}.".format(str(err)))
            return Message(500, HCCIException(653082))
        finally:
            SshTool.close_all_clients()

    def rollback(self, project_id, pod_id):
        return Message(200)

    def retry(self, project_id, pod_id):
        return self.execute(project_id, pod_id)

    def _get_request(self, url, headers):
        logger.info("url: %s", url)
        response = self.rest_api.get(url, headers=headers)
        self.check_response(response)
        return json.loads(response.content) if response.content else None
