# -*- coding:utf-8 -*-
import ipaddress

import utils.common.log as logger
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.DBAdapter.DBConnector import BaseOps
from utils.Driver.CloudDC.OpenStack.compute.aggregate_operate import AggregateOperate
from utils.business.OpenStack.cps_util import HostsInfoByPod
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.client.FSPAuthClient import OpenstackAuthClient
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.param_check import ping

import plugins.eBackup.scripts.common.ebackup_util as ebackup_handle
from plugins.eBackup.scene.deploy.params.params_check import check_openstack_region_sub
from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.ebackup_node import EBackupNode
from plugins.eBackup.common.util import SshTool
from plugins.eBackup.common.model import SshInfo, CmdInfo

logger.init("eBackup CheckeParam")


class CheckeParam(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        self.params = ParamUtil()
        self.project_id = project_id
        self.pod_id = pod_id
        self.fsp_node = ''
        self.__config_dict = self.params.get_service_cloud_param(pod_id,
                                                                 "eBackup")
        self.condition_dic = get_project_conditions(project_id)
        self.project_api = ProjectApi()
        self.region_id = self.__config_dict.get('region_id')
        self.param_db = BaseOps()
        self.ebackup_node = EBackupNode(pod_id, project_id)

        if get_project_condition_boolean(project_id, 'BCInstalled') and \
                get_project_condition_boolean(project_id, 'CSHAStorage'):
            return
        # 获取管理平面信息
        self.get_management_plane_info()
        self.get_produce_storage_plane_info()
        self.get_backup_storage_plane_info()
        self.get_business_storage_plane_info()
        self.get_fsp_info(project_id, pod_id)
        self.get_param_for_expand_az(pod_id)

        self.root_passwd = self.__config_dict.get('eBackup_os_root_password',
                                                  '')
        self.hcp_user_pwd = self.__config_dict.get('hcp_ssh_password', '')

    def get_management_plane_info(self):
        self.datamover_externalom_iplist = self.__config_dict.get('datamover_externalom_iplist', '')
        self.datamover_externalom_netmask = self.__config_dict.get('eBackup_externalom_netmask', '')
        self.datamover_externalom_gateway = self.__config_dict.get('eBackup_externalom_gateway', '')
        self.datamover_internal_float_ip = self.__config_dict.get('datamover_internal_float_ip', '')
        self.datamover_management_float_ip = self.__config_dict.get('datamover_management_float_ip', '')

    def get_produce_storage_plane_info(self):
        # 获取生产存储平面信息
        self.datamover_producestorage_iplist = \
            self.__config_dict.get('datamover_producestorage_iplist',
                                   '').lower()
        self.datamover_producestorage_netmask = \
            self.__config_dict.get('datamover_producestorage_netmask',
                                   '').lower()
        self.datamover_producestorage_gateway = \
            self.__config_dict.get('datamover_producestorage_gateway',
                                   '').lower()

    def get_backup_storage_plane_info(self):
        # 获取备份存储平面信息你
        self.datamover_backupstorage_iplist = \
            self.__config_dict.get('datamover_backupstorage_iplist', '')
        self.datamover_backupstorage_netmask = \
            self.__config_dict.get('datamover_backupstorage_netmask',
                                   '').lower()
        self.datamover_backupstorage_gateway = \
            self.__config_dict.get('datamover_backupstorage_gateway',
                                   '').lower()

    def get_business_storage_plane_info(self):
        # 获取业务存储信息
        self.business_storage_urllist = \
            self.__config_dict.get('eBackup_protected_enviroment_urllist')
        self.business_storage_admin_username = \
            self.__config_dict.get('business_storage_admin_username')
        self.business_storage_admin_password = \
            self.__config_dict.get('business_storage_admin_password')
        self.business_storage_superadmin_username = \
            self.__config_dict.get('business_storage_superadmin_username')
        self.business_storage_superadmin_password = \
            self.__config_dict.get('business_storage_superadmin_password')

    def get_fsp_info(self, project_id, pod_id):
        # 获取fsp信息
        self.fsp_root_pwd = self.params.get_param_value(
            pod_id, "OpenStack", 'root_password', 'openstack_root_password')
        self.fsp_user_pwd = self.params.get_param_value(
            pod_id, "OpenStack", 'fsp_password', 'openstack_fsp_password')
        self.fsp_env_pwd = self.params.get_param_value(
            pod_id, "OpenStack", 'env_os_password', 'openstack_env_os_password')
        self.fsp_node = ''
        if get_project_condition_boolean(project_id, 'ExpansionAdCloudService') \
                or get_project_condition_boolean(project_id, 'BCInstalled'):
            fsp_node_info = HostsInfoByPod(project_id, pod_id)
            self.fsp_node_list = fsp_node_info.get_hosts_info()
            self.fsp_node = self.fsp_node_list[0].get('external_omip', '')
            if self.fsp_node == '':
                logger.error("get fsp node info is empty.")
                raise HCCIException(653045)

    def get_param_for_expand_az(self, pod_id):
        # 扩容AZ场景
        self.existed_datamover_externalom_iplist = self.__config_dict.get(
            'existed_datamover_externalom_iplist', '').lower()
        self.existed_datamover_producestorage_iplist = self.__config_dict.get(
            'existed_datamover_producestorage_iplist', '')
        self.existed_datamover_producestorage_netmask = self.__config_dict.get(
            'existed_datamover_producestorage_netmask', '')
        self.existed_datamover_producestorage_gateway = self.__config_dict.get(
            'existed_datamover_producestorage_gateway', '')

    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """

        try:
            if get_project_condition_boolean(project_id, 'BCInstalled') and \
                    get_project_condition_boolean(project_id, 'CSHAStorage'):
                return Message(200)
        except Exception as err:
            logger.error("Failed get project condition, ErrMsg: %s" % err)
            return Message(500, HCCIException("653067", str(err)))

        try:
            self.exec_check(project_id, pod_id)
        except HCCIException as err:
            logger.error("Failed executing checkParm, ErrMsg: %s" % str(err))
            return Message(500, err)
        except Exception as err:
            logger.error("Failed executing checkParm, ErrMsg: %s" % err)
            return Message(500, HCCIException("653067", str(err)))
        finally:
            SshTool.close_all_clients()

        logger.info('Succeed executing checkParm')
        return Message(200)

    def exec_check(self, project_id, pod_id):
        if not get_project_condition_boolean(project_id, 'ExpansionComputeRes_KVMNode') and not \
                get_project_condition_boolean(project_id, 'TenantStorNewPool'):
            # 扩容计算节点仅安装Driver 不需要校验eBackup相关密码
            self.ebackup_node.check_nodes_password_infos()
        else:
            account_info = {
                'fsp': self.fsp_user_pwd,
                'root': self.fsp_root_pwd
            }
            self.ebackup_node.check_node_password_expired(self.fsp_node, account_info, user='fsp', root='root')
        self.ebackup_node.unified_error_reporting()

        # 扩存储池场景不需要再验证其他内容
        if get_project_condition_boolean(project_id, 'TenantStorNewPool'):
            return
        # 扩容计算节点(仅安装Driver不配置，没有参数，不需要参数校验)/扩容业务存储
        if get_project_condition_boolean(project_id, 'ExpansionServiceStorage'):
            self.expand_servicetorage(project_id, pod_id)

        elif not get_project_condition_boolean(project_id, 'ExpansionComputeRes_KVMNode'):
            # 首region、扩容region、扩容az、扩容eBackup
            if get_project_condition_boolean(project_id, 'eBackup_Proxy') \
                    or get_project_condition_boolean(project_id, 'eBackup_ServerProxy_Driver'):
                check_ip_list = self.check_management_plane_network(project_id)
                self.check_plan_ip_duplicate(check_ip_list)
                self.check_ip_occupied(project_id, check_ip_list)
                self.check_producestorage_plane_network(project_id)
                self.check_backupstorage_ip_duplicate(project_id)

            # 仅这一种场景有
            if get_project_condition_boolean(project_id, 'eBackup_ServerProxy_Driver'):
                self.check_az_and_storage_password(project_id)
            self.check_ebackup_proxy_scene(project_id, pod_id)
            self.check_bcinstalled_scene(project_id, pod_id)

    def check_bcinstalled_scene(self, project_id, pod_id):
        # 扩容AZ场景，复用eBackup
        if not get_project_condition_boolean(project_id, 'BCInstalled'):
            return
        cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        externalom_iplist = self.existed_datamover_externalom_iplist.split(',')
        node_info_list = cmdb_util.get_deploy_node_info_for_mo(self.region_id, "eBackup")
        for params_ip in externalom_iplist:
            is_find = False
            for node_info in node_info_list:
                ip_address_list = node_info.get('ipAddresses')
                if not ip_address_list:
                    continue
                ips = list(map(lambda x: x.get('ip'), ip_address_list))
                if params_ip in ips:
                    is_find = True
                    break
            if not is_find:
                raise HCCIException(653130, params_ip)
        success_ip_list, failed_ip_list = self.check_ip_available(externalom_iplist, self.fsp_node, 'True')
        if failed_ip_list:
            raise HCCIException(653073, self.fsp_node, str(failed_ip_list))
        self.check_producestorage_plane_network_for_expand_az()
        self.check_storage_password()
        self.check_fusionstorage_status(project_id)

    def check_producestorage_plane_network_for_expand_az(self):
        # 检查生产存储平面IP是否与网关共平面，Huawei Distributed Block Storag场景才检测
        if self.existed_datamover_producestorage_netmask != '':
            check_ip_list = \
                self.existed_datamover_producestorage_iplist.split(
                    ',')
            self.check_ip_segment(
                self.existed_datamover_producestorage_gateway,
                self.existed_datamover_producestorage_netmask,
                check_ip_list)

    def check_storage_password(self):
        # 检查存储密码正确性
        if self.business_storage_urllist != '':
            self.check_business_info_correct(
                self.business_storage_urllist,
                self.business_storage_admin_username,
                self.business_storage_admin_password
            )
            self.check_business_info_correct(
                self.business_storage_urllist,
                self.business_storage_superadmin_username,
                self.business_storage_superadmin_password
            )

    def check_ebackup_proxy_scene(self, project_id, pod_id):
        # 扩容proxy独有
        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            check_ip_list = [self.datamover_internal_float_ip]
            datamover_server_ip = self.__config_dict.get('datamover_server_ip').lower()
            check_ip_list.append(datamover_server_ip)
            success_ip_list, failed_ip_list = self.check_ip_available(check_ip_list, '', 'True')
            if failed_ip_list:
                raise HCCIException(653073, "FCD host", str(failed_ip_list))

            ebk_component_ip = datamover_server_ip

            try:
                # 修改为从主节点IP获取，不用判断是否存在浮动IP
                self.get_account_info(project_id, pod_id, ebk_component_ip)
            except Exception as err:
                logger.error("query pwd from manageone failed:%s" % str(err))

            # 检查节点是否存在，节点是否为主节点
            cmd = 'service hcp status | grep AdminNode; echo result:$?'
            try:
                ssh_info = SshInfo(datamover_server_ip, "hcp", self.hcp_user_pwd, self.root_passwd)
                ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=True)
                result = SshTool.ssh_send_comand_return(ssh_client, cmd)
            except Exception as err:
                logger.error("Failed to create ssh error, IP: %s. stdout: %s" % (datamover_server_ip, str(err)))
                result = "result:1"
            if 'result:0' not in str(result):
                raise HCCIException(653074, datamover_server_ip)

    def check_fusionstorage_status(self, project_id):
        if get_project_condition_boolean(project_id,
                                         '(BCInstalled&ExpansionAZ)&(TenantStorFB|TenantStorFBHCI|TenantStorHCI'
                                         '|TenantStorHCIFS800|TenantStorFB80|TenantStorFBHCI80)'):
            # 此处为扩AZ场景并且新建存储集群，校验eBackup是否已经对接存储
            result = ''
            server_iplist = self.__config_dict.get('existed_datamover_externalom_iplist', '').lower().split(',')
            cmd = "ps -ef | grep -v grep | grep -q dsware;echo \"last cmd ret:$?\""
            for ebk_ip in server_iplist:
                self.get_account_info(project_id, self.pod_id, ebk_ip)
                try:
                    ssh_info = SshInfo(ebk_ip, "hcp", self.hcp_user_pwd, self.root_passwd)
                    ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=True)
                    result = SshTool.ssh_send_comand_return(ssh_client, cmd)
                except Exception as err:
                    logger.error("Failed to create ssh error, IP: %s. stdout: %s" % (ebk_ip, str(err)))
                if str(result).find("ret:0") != -1:
                    raise HCCIException(653136, ebk_ip)
            logger.info("check_fusionstorage_status successful")

    def get_account_info(self, project_id, pod_id, ebk_component_ip):
        account_info = API.get_account_info(project_id, pod_id, 'eBackupServer')
        account_dict = account_info.get(ebk_component_ip, dict())
        try:
            if account_dict and account_dict.get('hcp'):
                self.hcp_user_pwd = account_dict.get('hcp')
                self.root_passwd = account_dict.get('root')
        except Exception as err:
            logger.error(f"Occurred an exception when obtaining the dictionary value. error : {err}")
            raise err

        logger.info("query pwd from manageone succ.")

    def check_az_and_storage_password(self, project_id):
        if get_project_condition_boolean(
                project_id, 'ExpansionAdCloudService'):
            # 登录fsp检查az是否存在
            az_list = \
                self.__config_dict[
                    'az_related_to_datamover'].split(',')
            failed_az_list = self.check_az_available(az_list)
            if failed_az_list:
                tmp_str = str(failed_az_list)
                raise HCCIException(653138, tmp_str)
            # 检查存储密码正确性
            if get_project_condition_boolean(
                    project_id, 'TenantStorIPSAN') or \
                    get_project_condition_boolean(
                        project_id, 'TenantStorFCSAN'):
                self.check_business_info_correct(
                    self.business_storage_urllist,
                    self.business_storage_admin_username,
                    self.business_storage_admin_password)
                self.check_business_info_correct(
                    self.business_storage_urllist,
                    self.business_storage_superadmin_username,
                    self.business_storage_superadmin_password)

    def check_backupstorage_ip_duplicate(self, project_id):
        # 检查备份存储平面共网段和ip是否重复
        if self.datamover_backupstorage_iplist != '':
            check_ip_list = \
                self.datamover_backupstorage_iplist.split(',')
            if not get_project_condition_boolean(
                    project_id, 'eBackupHeterogeneous'):
                self.check_ip_segment(
                    self.datamover_backupstorage_gateway,
                    self.datamover_backupstorage_netmask,
                    check_ip_list)
            if len(check_ip_list) != len(set(check_ip_list)):
                raise HCCIException(653071, str(check_ip_list))

    def check_producestorage_plane_network(self, project_id):
        # 检查生产存储平面共网段
        success_ip_list = []
        if self.datamover_producestorage_iplist != '':
            check_ip_list = \
                self.datamover_producestorage_iplist.split(',')
            self.check_ip_segment(
                self.datamover_producestorage_gateway,
                self.datamover_producestorage_netmask,
                check_ip_list)
            if len(check_ip_list) != len(set(check_ip_list)):
                raise HCCIException(653071, str(check_ip_list))

            if get_project_condition_boolean(project_id, 'ExpansionAdCloudService') and not \
                    get_project_condition_boolean(project_id, 'eBackupHeterogeneous'):
                try:
                    ip_list, failed_ip_list = self.check_ip_available(check_ip_list, self.fsp_node, 'False')
                except Exception as err:
                    logger.error(f"check producestorage ip failed. output:{err}")
                    raise err
                success_ip_list = ip_list
        if success_ip_list:
            raise HCCIException(653072, str(success_ip_list))

    def check_ip_occupied(self, project_id, check_ip_list):
        # 检查管理平面规划ip是否被占用
        if self.datamover_backupstorage_iplist:
            check_ip_list.extend(self.datamover_backupstorage_iplist.split(','))
        if self.datamover_producestorage_iplist:
            check_ip_list.extend(self.datamover_producestorage_iplist.split(','))
        if not get_project_condition_boolean(
                project_id, 'eBackupHeterogeneous'):
            success_ip_list, failed_ip_list = \
                self.check_ip_available(check_ip_list, '', 'True')
            if success_ip_list:
                raise HCCIException(653072, str(success_ip_list))

    @staticmethod
    def check_plan_ip_duplicate(check_ip_list):
        # 检查规划IP是否重复
        if len(check_ip_list) != len(set(check_ip_list)):
            raise HCCIException(653071, str(check_ip_list))

    def check_management_plane_network(self, project_id):
        # 检查管理平面共网段
        check_ip_list = self.datamover_externalom_iplist.split(',')
        if not get_project_condition_boolean(
                project_id, 'eBackup_Proxy'):
            check_ip_list.append(self.datamover_internal_float_ip)
            if self.datamover_management_float_ip != '':
                check_ip_list.append(self.datamover_management_float_ip)
        if not get_project_condition_boolean(
                project_id, 'eBackupHeterogeneous'):
            self.check_ip_segment(
                self.datamover_externalom_gateway,
                self.datamover_externalom_netmask,
                check_ip_list)
        return check_ip_list

    def expand_servicetorage(self, project_id, pod_id):
        availability_zone = self.param_db.get_value_from_cloudparam(
            pod_id, "OpenStack", "openstack_expansion_az")
        compute_client = OpenstackAuthClient.get_compute_client(
            self.param_db, project_id, pod_id)

        fsp_host_id = ''
        agg = AggregateOperate(compute_client)
        aggr_details = agg.get_group_details()
        aggr_details = aggr_details.get('aggregates')
        for sub_aggr_details in aggr_details:
            aggr_name = sub_aggr_details.get('availability_zone', '')
            if aggr_name == availability_zone:
                aggr_hosts = sub_aggr_details.get('hosts', '')
                fsp_host_id = aggr_hosts[0]
                break
        logger.info("get host id is %s." % fsp_host_id)
        for fsp_node_info in self.fsp_node_list:
            host_id = fsp_node_info.get('hostid', '')
            if host_id == fsp_host_id:
                self.fsp_node = fsp_node_info.get('external_omip', '')

        logger.info("get fsp host ip is %s." % self.fsp_node)
        check_ip_list = \
            self.existed_datamover_externalom_iplist.split(',')
        success_ip_list, failed_ip_list = \
            self.check_ip_available(
                check_ip_list, self.fsp_node, 'True')
        if failed_ip_list:
            raise HCCIException(653073, self.fsp_node, str(failed_ip_list))
        # 检查生产存储平面IP是否与网关共平面，Huawei Distributed Block Storag场景才检测
        if self.existed_datamover_producestorage_netmask != '':
            check_ip_list = \
                self.existed_datamover_producestorage_iplist.split(',')
            self.check_ip_segment(
                self.existed_datamover_producestorage_gateway,
                self.existed_datamover_producestorage_netmask,
                check_ip_list)

    def check_ip_segment(self, ip_gateway, ip_netmask, check_ip_list):
        if 'ManageIPV6' in self.condition_dic:
            ip_mask_num = ip_netmask
        else:
            ip_netmask_parses = [bin(int(x)).count("1") for x in ip_netmask.split(".")]
            ip_mask_num = str(sum(ip_netmask_parses))

        ip_and_mask = ip_gateway + '/' + ip_mask_num

        for check_ip in check_ip_list:
            result = ipaddress.ip_address(check_ip) in ipaddress.ip_network(
                ip_and_mask, strict=False)
            logger.info("check ip:%s in the same segmeng as gateway, "
                        "result is %s." % (check_ip, result))
            if not result:
                logger.error(
                    "check ip:%s is not in the %s network"
                    " segment." % (check_ip, ip_and_mask))
                raise HCCIException(653070, check_ip + ',' + ip_gateway)
        return

    # 当local为True时，在本节点ping，如果local不为True，则到指定fsp节点ping。
    def check_ip_available(self, check_ip_list, node_ip, local):
        success_ip_list = []
        failed_ip_list = []
        if local == 'True':
            for check_ip in check_ip_list:
                logger.info("begin to ping:" + check_ip)
                if ping(check_ip):
                    success_ip_list.append(check_ip)
                else:
                    failed_ip_list.append(check_ip)
        else:
            for check_ip in check_ip_list:
                logger.info("begin to ping:" + check_ip)
                if not ebackup_handle.check_ip_valid(check_ip, self.project_id):
                    raise HCCIException(653103, check_ip)
                if 'ManageIPV6' in self.condition_dic:
                    cmd = 'ping6 -c 3 ' + check_ip + '; echo result:$?'
                else:
                    cmd = 'ping -c 3 ' + check_ip + '; echo result:$?'

                try:
                    result = self.exec_cmd_in_node(node_ip, cmd)
                except Exception as err:
                    message = "Fsp node(%s) is unreachable. Please check the network connection or fsp password." \
                              " stdout: %s" % (node_ip, str(err))
                    logger.error(message)
                    raise HCCIException(653092, node_ip, "fsp") from err

                if 'result:0\n' in result:
                    success_ip_list.append(check_ip)
                else:
                    failed_ip_list.append(check_ip)
        return success_ip_list, failed_ip_list

    def exec_cmd_in_node(self, node_ip, cmd):
        ssh_info = SshInfo(node_ip, "fsp", self.fsp_user_pwd)
        ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=False)
        result = SshTool.ssh_send_comand_return(ssh_client, cmd)
        return result

    def check_az_available(self, check_az_list):
        failed_az_list = []
        for az_name in check_az_list:
            logger.info("begin to check:" + az_name)
            if not check_openstack_region_sub(az_name):
                raise HCCIException(653103, 'az_related_to_datamover')
            cmd = 'nova availability-zone-list | grep ' + az_name + \
                  ';echo result:$?'
            ssh_info = SshInfo(self.fsp_node, "fsp", self.fsp_user_pwd, self.fsp_root_pwd)
            ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=True)
            cmd_info = CmdInfo(ssh_client, cmd_list=["TMOUT=0", "source set_env", '1', self.fsp_env_pwd],
                               expect_list=['#', "1|2", 'OS_PASSWORD', '#'], is_expect=True)
            SshTool.send_ssh_command(cmd_info, 100)
            result = SshTool.ssh_send_comand_return(ssh_client, cmd, ebk_timeout=30)
            if "result:0" not in str(result):
                failed_az_list.append(az_name)
        return failed_az_list

    def check_business_info_correct(self, check_storage_urllist, user_name, user_passwd):
        storage_ip_list = []
        storage_url_list = check_storage_urllist.split(',')
        for storage_url in storage_url_list:
            if 'ManageIPV6' in self.condition_dic:
                start_index = storage_url.find('[')
                end_index = storage_url.find(']')
                storage_ip_list.append(storage_url[start_index + 1:end_index])
            else:
                temp_url_list = storage_url.split(':')
                storage_ip_list.append(temp_url_list[0])

        failed_ip_list = []
        for check_ip in storage_ip_list:
            logger.info("begin to check business info:" + check_ip)
            ssh_client = dict()
            try:
                ssh_info = SshInfo(check_ip, user_name, user_passwd)
                ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=False)
            except Exception as err:
                logger.error("Login to %s with %s account failed. stdout: %s." % (check_ip, user_name, str(err)))
                failed_ip_list.append(check_ip)

            try:
                if ssh_client:
                    SshTool.close_ssh_clinet(ssh_client)
            except Exception as err:
                logger.error("Close client %s with %s account failed. stdout: %s." % (check_ip, user_name, str(err)))
                if check_ip not in failed_ip_list:
                    failed_ip_list.append(check_ip)

        if failed_ip_list:
            logger.error("Login to %s with %s account failed." % (str(failed_ip_list), user_name))
            raise HCCIException(653075, failed_ip_list)

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，
        工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return self.execute(project_id, pod_id)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

# the code has been updated to python3.7
