# -*- coding:utf-8 -*-
from jinja2 import Environment, FileSystemLoader

from utils.PasswordManager.PasswordManager import FSPasswordManager
from utils.business.OpenStack.cps_util import HostsInfoByPod
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.util import Utils, SshTool
from plugins.eBackup.common.ebackup_util import DMKDriverUtil
from plugins.eBackup.common.param_tool import ParamTool
from plugins.eBackup.common.model import DmkUserInfo, SshInfo


class DeployeBackupDriverWithDMK(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup")
        self.pod_id = pod_id
        self.project_id = project_id
        self.params = ParamUtil()
        self.__db_param_dict = None
        self.fsp_node_list = None
        self.dmk_config_content = {}
        self.manage_float_ip = ""
        self.externalom_ip_list = ""
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.is_expand_storage_scense = get_project_condition_boolean(self.project_id, 'ExpansionServiceStorage')
        self.is_expand_kvm_node_scense = get_project_condition_boolean(self.project_id, 'ExpansionComputeRes_KVMNode')
        self.is_expand_az_scense = get_project_condition_boolean(project_id, 'ExpansionAZ&BCInstalled')
        param_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.region_id = param_dict.get('region_id')
        self.dmk_util = DMKDriverUtil(self.project_id, self.pod_id, self.region_id)

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def main_handle(self, project_id, pod_id):
        self.__db_param_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            logger.info("no need to deploy driver")
        else:
            self.exec_dmk_operate()
        return Message(200)

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            return self.main_handle(project_id, pod_id)
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            return Message(500, err, err)

    def init_driver_config_file(self):
        logger.info("begin to init dmk config file for driver.")
        logger.info("get param info succ from db.")
        # 装备config块
        config_val, host_val = self.assembly_dmk_conf()
        self.dmk_util.set_dmk_parm(config_val, host_val)
        return

    def assembly_dmk_conf(self):
        logger.info("begin to get fsp node info.")
        fsp_node_info = HostsInfoByPod(self.project_id, self.pod_id)
        self.fsp_node_list = fsp_node_info.get_ebk_driver_hosts_info()
        if len(self.fsp_node_list) == 0:
            logger.error("fsp node info is empty.")
            raise HCCIException(653045)
        logger.info("get fsp node info succ.")
        az_str = self.get_az()
        tmpip_list = self.externalom_ip_list.split(',')
        if len(tmpip_list) == 1:
            datamover_portal_ip = tmpip_list[0]
        else:
            datamover_portal_ip = self.manage_float_ip

        nfs_datalayout = ''
        s3_datalayout = ''
        ebk_configmode = '0'
        nbi_user_passwd = ""
        ebk_installmode = '0'
        fsp_env_pwd = FSPasswordManager(self.pod_id).get_cps_admin_password()
        if self.is_expand_az_scense or self.param_tool.is_expand_ebackup_server():
            if get_project_condition_boolean(self.project_id, 'eBackup_BSTU_NFS'):
                nfs_datalayout_default = self.get_data_layout(az_str, fsp_env_pwd, 'nfs')
                nfs_datalayout = nfs_datalayout_default if nfs_datalayout_default else "2"
            if get_project_condition_boolean(self.project_id, 'eBackup_BSTU_S3'):
                s3_datalayout_default = self.get_data_layout(az_str, fsp_env_pwd, 'uds')
                s3_datalayout = s3_datalayout_default if s3_datalayout_default else "2"
            nbi_user_passwd = self.__db_param_dict.get('eBackup_azconfig_apipwd')
        elif not self.is_expand_storage_scense and not self.is_expand_kvm_node_scense:
            # 默认配置数据布局为2，需要特殊不许安装后修改
            if get_project_condition_boolean(self.project_id, 'eBackup_BSTU_S3'):
                s3_datalayout = '2'
            if get_project_condition_boolean(self.project_id, 'eBackup_BSTU_NFS'):
                nfs_datalayout = '2'
            nbi_user_passwd = self.__db_param_dict.get('eBackup_azconfig_apipwd')
        if self.is_expand_storage_scense:
            ebk_installmode = '2' # 仅安装以及同步配置
        elif self.is_expand_kvm_node_scense:
            ebk_installmode = '1' # 仅在新节点安装Driver

        config_val = {
            "ebk_installmode": ebk_installmode,
            "ebk_configMode": ebk_configmode,
            "ebk_IfConfigDriver": "1",
            "ebk_Ipaddr": datamover_portal_ip,
            "ebk_Azs": az_str,
            "ebk_S3DataLayout": s3_datalayout,
            "ebk_NfsDataLayout": nfs_datalayout,
            "ebk_PasswordUsr": nbi_user_passwd,
            "ebk_PasswordEnv": fsp_env_pwd,
            "ebk_CertFilePath": "None"
        }
        host_val = {
            "eBackup_host": self.fsp_node_list
        }
        return config_val, host_val

    def get_openstack_params(self, params):

        if get_project_condition_boolean(self.project_id, "ProjectDeploy"):
            params = params.replace('openstack_', '')
            result = self.params.get_param_value(
                self.pod_id, 'Openstack', params)
        else:
            result = self.params.get_param_value(self.pod_id, '', params)
        return result

    @staticmethod
    def ssh_connect(host_ip, passwd, cmd, user='fsp'):
        ssh_info = SshInfo(host_ip, user, passwd)
        ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=False)
        return SshTool.ssh_send_comand_return(ssh_client, cmd)

    def get_data_layout(self, az_name, fsp_env_pwd, layout_type):
        logger.info(f"try to get data layout for {az_name}")
        if layout_type not in ("nfs", "uds"):
            logger.warn(f"layout type is invalid.")
            return ""
        fsp_ip = self.get_openstack_params('openstack_reverse_proxy_ip')
        fsp_pwd = self.params.get_param_value(self.pod_id, "OpenStack", 'fsp_password', 'openstack_fsp_password')
        fsp_root_pwd = self.params.get_param_value(self.pod_id, "OpenStack", 'root_password', 'openstack_root_password')
        fsp_ssh_info = SshInfo(fsp_ip, 'fsp', fsp_pwd, fsp_root_pwd)
        ssh_client = SshTool.get_ssh_client(fsp_ssh_info)
        Utils.login_openstack(fsp_env_pwd, ssh_client)
        az_list = az_name.split('|')
        cmd = "temp_list=(`cps template-list | grep cinder-backup- | awk '{print $4}'`);" \
              "for temp_name in \"${temp_list[@]}\"; do output=`cps template-ext-params-show " \
              "--service cinder ${temp_name}`;echo \"${output}\"  | grep -q %s && echo \"${output}\" " \
              "| grep --color=never %s_data_layout ;done"
        cmd_list = [cmd % (az_id, layout_type) for az_id in az_list]
        cmd_list.append('echo "result: $?"')
        result = SshTool.ssh_send_comand_return_list(ssh_client, " && ".join(cmd_list))
        if str(result).find("result: 0") == -1:
            logger.error(f"exec get data layout command failed. stdout: {str(result)}")
            return ""

        for line in result:
            if "_data_layout" not in line:
                continue
            stdout_list = line.split("|")
            if len(stdout_list) != 4:
                logger.warn(f"get data layout result from output failed. output: {line}.")
                continue
            return stdout_list[2].strip()
        return ""

    def get_az(self):
        condition_dic = get_project_conditions(self.project_id)
        is_bc_install = condition_dic.get('BCInstalled', '')

        if self.is_expand_storage_scense or self.is_expand_kvm_node_scense:
            return ""
        elif is_bc_install:
            # 扩az场景
            self.manage_float_ip = self.__db_param_dict.get('existed_datamover_mg_float_ip')
            self.externalom_ip_list = self.__db_param_dict.get('existed_datamover_externalom_iplist').lower()
            az_str = self.params.get_param_value(self.pod_id, "OpenStack", "openstack_expansion_az", '')
            az_str = az_str.replace(' ', '')
            az_str = az_str.replace(',', '|')
        else:
            # 安装&扩备份场景
            self.externalom_ip_list = self.__db_param_dict.get('datamover_externalom_iplist').lower()
            self.manage_float_ip = self.__db_param_dict.get('datamover_management_float_ip').lower()
            az_str = self.__db_param_dict.get('az_related_to_datamover').replace(' ', '')
            az_str = az_str.replace(',', '|')
        return az_str

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def get_fsp_account_info(self):
        self.__db_param_dict['fsp_root_pwd'] = self.params.get_param_value(
            self.pod_id, "OpenStack", 'root_password',
            'openstack_root_password')
        self.__db_param_dict['fsp_user_pwd'] = self.params.get_param_value(
            self.pod_id, "OpenStack", 'fsp_password', 'openstack_fsp_password')
        self.__db_param_dict['fsp_env_pwd'] = FSPasswordManager(self.pod_id). \
            get_cps_admin_password()

    def get_accounts(self):
        self.get_fsp_account_info()
        self.__db_param_dict['dmk_floatIp'] = self.params.get_param_value(
            self.pod_id, "DMK", 'dmk_floatIp', 'dmk_floatIp')
        API.login_dmk(
            self.__db_param_dict.get('dmk_floatIp'),
            self.__db_param_dict.get('eBackup_dmk_user'),
            self.__db_param_dict.get('eBackup_dmk_password'))
        team_id = API.get_dmk_team_id("eBackup")
        team_list = ["eBackup", "OTD"]
        try:
            account1 = API.get_dmk_account_id(
                self.__db_param_dict.get('eBackup_dmk_user'), "fsp")
        except Exception as err:
            logger.info("query account faild, the reason %s" % err)
            account1 = API.addaccounttodmk(
                "fsp", team_id, self.__db_param_dict.get('fsp_user_pwd'),
                self.__db_param_dict.get('fsp_root_pwd'))
            dmkinfo = DmkUserInfo(
                self.__db_param_dict.get('eBackup_dmk_user'), "fsp", team_list,
                [self.__db_param_dict.get('fsp_user_pwd'), self.__db_param_dict.get('fsp_root_pwd')]
            )
            bret = API.update_account_to_team(dmkinfo)
            if not bret:
                logger.error("eBackup,update fsp account to multi team failed.")
                raise HCCIException(653113, "fsp") from e
            return account1
        try:
            API.validate_account("fsp", account1, self.fsp_node_list[0], "su")
            return account1
        except HCCIException as msg:
            logger.error('method "validate_account" raise an error, the message is "%s".' % msg)
            msg_str = str(msg)
            checked_str = "ssh to " + self.fsp_node_list[0] + " by fsp failed"
            if checked_str in msg_str:
                API.delete_dmk_account(account1)
            else:
                raise msg
            team_list = ["eBackup", "OTD"]
            account1 = API.addaccounttodmk(
                "fsp", team_id, self.__db_param_dict.get('fsp_user_pwd'),
                self.__db_param_dict.get('fsp_root_pwd'))
            dmkinfo = DmkUserInfo(
                self.__db_param_dict.get('eBackup_dmk_user'), "fsp", team_list,
                [self.__db_param_dict.get('fsp_user_pwd'), self.__db_param_dict.get('fsp_root_pwd')]
            )
            bret = API.update_account_to_team(dmkinfo)
            if not bret:
                logger.error("update fsp account to multi team failed.")
                raise HCCIException(653113, "fsp") from msg
            return account1

    def exec_dmk_operate(self):
        self.init_driver_config_file()
        logger.info("begin to deploy eBackup driver.")
        is_ok = self.dmk_util.execute_action("install")
        if is_ok:
            logger.info("Install and configure eBackup driver success.")
        else:
            logger.error("Install and configure eBackup dirver failed.")
            raise HCCIException(653118, 'driver')

# the code has been updated to python3.7
