# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
from utils.Logic.dmk.dmk_data import Host
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from plugins.eBackup.common.api_adapter import API

logger.init("eBackup install MaglevAgent")


class InstallMaglevAgent(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.params = ParamUtil()
        self.__config_dict = \
            self.params.get_service_cloud_param(pod_id, "eBackup")
        self.root_passwd = self.__config_dict.get('eBackup_os_root_password')
        self.hcp_user_pwd = self.__config_dict.get('hcp_ssh_password')

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """

    def main_handle(self, project_id):
        logger.info("Processes started")
        ip_list, host_list = [], []
        externalom_ip_lower = self.__config_dict['datamover_externalom_iplist'].lower()
        if get_project_condition_boolean(project_id, 'eBackup_ServerProxy_Driver') \
                or get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            ip_list = externalom_ip_lower.split(',')
        logger.info(f"Need to install MaglevAgent on these nodes: {ip_list}")
        for index, ip in enumerate(ip_list):
            host_list.append(Host(ip=ip, name=f'eBackup_node-{index}', ssh_user='hcp',
                                  ssh_password=self.hcp_user_pwd, root_password=self.root_passwd))
        dmk_ip = self.params.get_param_value(self.pod_id, 'DMK', 'dmk_floatIp', 'dmk_floatIp')
        dmk_user_name = self.params.get_value_from_cloudparam(self.pod_id, "eBackup", "eBackup_dmk_user")
        dmk_pwd = self.params.get_value_from_cloudparam(self.pod_id, "eBackup", "eBackup_dmk_password")
        API.login_dmk(dmk_ip, dmk_user_name, dmk_pwd)
        logger.info("Start to install MaglevAgent")
        try:
            API.install_agent(host_list)
        except Exception as ex:
            logger.error(f"Install MaglevAgent failed! msg: {ex}")
            return Message(500, ex)
        return Message(200)

    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            return self.main_handle(project_id)
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            logger.error("Failed to install Ops Agent" + str(err))
            return Message(500, err, err)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        pass

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(self.project_id, self.pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
