# -*- coding:utf-8 -*-
import utils.common.log as logger
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.exception import HCCIException

from plugins.eBackup.common.util import Utils, SshTool
from plugins.eBackup.common.model import SshInfo, CmdInfo, SshCmdInfo
from plugins.eBackup.common.ebackup_util import EBackupUtil
from plugins.eBackup.scripts.upgrade.util import CommonConfig
from plugins.eBackup.scripts.upgrade.ebackup_upgrade import EbackupUpgrader
from plugins.eBackup.scripts.upgrade.config_after_upgrade import UpdateCMDB


class DatamoverRollback(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DatamoverRollback, self).__init__(project_id, pod_id,
                                                regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.__db_param_dict = Utils.init_system_params(project_id,
                                                        regionid_list[0])
        self.hcp_password = self.__db_param_dict["eBackup_hcp_pwd"]
        self.root_password = self.__db_param_dict["eBackup_root_pwd"]
        self.param_dict = Utils.init_system_params(project_id, regionid_list[0])
        self.az_list_str = UpdateCMDB(project_id, pod_id, regionid_list).get_az_info()

    def pre_check(self, project_id, pod_id, regionid_list=None):
        pass

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_execute():
            host_info = self.__db_param_dict["eBackup_Datamover_nodes"]
            host_info = host_info.replace(" ", "").split("|")
            all_result = []
            logger.info("Begin to rollback all datamover nodes(%s)" % str(host_info))
            for host_ips in host_info:
                host_ips = host_ips.split(";")
                self._verify_tmp_upgrade(host_ips)
                common_config = CommonConfig(self.__db_param_dict, pod_id=self.pod_id, region_id=self.regionid_list[0],
                                             project_id=self.project_id)
                common_config.handle_icagent_process(host_ips, 'disable')
                logger.info("Begin to rollback datamover nodes(%s)" % str(host_ips))
                EBackupUtil(self.project_id, self.pod_id).set_cloud_service_info_of_ebackup(
                    self.param_dict, self.az_list_str, host_ips, is_upgrade=False)
                ebckup_upgrade = EbackupUpgrader(host_ips, self.__db_param_dict, "datamover")
                is_true = ebckup_upgrade.rollback()
                if not is_true:
                    logger.error("Rollback datamover nodes(%s) failed." % str(host_ips))
                else:
                    logger.info("Rollback datamover nodes(%s) successfully." % str(host_ips))
                all_result.append(is_true)
                self._clean_after_rollback(host_ips)

            if False in all_result:
                logger.error("Rollback all datamover nodes failed.")
                return Message(500, error_msg_cn="回退Datamover失败，请登录DMK查看失败原因，请勿直接重试",
                               error_msg_en="Rollback Datamover failed, please log in DMK to see the details, "
                                            "do not directly retry")

            logger.info("Rollback all datamover nodes successfully.")
            return Message(200)

        try:
            return do_execute()
        except Exception as err:
            logger.error("Failed to rollback datamover ,the reason is %s" % err)
            return Message(500, str(err), str(err))

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(self.project_id, self.pod_id, self.regionid_list)

    def _find_folder_exists(self, ssh_client, folder_path):
        check_cmd = f"ll {folder_path}; echo check file exists:$?"
        check_result = SshTool.ssh_send_comand_return(ssh_client, check_cmd)
        return str(check_result).find('check file exists:0') != -1

    def _verify_tmp_upgrade(self, server_ips):
        logger.info("Start to deal with the /tmp folder")
        check_cmd = "df -h /tmp"
        for ebk_ip in server_ips:
            ssh_info = SshInfo(ebk_ip, "hcp", self.hcp_password, sudo_passwd=self.root_password)
            ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=True)
            result = SshTool.ssh_send_comand_return(ssh_client, check_cmd)
            if str(result).find('tmpfs') != -1:
                if self._find_folder_exists(ssh_client, "/tmp/upgrade"):
                    logger.info(f"The /upgrade folder on node({ebk_ip}) exists, continue rollback process")
                    continue
                elif self._find_folder_exists(ssh_client, "/opt/upgrade_bak/"):
                    logger.info(f"The /upgrade folder on node({ebk_ip}) does not exist, copy from backup files")
                    cmd_info = CmdInfo(ssh_client, ["cp -r /opt/upgrade_bak /tmp/upgrade"], ["#"], is_expect=True)
                    SshTool.send_ssh_command(cmd_info, 100)
                else:
                    logger.error("The upgrade files are missing, cannot execute rollback")
                    raise HCCIException(650093, ebk_ip)
            if ssh_client:
                SshTool.close_ssh_clinet(ssh_client)

    def _clean_after_rollback(self, server_ips):
        logger.info("Start to clean the system after rollback")
        clean_useless_path_cmd = 'rm -rf /tmp/upgrade/upgrade_pkg ' \
                                 '/opt/link_var_ebackup_bak ' \
                                 '/var/ebackup_bak ' \
                                 '/home/hcp/ebackup ' \
                                 '/home/hcp/precheck_ebackup.sh'
        for ebk_ip in server_ips:
            ssh_cmd_info = SshCmdInfo(ebk_ip, clean_useless_path_cmd, 'hcp', self.hcp_password,
                                      self.root_password, "", "")
            SshTool.ssh_cmds(ssh_cmd_info)
            logger.info(f"cleaned useless files on node {ebk_ip}")

