# -*- coding:utf-8 -*-
import os

import utils.common.log as logger

from plugins.eBackup.common.util import SshTool
from plugins.eBackup.common.model import SshInfo
from plugins.eBackup.common.ebackup_util import DMKDriverUtil
from plugins.eBackup.scripts.upgrade.util import DriverConfig
from plugins.eBackup.scripts.upgrade.util import get_ebackup_driver_nodes_from_file


class EbackupDriverUpgrader(object):
    def __init__(self, reverse_proxy_ip, param_dict, project_id, pod_id, region_id):
        self.reverse_proxy_ip = reverse_proxy_ip
        self.param_dict = param_dict
        self.dmk_task = DMKDriverUtil(project_id, pod_id, region_id=region_id)
        self.hosts_dict = self._get_host_dict(reverse_proxy_ip)

    def upgrade(self):
        if 0 == len(self.hosts_dict):
            logger.info("driver node is empty, reverse proxy ip:%s" % self.
                        reverse_proxy_ip)
            return True
        is_true = self.is_all_upgraded()
        if is_true:
            logger.info("All nodes are upgraded.")
            return True
        return self.deploy_with_dmk("upgrade")

    def rollback(self):
        if 0 == len(self.hosts_dict):
            logger.info("driver node is empty, reverse proxy ip:%s" % self.
                        reverse_proxy_ip)
            return True
        is_true = self.is_all_rollbacked()
        if is_true:
            logger.info("All nodes are rollbacked.")
            return True
        return self.deploy_with_dmk("rollback")

    def _get_host_dict(self, reverse_proxy_ip):
        region_id = self.param_dict["current_region_id"]
        file_path = os.path.realpath(
            __file__ + '/../' + region_id + '_' + reverse_proxy_ip + '.txt')
        driver_host_dict = get_ebackup_driver_nodes_from_file(file_path)
        return driver_host_dict

    def is_all_upgraded(self):
        logger.info("Begin to check whether all node are upgraded.")
        for driver_ip in self.hosts_dict.get('eBackupDriver'):
            if not self.is_upgraded(driver_ip):
                logger.info("There some nodes are not upgraded.")
                return False
        logger.info("All nodes are upgraded.")
        return True

    def is_all_rollbacked(self):
        logger.info("Begin to check whether all node are rollbacked.")
        for driver_ip in self.hosts_dict.get('eBackupDriver'):
            if self.is_upgraded(driver_ip):
                logger.info("There some nodes are not rollbacked.")
                return False
        logger.info("All nodes are rollbacked.")
        return True

    def is_upgraded(self, driver_ip):
        ssh_client = None
        logger.info("Begin to check current version at host %s" % driver_ip)

        def do_is_upgraded():
            nonlocal ssh_client
            ssh_info = SshInfo(driver_ip, "fsp", self.param_dict.get('openstack_fsp_pwd'),
                               self.param_dict.get('openstack_root_pwd'))
            ssh_client = SshTool.get_ssh_client(ssh_info, need_root_login=True)
            version_check_cmds = self.check_python_version()

            result = SshTool.ssh_send_comand_return_list(ssh_client, version_check_cmds)
            current_version = "No patch" if len(result) == 0 else result[0].strip().replace('\n', '')
            logger.info("The current version is %s at host %s" % (current_version, driver_ip))
            return current_version == self.param_dict.get('eBackup_Version')

        try:
            res = do_is_upgraded()
        except Exception as err:
            logger.error("Exception occurs when check current version,the Exception is:" + str(err))
            raise err
        finally:
            if ssh_client:
                SshTool.close_ssh_clinet(ssh_client)
        return res

    def check_python_version(self):

        # 底座未升级 单升eBackup时 路径为python3.7 与底座一起升级时路径为python
        version_check_cmd = '''version_info=$(''' \
                            '''cat /usr/lib/python/site-packages/cinder/backup/drivers/ebackupversion.conf ''' \
                            '''2>/dev/null || ''' \
                            '''cat /usr/lib/python3.7/site-packages/cinder/backup/drivers/ebackupversion.conf ''' \
                            '''2>/dev/null);echo ${version_info} | awk -F "[=| ]" '{print $2}' '''

        return version_check_cmd

    def deploy_with_dmk(self, action):
        ebackup_driver_config, ebackup_driver_host = self.get_all_params()

        # install eBackup
        ebackup_action = ""
        if action == "upgrade":
            ebackup_action = "upgrade" if self.param_dict.get('update_scene') == 'upgrade' else 'patch'
        elif action == "rollback":
            ebackup_action = "rollback" if self.param_dict.get('update_scene') == 'upgrade' else 'patch_rollback'

        self.dmk_task.set_dmk_parm(ebackup_driver_config, ebackup_driver_host)
        result = self.dmk_task.execute_action(ebackup_action)

        if not result:
            logger.error("Do DMK task %s failed." % ebackup_action)
        return result

    def get_all_params(self):
        config = DriverConfig(hosts_dict=self.hosts_dict,
                              param_dict=self.param_dict, one2all=0)
        return config.get_config()
