# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
from utils.DBAdapter.DBConnector import BaseOps
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.util import Utils, SshTool
from plugins.eBackup.common.model import SshInfo, CmdInfo
from plugins.eBackup.scripts.upgrade.ebackup_driver_upgrade import EbackupDriverUpgrader
from plugins.eBackup.scripts.upgrade.util import update_driver_cert_path, get_ebackup_driver_nodes_from_file


class DriverUpgrade(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DriverUpgrade, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region_id = regionid_list[0]
        self.database = BaseOps()
        self._db_param_dict = Utils.init_system_params(project_id, regionid_list[0])
        self.reverse_proxy_ip = self._db_param_dict['openstack_reverse_proxy_ip']
        self.fsp_user = "fsp"
        self.fsp_pwd = self._db_param_dict['openstack_fsp_pwd']
        self.fsp_root_pwd = self._db_param_dict['openstack_root_pwd']
        self.cps_pwd = self._db_param_dict['openstack_cps_admin_pwd']
        self.driver_ssh_info = SshInfo(self.reverse_proxy_ip, self.fsp_user, self.fsp_pwd, self.fsp_root_pwd)
        self.ssh_client = SshTool.get_ssh_client(self.driver_ssh_info)

    def _execute_upgrade(self):
        self.login_openstack(self.ssh_client)
        driver_upgrade = EbackupDriverUpgrader(self.reverse_proxy_ip, self._db_param_dict, self.project_id,
                                               self.pod_id, self.region_id)
        if self.get_cps_upg_state() == "upg":
            self.set_cps_upg_state("all_open")
            # 将状态保存到文件 防止失败重试时未将状态置回 OpenStack在网络服务升级大工步结束后才将升级状态置为all_open
            Utils.set_value_to_config_file("need_rollback_state", "True")
        upgrade_result = driver_upgrade.upgrade()
        region_id = self._db_param_dict["current_region_id"]
        file_path = os.path.realpath(__file__ + '/../' + region_id + '_' + self.reverse_proxy_ip + '.txt')
        driver_host_dict = get_ebackup_driver_nodes_from_file(file_path)
        if len(driver_host_dict.get('eBackupDriver', [])) >= 1:
            driver_ip = driver_host_dict.get('eBackupDriver')[0]
            update_driver_cert_path(driver_ip, self._db_param_dict)
        if Utils.get_value_from_config_file("need_rollback_state") == "True":
            self.set_cps_upg_state("upg")
            Utils.set_value_to_config_file("need_rollback_state", "False")
        self.clean_up_env()
        return upgrade_result

    def clean_up_env(self):
        cmd = "sed -i '/CPS_PASSWORD/d' ~/.bash_history;sed -i '/CPS_PASSWORD/d' /var/log/messages"
        cmd_info = CmdInfo(self.ssh_client, cmd_list=[cmd], is_expect=False)
        SshTool.send_ssh_command(cmd_info, 100)
        logger.info("OpenStack SSH client is closed")

    def login_openstack(self, ssh_client):
        cmd_list = ["TMOUT=0", "source set_env", "2", "cps_admin", self.cps_pwd]
        expect_list = ["#", "1|2", "CPS_USERNAME", "CPS_PASSWORD", "#"]
        cmd_info = CmdInfo(ssh_client, cmd_list=cmd_list, expect_list=expect_list, is_expect=True)
        resp = SshTool.send_ssh_command(cmd_info, 300)
        if -1 != str(resp).find("Authenticate Failed"):
            raise HCCIException(650051)

    def set_cps_upg_state(self, state):
        cmd = f"upgrade cps reststate-update {state}"
        cmd_info = CmdInfo(self.ssh_client, cmd_list=[cmd], is_expect=False)
        result = SshTool.send_ssh_command(cmd_info, 100)
        if str(result).find("last cmd result: 0") == -1:
            raise HCCIException(650063, state, str(result), self.reverse_proxy_ip, state)

    def get_cps_upg_state(self):
        cmd = f"upgrade cps reststate-show"
        cmd_info = CmdInfo(self.ssh_client, cmd_list=[cmd], is_expect=False)
        result = SshTool.send_ssh_command(cmd_info, 100)
        if str(result).find("upg") != -1:
            return "upg"
        elif str(result).find("all_open") != -1:
            return "all_open"
        else:
            raise HCCIException(650064, str(result))

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            is_true = self._execute_upgrade()
        except HCCIException as ex:
            logger.error("Exception occurs when upgrade eBackup driver: " + str(ex))
            return Message(500, ex)
        except Exception as err:
            logger.error("Upgrade eBackup driver failed, the reason is: " + str(err))
            return Message(500, error_msg_cn="升级eBackup Driver失败，请查看日志详情, 异常信息：%s" % str(err),
                           error_msg_en="Upgrade eBackup driver failed,please see the log for detail, error info：%s"
                                        % str(err))
        finally:
            SshTool.close_all_clients()
            logger.info("OpenStack SSH client is closed")
        if not is_true:
            logger.info("Fail to Upgrade eBackup driver.")
            return Message(500, "Fail to Upgrade eBackup driver.")
        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)
