# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import os
import re
import time

import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.software_package_util import find_software_package_by_name
from utils.common.check_result import CheckResult

from plugins.eBackup.common.util import Utils, SshTool
from plugins.eBackup.common.model import SshInfo, CmdInfo
from plugins.eBackup.scripts.upgrade.util import g_check_item
from plugins.eBackup.scripts.common.ebackup_util import check_compress_file

VALID_STATE_LIST = ["PRE_UPGRADING", "UPGRADING", "POST_UPGRADING", "PRE_CHECKING", "ROLLBACKING"]
UPGRADE_PRECENT_MAP = {
    "PRE_UPGRADING": "10%",
    "UPGRADING": "20%",
    "POST_UPGRADING": "80%",
    "REBOOTING": "90%",
    "SUCCESS": "100%",
    "SKIP": "100%"
}
UPGRADE_RETRY_COUNT = 300
MAX_RETRY_COUNT = 10
MIN_SIZE_HCP = 2 * 1024 * 1024
HCP_PATH = "/home/hcp"
CHECK_ACTIVE_NODE_CMD = 'su -s /bin/bash hcpprocess -c "sh /opt/huawei-data-protection/ebackup/bin/' \
                        'config_omm_ha.sh query" | grep HaLocalName | grep -o "active";echo "isactive: $?"'
CHECK_ULTRAPATH_INSTALL_CMD = "rpm -qa | grep -q Ultra; echo \"UltraPath installed: $?\""
SHELL_FILE_PATH = "/root/upgrade_iso.sh"

CHECK_IS_REBOOT = f'[ "$(uname -r)" == "$(grep G_ISO_VERSION= {SHELL_FILE_PATH}' \
                  ' | awk -F "=" \'{print $2}\' | sed \'s/"//g\')" ]; echo "reboot_status:$?"'
OS_PATTERN = "Current OS version\[.*\] is not compatible."
OS_PATTERN_NEW_VERSION = "current\[.*\] is already new version."


class OSUpgradeBase(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super().__init__(project_id, pod_id)
        logger.init("eBackup OS upgrade Base")
        self.project_id = project_id
        self.pod_id = pod_id
        self.params = ParamUtil()
        self.__db_param_dict = Utils.init_system_params(project_id, regionid_list[0])
        self.region_id = regionid_list[0]
        host_str = self.__db_param_dict["eBackup_Datamover_nodes"]
        self.host_list = host_str.replace(" ", "").replace(";", "|").split("|")
        self.ssh_password = self.__db_param_dict['eBackup_hcp_pwd']
        self.ssh_root_password = self.__db_param_dict['eBackup_root_pwd']
        self.ssh_user = "hcp"
        self.ssh_info = SshInfo(self.host_list, self.ssh_user, self.ssh_password, self.ssh_root_password)
        self.skip_os_list = Utils.get_value_from_config_file("skip_os_upgrade_ips").split(",")
        self.host_list_tmp = self.host_list.copy()
        self.upgrade_state_map = {}
        self.upgrade_scene = "upgrade"

    @staticmethod
    def add_node_ip_to_cfg(ebackup_ip, key="skip_os_upgrade_ips"):
        """
        :param ebackup_ip: 需要添加的ip
        :param key: skip_os_upgrade_ips: 记录版本已经是新版本或者不用升级Os的ip列表,用于重试跳过，为节约时间已经升级过的OS不再升级
        :param key: upgrade_success_ips: 记录当前升级成功的OS版本 用于判断是否需要升级UltraPath
        :param key: active_ips: 记录待升级节点中所有的主节点ip
        """
        ip_string = Utils.get_value_from_config_file(key)
        ip_list = ip_string.split(',')
        if ip_string and ebackup_ip not in ip_list:
            Utils.set_value_to_config_file(key, f"{ip_string},{ebackup_ip}")
        elif ip_list and ebackup_ip in ip_list:
            Utils.set_value_to_config_file(key, f"{ip_string}")
        else:
            Utils.set_value_to_config_file(key, f"{ebackup_ip}")

    @staticmethod
    def check_iso_mounted(ssh_client, os_arch) -> bool:
        # 检查OS升级脚本是否已经在指定位置，如果存在不再挂载os
        ret_string = SshTool.ssh_send_comand_return(ssh_client, f"test -e {SHELL_FILE_PATH}; echo \"mounted: $?\"")
        if str(ret_string).find("mounted: 0") != -1:
            return True
        check_mount_cmd = f"mount | grep -v deleted | grep -qP '/root/OceanStor_BCManager_EulerOS_{os_arch}.iso" \
                          f".*/mnt/upgrade '; echo \"mounted: $?\""
        ret_string = SshTool.ssh_send_comand_return(ssh_client, check_mount_cmd, ebk_timeout=1)
        if str(ret_string).find("mounted: 0") == -1:
            return False
        else:
            return True

    @staticmethod
    def get_os_arch(ssh_client):
        # 有可能多套AZeBackup节点架构不一致，需要判断
        arch_check_cmd = 'uname -m | grep -q x86_64;echo "x86type: $?"'
        ret_string = SshTool.ssh_send_comand_return(ssh_client, arch_check_cmd, ebk_timeout=1)
        arch = "unknown"
        if str(ret_string).find("x86type: 0") != -1:
            arch = "X86"
        elif str(ret_string).find("x86type: 1") != -1:
            arch = "ARM"
        return arch

    @staticmethod
    def get_ip_list_from_config_file(key):
        ip_list_str = Utils.get_value_from_config_file(key)
        if not ip_list_str:
            return []
        return ip_list_str.strip().split(',')

    @staticmethod
    def check_destination_space(destination, ssh_client, min_size=1024 * 1024):
        cmd = f"df {destination} | grep -v 'Avail' | awk '{{print $4}}'"
        available_space = int(SshTool.ssh_send_comand_return(ssh_client, cmd, ebk_timeout=20)[0])
        if min_size > available_space:
            return False, available_space
        else:
            return True, available_space

    @staticmethod
    def get_tail_digit(config_name):
        tmp_digit = ''
        for element in config_name[::-1]:
            if element.isdigit():
                tmp_digit = f"{element}{tmp_digit}"
            else:
                break
        return tmp_digit

    def mount_iso(self, ssh_client, os_arch):
        file_path, file_name = self.get_os_package_name(os_arch)
        ssh_info = SshInfo(ssh_client.get('ip'), self.ssh_user, self.ssh_password)
        SshTool.upload_file(ssh_info, file_path, "/home/hcp/")
        mount_cmd = f"mkdir -p '/mnt/upgrade' && unzip -o /home/hcp/'{file_name}' -d /root && " \
                    f"mount -t iso9660 -o nosuid,noexec,nodev " \
                    f"/root/OceanStor_BCManager_EulerOS_{os_arch}.iso /mnt/upgrade;echo \"mount status:$?\""
        ret_string = SshTool.ssh_send_comand_return(ssh_client, mount_cmd, ebk_timeout=1)
        if str(ret_string).find("mount status:0") == -1:
            raise HCCIException(650076, str(ret_string))
        SshTool.ssh_send_comand_return(ssh_client, f"\cp -f /mnt/upgrade/upgrade_iso.sh {SHELL_FILE_PATH}")

    def pre_check(self, project_id, pod_id, regionid_list=None):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        pass

    def rollback(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def get_os_package_name(self, arch):
        if arch == "X86":
            package_post_name = "EulerOS_X86.zip"
        else:
            package_post_name = "EulerOS_ARM.zip"

        path, name = find_software_package_by_name(
            pkg_pre_name="OceanStor",
            pkg_post_name=package_post_name,
            pkg_version='8.5.1',
            project_id=self.project_id)
        file_path = os.path.join(path, name)
        check_compress_file([file_path])
        return file_path, name

    def get_ultrapath_package_name(self):
        path, name = find_software_package_by_name(
            pkg_pre_name="OceanStor_UltraPath_",
            pkg_post_name="_LinuxOther.zip",
            project_id=self.project_id)
        file_path = os.path.join(path, name)
        check_compress_file([file_path])
        return file_path, name

    def save_ultra_path_state_to_file(self):
        logger.info("Start save ultraPath status to cfg file.")
        for ebackup_ip in self.host_list:
            install_state = Utils.get_value_from_config_file(f"{ebackup_ip}_UltraPath_state")
            if install_state:
                logger.info(f"Already save UltraPath status of node({ebackup_ip}) to config file, skip")
                continue
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
            ret_strings = SshTool.ssh_send_comand_return(ssh_client, CHECK_ULTRAPATH_INSTALL_CMD)
            if str(ret_strings).find("UltraPath installed: 0") == -1:
                logger.info(f"The node({ebackup_ip}) is not install UltraPath software.")
                Utils.set_value_to_config_file(f"{ebackup_ip}_UltraPath_state", "False")
                continue
            logger.info(f"The node({ebackup_ip}) is installed UltraPath software.")
            Utils.set_value_to_config_file(f"{ebackup_ip}_UltraPath_state", "True")

    def save_ha_relation_to_file(self):
        logger.info("Start save active node to cfg file.")
        active_nodes = Utils.get_value_from_config_file("active_ips")
        if active_nodes:
            logger.info("Already save relation to file.")
            return

        for ebackup_ip in self.host_list:
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
            ret_strings = SshTool.ssh_send_comand_return(ssh_client, CHECK_ACTIVE_NODE_CMD)
            if str(ret_strings).find("isactive: 0") != -1:
                self.add_node_ip_to_cfg(ebackup_ip, "active_ips")

    def restore_ha_relation(self, active_ip):
        restore_ha_cmd = "export LD_LIBRARY_PATH=/opt/huawei-data-protection/ebackup/libs/ &&" \
                         " /opt/huawei-data-protection/ebackup/ha/module/hacom/tools/ha_client_tool " \
                         "--switchover --name=product;echo \"restore status:$?\""
        ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=active_ip)
        times = 0
        while times < MAX_RETRY_COUNT:
            ret_strings = SshTool.ssh_send_comand_return(ssh_client, restore_ha_cmd)
            if str(ret_strings).find("restore status:0") != -1 or str(ret_strings).find("restore status:17") != -1:
                logger.info(f"Restore ha relation success on node{active_ip}.")
                return
            logger.warning(f"restore ha relation failed, stderr: {ret_strings}."
                           f" Possibly reboot cause the failure, wait 20s to retry.")
            times += 1
            time.sleep(20)

        logger.error(f"Restore ha relation timeout.")
        HCCIException(650081, active_ip)

    def check_and_restore_ha_relation(self):
        active_nodes = self.get_ip_list_from_config_file("active_ips")
        all_nodes = self.__db_param_dict["eBackup_Datamover_nodes"].replace(" ", "").replace(";", "|").split("|")
        for ebackup_ip in all_nodes:
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
            ret_strings = SshTool.ssh_send_comand_return(ssh_client, CHECK_ACTIVE_NODE_CMD)
            if str(ret_strings).find("isactive: 0") != -1 and ebackup_ip not in active_nodes:
                logger.info(f"The node({ebackup_ip}) is active node. but not in original active node list, "
                            f"need restore ha relationship.")
                self.restore_ha_relation(ebackup_ip)

    def get_current_state(self, ssh_client, os_arch):
        try:
            if not self.check_iso_mounted(ssh_client, os_arch):
                self.mount_iso(ssh_client, os_arch)
        except Exception as mount_error:
            if self.upgrade_scene == "upgrade":
                logger.warning(f"Mount failed, reason:{mount_error}.")
                return "RETRY", "Mount failed."
            else:  # 非升级循环检查场景下 直接抛出失败不重试
                return "FAILED", f"Check mounted or mount iso failed, stderr: {mount_error}."
        get_state_cmd = f"sh {SHELL_FILE_PATH} get_current_state"
        ret_string = SshTool.ssh_send_comand_return(ssh_client, get_state_cmd, ebk_timeout=1)
        if not ret_string and self.upgrade_scene == "upgrade":
            return "RETRY", "Get current state shell execute failed."
        for output in ret_string:
            if output.strip() in VALID_STATE_LIST:
                return output.strip(), ""
            if "|" in output:
                description = output.split("|")[1].strip()
                state = output.split("|")[0].strip()
                return state, description
            if "GLIBC" in output:  # 当输出中出现GLIBC 说明正处于升级glibc等关键rpm包，此时部分命令会报错找不到glibc，为正常现象
                return "UPGRADING", ""
        raise HCCIException(650077, str(ret_string), ssh_client.get('ip'))

    def modify_network_card_configs(self):
        logger.info(f'Start to modify the network card config files on: {self.host_list}')
        network_script_path = "/etc/sysconfig/network-scripts"
        get_card_name_cmd = "ll /etc/sysconfig/network-scripts/ | grep ifcfg | awk '{print $9}' | " \
                            "grep -v 'ifcfg-lo' | grep -v ifcfg-bond"
        for node_ip in self.host_list:
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=node_ip)
            network_card_config_list = SshTool.ssh_send_comand_return_list(ssh_client, get_card_name_cmd, 20)
            for config_name in network_card_config_list:
                tail_digit = self.get_tail_digit(config_name)
                device_name = config_name[6:]
                new_config_name = f"{config_name}np{tail_digit}"
                new_device_name = f"{device_name}np{tail_digit}"
                cmd_list = [f'\cp {network_script_path}/{config_name} {network_script_path}/{new_config_name}',
                            f'sed -i "s/{device_name}/{new_device_name}/g" {network_script_path}/{new_config_name}']
                cmd_info = CmdInfo(ssh_client, cmd_list)
                SshTool.send_ssh_command(cmd_info, 30)
            logger.info(f"Successfully modified the network card config files on {node_ip}")

    def os_precheck(self):
        self.modify_network_card_configs()
        self.save_ultra_path_state_to_file()
        self.save_ha_relation_to_file()
        check_results = [
            CheckResult(itemname_ch=g_check_item.get('os_upgrade_precheck_status', dict()).get('name_cn'),
                        itemname_en=g_check_item.get('os_upgrade_precheck_status', dict()).get('name_en'),
                        status="success")
        ]
        for ebackup_ip in self.host_list:
            logger.info(f"Checking the available space on node({ebackup_ip})")
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
            delete_cmd_info = CmdInfo(ssh_client, ['rm -f /etc/yum.repos.d/sut.repo'])
            SshTool.send_ssh_command(delete_cmd_info, 50)
            check_space_res, avail_space = self.check_destination_space(HCP_PATH, ssh_client, min_size=MIN_SIZE_HCP)
            avail_space = round(avail_space / (1024 * 1024), 3)  # Describe in GB, keep 3 numbers after decimal point
            if not check_space_res:
                logger.error(f"The available space of {HCP_PATH} on node {ebackup_ip} is not enough.")
                check_results = [
                    CheckResult(itemname_ch=g_check_item.get('os_upgrade_precheck_status', dict()).get('name_cn'),
                                itemname_en=g_check_item.get('os_upgrade_precheck_status', dict()).get('name_en'),
                                status="failure", error_msg_cn=HCCIException(650086, HCP_PATH, ebackup_ip,
                                                                             MIN_SIZE_HCP / (1024*1024), avail_space))]
                break
            logger.info(f"Start precheck iso on ebackup node({ebackup_ip})")
            os_arch = self.get_os_arch(ssh_client)
            if not self.check_iso_mounted(ssh_client, os_arch):
                self.mount_iso(ssh_client, os_arch)
            precheck_cmd = f"sh {SHELL_FILE_PATH} precheck /root/OceanStor_BCManager_EulerOS_{os_arch}.iso"
            SshTool.ssh_send_comand_return(ssh_client, precheck_cmd)
            state, description = self.get_current_state(ssh_client, os_arch)

            is_version_skip = re.search(OS_PATTERN, description) or re.search(OS_PATTERN_NEW_VERSION, description)
            if state == 'SUCCESS':
                logger.info(f"Precheck iso success on ebackup node({ebackup_ip})")
                continue
            elif state == 'FAILED' and is_version_skip:
                logger.info(f"The os version is not compatible with current iso on node({ebackup_ip}). Skip upgrade.")
                self.add_node_ip_to_cfg(ebackup_ip)
            else:
                description = description.replace('[', '(').replace(']', ')')
                logger.error(f"Precheck os failed. Error reason is {description}.")
                check_results = [
                    CheckResult(itemname_ch=g_check_item.get('os_upgrade_precheck_status', dict()).get('name_cn'),
                                itemname_en=g_check_item.get('os_upgrade_precheck_status', dict()).get('name_en'),
                                status="failure",
                                error_msg_cn=HCCIException(
                                    g_check_item.get('os_upgrade_precheck_status', dict()).get('error_code'),
                                    ebackup_ip, description))]
                break
        return check_results

    def send_upgrade_os_cmd(self, ebackup_ip):
        if ebackup_ip in self.skip_os_list:
            logger.warning(f"Skip os upgrade on node({ebackup_ip}).")
            return
        ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
        if not ssh_client:
            logger.warning(f"Node({ebackup_ip}) is rebooting, skip send upgrade cmd.")
            return
        os_arch = self.get_os_arch(ssh_client)
        if not self.check_iso_mounted(ssh_client, os_arch):
            self.mount_iso(ssh_client, os_arch)

        upgrade_cmd = f"bash -c 'sh {SHELL_FILE_PATH} upgrade " \
                      f"/root/OceanStor_BCManager_EulerOS_{os_arch}.iso " \
                      f"1>>/var/log/upgrade_iso.log.`date +%s` 2>&1 &'"

        SshTool.ssh_send_comand_return(ssh_client, upgrade_cmd)

    def update_upgrade_state(self, ebackup_ip):
        if ebackup_ip in self.skip_os_list:
            self.upgrade_state_map[ebackup_ip] = ("SKIP", "")
            self.host_list_tmp.remove(ebackup_ip)
            return
        ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
        if not ssh_client:
            self.upgrade_state_map[ebackup_ip] = ("REBOOTING", "")
            logger.info(f"The node({ebackup_ip}) is rebooting.")
            return
        os_arch = self.get_os_arch(ssh_client)
        if os_arch == "unknown":  # 获取架构的命令返回为空，为ssh链接中断错误，重试三次
            current_state, description = "RETRY", "Get os arch failed."
        else:
            current_state, description = self.get_current_state(ssh_client, os_arch)
        if current_state == 'FAILED' and re.search("current\[.*\] is already new version.", description):
            logger.info(f"The os version is not compatible with current iso on node({ebackup_ip}). Skip upgrade.")
            current_state = 'SUCCESS'
        elif current_state == 'SUCCESS':
            # 如果SUCCESS的上一次状态为UPGRADING，说明当前节点仍处于更新状态后即将重启的状态，此时仍为升级中需要等待重启再次查询一次
            ret_strings = SshTool.ssh_send_comand_return(ssh_client, CHECK_IS_REBOOT, ebk_timeout=1)
            if str(ret_strings).find("reboot_status:0") == -1:
                current_state = 'UPGRADING'
        if current_state == "RETRY":  # 由于链接中断的失败时重试三次，排除由于重启导致的上传包、命令执行失败
            logger.warning(f"Retrying on node{ebackup_ip}, description: {description}.")
            if not self.upgrade_state_map.get(ebackup_ip) or self.upgrade_state_map[ebackup_ip][0] != "RETRY":
                self.upgrade_state_map[ebackup_ip] = ("RETRY", 1)
            elif self.upgrade_state_map[ebackup_ip][0] == "RETRY":
                retry_times = self.upgrade_state_map[ebackup_ip][1]
                if retry_times == 3:
                    raise HCCIException(650087, ebackup_ip)
                self.upgrade_state_map[ebackup_ip] = ("RETRY", retry_times + 1)
        else:
            self.upgrade_state_map[ebackup_ip] = (current_state, description)

        if current_state == 'FAILED' or current_state == 'SUCCESS':
            self.host_list_tmp.remove(ebackup_ip)

    def assemble_logger_strings(self):
        logger_strings = "Current state is [ "
        for key, value in self.upgrade_state_map.items():
            state = UPGRADE_PRECENT_MAP.get(value[0], '')
            logger_strings += f"{key}: {value[1] if not state else state} "
        logger_strings += "]."
        return logger_strings

    def upgrade_ultra_path(self, ebackup_ip):
        if Utils.get_value_from_config_file(f"{ebackup_ip}_UltraPath_state") != "True" or \
                ebackup_ip in self.get_ip_list_from_config_file("ultrapath_upgrade_success_ips"):
            logger.info(f"Skip to upgrade UltraPath on node({ebackup_ip})")
            return
        logger.info(f"Start to upgrade UltraPath on node({ebackup_ip})")
        file_path, file_name = self.get_ultrapath_package_name()
        ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
        script_path = os.path.join(os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
                                   'shell_tool', 'InstallUltraPath.sh')
        clean_cmd = "rm -rf /home/hcp/OceanStor_UltraPath_*_LinuxOther.zip"
        SshTool.ssh_send_comand_return(ssh_client, clean_cmd, ebk_timeout=1)
        upload_ssh_info = SshInfo(ebackup_ip, self.ssh_user, self.ssh_password)
        SshTool.upload_file(upload_ssh_info, file_path, "/home/hcp/")
        SshTool.upload_file(upload_ssh_info, script_path, "/home/hcp/")

        install_cmd = f"chmod 550 /home/hcp/InstallUltraPath.sh;dos2unix /home/hcp/InstallUltraPath.sh;" \
                      f"/bin/bash /home/hcp/InstallUltraPath.sh upgrade;" \
                      f"rm -rf /home/hcp/InstallUltraPath.sh;rm -rf /home/hcp/{file_name}"

        ret_strings = SshTool.ssh_send_comand_return(ssh_client, install_cmd, ebk_timeout=10)
        if str(ret_strings).find("install ultrapath succ") != -1:
            logger.info(f"execute upgrade ultra path successful on node({ebackup_ip})")
        else:
            logger.error(f"execute upgrade ultra path failed on {ebackup_ip}, stderr is:{ret_strings}")
            raise HCCIException(653013, install_cmd)

    def check_os_start_status(self, node_ips):
        num = 0
        success_ip_list = []
        while num < 60:
            success_ip_list = []
            for server_ip in node_ips:
                logger.info("begin to connect:" + server_ip)
                if SshTool.get_ssh_client(self.ssh_info, login_ip=server_ip):
                    logger.info(server_ip + " can connected.")
                    success_ip_list.append(server_ip)
            if len(success_ip_list) == len(node_ips):
                break
            time.sleep(20)
            num = num + 1
        failed_ips = []
        for node_ip in node_ips:
            if node_ip not in success_ip_list:
                failed_ips.append(node_ip)

        if num >= 60 and failed_ips:
            ip_str = ','.join(failed_ips)
            raise HCCIException(650082, ip_str)

    def reboot_node(self, node_ips):
        for server_ip in node_ips:
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=server_ip)
            reboot_cmd = "reboot -f"
            SshTool.ssh_send_comand_return(ssh_client, reboot_cmd, ebk_timeout=5)
        time.sleep(120)
        self.check_os_start_status(node_ips)

    def check_service_status(self, ebackup_ip, post_upgrade_os=False):
        ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
        if not ssh_client:
            raise HCCIException(650091, ebackup_ip)
        check_cmd = "service hcp status; echo hcp_status:$?"
        retry_limit = 3 if not post_upgrade_os else 20
        times = 1
        while times <= retry_limit:
            ret_strings = SshTool.ssh_send_comand_return(ssh_client, check_cmd, ebk_timeout=3)
            if str(ret_strings).find("hcp_status:0") != -1:
                logger.info("Check hcp service status success.")
                return
            logger.warning(f"Service hcp is not running, stdout:{ret_strings}, retry to check.")
            time.sleep(20)
            times += 1
        error_code = 650092 if not post_upgrade_os else 650094
        raise HCCIException(error_code, ebackup_ip)

    def stop_service(self, ebackup_ip):
        logger.info("stop hcp service before upgrading OS to avoid problems")
        stop_cmd = "service hcp stop force"
        ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
        cmd_info = CmdInfo(ssh_client, cmd_list=[stop_cmd])
        SshTool.send_ssh_command(cmd_info, 500)
        logger.info("Unmount the iso first to avoid mounting problem during upgrading")
        SshTool.ssh_send_comand_return(ssh_client, "umount /mnt/upgrade")

    def start_service(self, ebackup_ip):
        logger.info("start hcp service after upgrading OS")
        start_cmd = 'service hcp start force>/dev/null 2>&1 && echo "successful"'
        ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
        cmd_info = CmdInfo(ssh_client, cmd_list=[start_cmd])
        SshTool.send_ssh_command(cmd_info, 50)

    def os_upgrade(self):
        for ebackup_ip in self.host_list:  # 首先确认所有节点服务正常，然后停止服务
            self.check_service_status(ebackup_ip)
            self.stop_service(ebackup_ip)
        for ebackup_ip in self.host_list:
            self.send_upgrade_os_cmd(ebackup_ip)
            logger.info(f"Success send upgrade os cmd to node ({ebackup_ip}).")

        retry_count = 0
        while retry_count <= UPGRADE_RETRY_COUNT:
            for ebackup_ip in self.host_list:
                try:
                    self.update_upgrade_state(ebackup_ip)
                except Exception as upgrade_err:
                    self.upgrade_state_map[ebackup_ip] = ("EXCEPTION", repr(upgrade_err))
                    self.host_list_tmp.remove(ebackup_ip)
                    continue
            self.host_list = self.host_list_tmp.copy()
            logger.info(self.assemble_logger_strings())
            if len(self.host_list) == 0:
                logger.info("All upgrade process exit.")
                break
            time.sleep(12)  # 12 * 300 = 3600s 1h

        if len(self.host_list) != 0:
            logger.error(f"The node {self.host_list} upgrade timeout.")
            for host_ip in self.host_list:
                self.upgrade_state_map[host_ip] = ("TIMEOUT", "The node is still upgrading.")

        logger.info(self.assemble_logger_strings())
        failed_node_list = []
        for key, value in self.upgrade_state_map.items():
            if value[0] != "SKIP" and value[0] != "SUCCESS":  # SKIP状态已经在skip_os_upgrade_ips列表中无需再次添加
                failed_node_list.append(key)
            elif value[0] == "SUCCESS":
                self.add_node_ip_to_cfg(key, "upgrade_success_ips")
        if failed_node_list:
            raise HCCIException(650080, failed_node_list)
        Utils.get_value_from_config_file("upgrade_success_ips")
        successful_ips = self.get_ip_list_from_config_file("upgrade_success_ips")
        for ebackup_ip in successful_ips:
            self.upgrade_ultra_path(ebackup_ip)
            self.add_node_ip_to_cfg(ebackup_ip, "ultrapath_upgrade_success_ips")
            self.restore_os_config(ebackup_ip)

        self.reboot_node(successful_ips)
        self.check_and_restore_ha_relation()
        self.clean_network_card_config(successful_ips)
        for ebackup_ip in successful_ips:  # Verify the service status after upgrading the OS
            self.start_service(ebackup_ip)
            self.check_service_status(ebackup_ip, post_upgrade_os=True)

    def restore_os_config(self, upgrade_ip):
        logger.info(f"Start to restore tmp config on node {upgrade_ip}.")
        tmp_config_file = "/usr/lib/tmpfiles.d/tmp.conf"
        tmp_config_check = f"sed -i '/\/tmp\/upgrade/d' {tmp_config_file} &&" \
                           f" echo 'x /tmp/upgrade' >> {tmp_config_file};echo \"sedconfig_status:$?\""
        ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=upgrade_ip)
        ret_string = SshTool.ssh_send_comand_return(ssh_client, tmp_config_check, ebk_timeout=1)
        if str(ret_string).find("sedconfig_status:0") == -1:
            raise HCCIException(650088, upgrade_ip, str(ret_string))
        logger.info(f"Start to resore sudo config on node {upgrade_ip}.")
        restore_sudo_harden = "sh /opt/root_tool/StandardHardening/4.3.authentication.sh 1>/dev/null 2>&1;" \
                              "echo \"restore_status:$?\""
        ret_string = SshTool.ssh_send_comand_return(ssh_client, restore_sudo_harden, ebk_timeout=1)
        if str(ret_string).find("restore_status:0") == -1:
            raise HCCIException(650089, upgrade_ip, str(ret_string))
        logger.info(f"Restore os config on node{upgrade_ip} success.")

    def restore_dsware_file_cap(self, restore_ips):
        vbs_path = "/opt/dsware/vbs/bin/dsware_vbs"
        vbs_cli_path = "/opt/dsware/vbs/bin/vbs_cli"
        vbs_insight_path = "/opt/dsware/vbs/bin/dsware_insight"
        check_vbs_cli_cmd = "[ -d \"/opt/dsware/vbs/bin/\" ]; echo \"vbs_status:$?\""
        cap_property = "CAP_SYS_ADMIN+eip CAP_SYS_RAWIO+eip CAP_SYS_NICE+eip CAP_SYS_PTRACE+eip"
        restore_cmd = f'[ -f {vbs_path} ] && setcap \'{cap_property} CAP_DAC_OVERRIDE+eip\' {vbs_path} && ' \
                      f'[ -f {vbs_cli_path} ] && setcap \'{cap_property}\' {vbs_cli_path} && ' \
                      f'[ -f {vbs_insight_path} ] && setcap \'{cap_property}\' {vbs_insight_path}; ' \
                      f'echo "setcap_status:$?"'
        for ebackup_ip in restore_ips:
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
            ret_string = SshTool.ssh_send_comand_return(ssh_client, check_vbs_cli_cmd)
            if str(ret_string).find("vbs_status:0") == -1:
                logger.info(f"VBS client is not existed on node{ebackup_ip}, skip setcap for vbscli.")
                continue

            ret_string = SshTool.ssh_send_comand_return(ssh_client, restore_cmd)
            if str(ret_string).find("setcap_status:0") == -1:
                logger.error(f"setcap failed for vbscli, stderr: {ret_string}.")
                raise HCCIException(653140, ebackup_ip, str(ret_string))
            logger.info(f"VBS client restore cap success on node{ebackup_ip}.")

    def os_rollback(self):
        need_rollback_ips = self.get_ip_list_from_config_file("upgrade_success_ips")
        for ebackup_ip in need_rollback_ips:
            logger.info(f"Start to rollback upgraded os on node ({ebackup_ip}).")
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
            os_arch = self.get_os_arch(ssh_client)
            if not self.check_iso_mounted(ssh_client, os_arch):
                self.mount_iso(ssh_client, os_arch)
            rollback_cmd = f"sh {SHELL_FILE_PATH} rollback"
            SshTool.ssh_send_comand_return(ssh_client, rollback_cmd, ebk_timeout=10)

        self.check_os_start_status(need_rollback_ips)
        self.check_and_restore_ha_relation()
        self.restore_dsware_file_cap(need_rollback_ips)

        for ebackup_ip in need_rollback_ips:
            logger.info(f"Start to clean file after rollback on node {ebackup_ip}")
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
            self.clean_process(ssh_client)
            logger.info(f"Successfully cleaned files on node {ebackup_ip}")

    def clean_network_card_config(self, ips):
        logger.info('Deleting the unused network card config files.')
        execute_cmd_list = ['sh /home/hcp/clean_network_card_cofig.sh', 'rm -f /home/hcp/clean_network_card_cofig.sh']
        for ebackup_ip in ips:
            ssh_info = SshInfo(ebackup_ip, 'hcp', self.ssh_password, self.ssh_root_password)
            SshTool.upload_file(ssh_info,
                                '/opt/cloud/hcci/src/HCCI/plugins/eBackup/shell_tool/clean_network_card_cofig.sh',
                                '/home/hcp')
            ssh_client = SshTool.get_ssh_client(ssh_info)
            cmd_info = CmdInfo(ssh_client, cmd_list=execute_cmd_list)
            SshTool.send_ssh_command(cmd_info, 30)

    def post_clean_upgrade(self):
        for ebackup_ip in self.host_list:
            logger.info(f"Start to clean file on node ({ebackup_ip}).")
            ssh_client = SshTool.get_ssh_client(self.ssh_info, login_ip=ebackup_ip)
            self.clean_process(ssh_client)
            logger.info(f"Successfully cleaned files on node {ebackup_ip}")

    def clean_process(self, ssh_client):
        os_arch = self.get_os_arch(ssh_client)
        _, file_name = self.get_os_package_name(os_arch)
        if not self.check_iso_mounted(ssh_client, os_arch):
            self.mount_iso(ssh_client, os_arch)
        if not file_name:
            raise HCCIException("Iso zip file cannot find.")
        clean_cmd = f"sh {SHELL_FILE_PATH} clean; " \
                    f"umount /mnt/upgrade; " \
                    f"rm -rf /var/log/upgrade_iso.log.*;" \
                    f"rm -rf /home/hcp/'{file_name}';" \
                    f"rm -rf '/root/OceanStor_BCManager_EulerOS_{os_arch}.iso';" \
                    f"rm -f /root/upgrade_iso.sh; rm -f /home/hcp/ultraPath.log"
        SshTool.ssh_send_comand_return(ssh_client, clean_cmd)


class OSPrecheck(OSUpgradeBase):

    def execute(self, project_id, pod_id, regionid_list=None):
        self.upgrade_scene = "precheck"
        try:
            check_results = self.os_precheck()
        except Exception as upgrade_err:
            logger.error(f"Precheck os failed: {upgrade_err}")
            return Message(500,
                           error_msg_cn=f"OS升级前检查出现异常，请查看日志并重试，异常信息:{upgrade_err}",
                           error_msg_en=f"Exception occurs while prechecking,"
                                        f"please check the log and try again, "
                                        f"error info:{upgrade_err}.")
        finally:
            SshTool.close_all_clients()
        return Message(200, check_results=check_results)


class OSUpgradeClean(OSUpgradeBase):

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self.post_clean_upgrade()
        except Exception as upgrade_err:
            logger.error(f"clean tmp file and backup file failed: {str(upgrade_err)}")
            return Message(500, upgrade_err, upgrade_err)
        finally:
            SshTool.close_all_clients()
        return Message(200)


class OSUpgrade(OSUpgradeBase):

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self.os_upgrade()
        except Exception as upgrade_err:
            logger.error(f"Upgrade os failed: {str(upgrade_err)}")
            return Message(500, upgrade_err, upgrade_err)
        finally:
            SshTool.close_all_clients()
        return Message(200)


class OSRollback(OSUpgradeBase):

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self.os_rollback()
        except Exception as upgrade_err:
            logger.error(f"Rollback os failed: {str(upgrade_err)}")
            return Message(500, upgrade_err, upgrade_err)
        finally:
            SshTool.close_all_clients()
        return Message(200)

# the code has been updated to python3.7
