# -*- coding:utf-8 -*-
import os
import tarfile
import zipfile

import utils.common.log as logger
import utils.common.software_package_util as file_util
from utils.common.exception import HCCIException

from plugins.eReplication.common.constant import Capacity
from plugins.eReplication.common.constant import Pkg


class API(object):

    @classmethod
    def find_file(
            cls, pkg_pre_name, pkg_post_name, pkg_version=None, pkg_path=None,
            scan_subdir=True):
        return file_util.find_software_package_by_name(
            pkg_pre_name, pkg_post_name, pkg_version, pkg_path=pkg_path,
            is_scan_subdir=scan_subdir)

    @classmethod
    def get_file_abs_path(
            cls, pkg_pre_name, pkg_post_name, pkg_version=None, pkg_path=None,
            scan_subdir=True):
        path, file_name = cls.find_file(
            pkg_pre_name, pkg_post_name, pkg_version=pkg_version,
            pkg_path=pkg_path, scan_subdir=scan_subdir)
        logger.info(f"Get pkg abs path return: {path}, {file_name}.")
        if not path or not file_name:
            raise HCCIException(
                "663601", f"{pkg_pre_name}***{pkg_post_name}")
        return os.path.join(path, file_name)

    @classmethod
    def find_file_by_re(cls, re_pkg, pkg_path=None, is_scan_subdir=True):
        return file_util.find_software_package_by_name_re(
            re_pkg, pkg_path=pkg_path, is_scan_subdir=is_scan_subdir)

    @classmethod
    def get_file_abs_path_by_re(
            cls, re_pkg, pkg_path=None, is_scan_subdir=True):
        pkg_path, file_path, pkg_version = cls.find_file_by_re(
            re_pkg, pkg_path=pkg_path, is_scan_subdir=is_scan_subdir)
        logger.info(
            f"Get pkg abs path by re {re_pkg} return: {pkg_path}, {file_path}.")
        if not pkg_path or not file_path:
            logger.error(f"Failed to get pkg abs path by re {re_pkg}.")
            raise HCCIException(
                "663613", f"Failed to get pkg: {re_pkg}.")
        return os.path.join(pkg_path, file_path)

    @classmethod
    def extract_file(cls, file_path, target_path):
        return file_util.extract_software_pkg(file_path, target_path)

    @classmethod
    def get_version(cls, package_abs):
        file_fs = None
        try:
            file_fs = zipfile.ZipFile(f'{package_abs}', 'r')
        except Exception as err:
            logger.error(f"Get version failed: {err}")
            raise err
        finally:
            if file_fs:
                file_fs.close()
        for file_name in file_fs.namelist():
            if file_name.endswith(".json"):
                return file_name.split("-")[1]
        return ""

    @classmethod
    def check_compressed_file(
            cls, file, unit=Capacity.MB_S, size_limit=1, file_num_limit=1):
        """
        Check the size of the zip and tar.gz files.
        :param file:
        :param unit:
        :param size_limit:
        :param file_num_limit:
        :return:
        """
        if unit == Capacity.KB_S:
            divisor = Capacity.KB
        elif unit == Capacity.MB_S:
            divisor = Capacity.MB
        elif unit == Capacity.GB_S:
            divisor = Capacity.GB
        else:
            divisor = Capacity.BYTES
        if file.endswith(".zip"):
            zip_file = zipfile.ZipFile(file, "r")
            zip_file_name_list = zip_file.namelist()
            if len(zip_file_name_list) > file_num_limit:
                logger.error("The number of compressed files is abnormal.")
                raise Exception("The number of compressed files is abnormal.")
            total_size = 0
            for zip_file_name in zip_file_name_list:
                file_info = zip_file.getinfo(zip_file_name)
                try:
                    file_size = file_info.file_size / divisor
                except ZeroDivisionError as err:
                    logger.error(f"{divisor} cannot be used as a divisor.")
                    raise err
                total_size += file_size
                logger.info(
                    f"The size of {zip_file_name} in the compressed "
                    f"package is {file_size} {unit}.")
            if total_size > size_limit:
                raise Exception(f"The zipfile: {file} is too large.")
        elif file.endswith(".tar.gz"):
            tar_files = tarfile.open(file, "r")
            if len(tar_files.getmembers()) > file_num_limit:
                logger.error("The number of compressed files is abnormal.")
                raise Exception("The number of compressed files is abnormal.")
            total_size = 0
            for tar_file in tar_files.getmembers():
                try:
                    file_size = tar_file.size / divisor
                except ZeroDivisionError as err:
                    logger.error(f"{divisor} cannot be used as a divisor.")
                    raise err
                total_size += file_size
            if total_size > size_limit:
                raise Exception(f"The tarfile: {file} is too large.")

    @classmethod
    def get_pkg_version(cls):
        pkg_prefix = Pkg.SERVER_PKG_PREFIX
        pkg_suffix = Pkg.SERVER_PKG_SUFFIX
        pkg_path = cls.get_file_abs_path(pkg_prefix, pkg_suffix)
        pkg_version = cls.get_version(pkg_path)
        return pkg_version
