# --*-- coding: utf-8 --*--
import re
import time
from functools import wraps

import utils.common.log as logger
import utils.common.param_check as checker
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.business.vm_util import can_vm_pinged_to
from utils.common.exception import HCCIException

from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.lib.conditions import Condition

params = ParamUtil()


def auto_retry(max_retry_times=5, delay_time=60, step_time=0,
               exceptions=BaseException, sleep=time.sleep,
               callback=None, validate=None):
    """函数执行出现异常时自动重试的简单装饰器。

    :param max_retry_times:  最多重试次数。
    :param delay_time:  每次重试的延迟，单位秒。
    :param step_time:  每次重试后延迟递增，单位秒。
    :param exceptions:  触发重试的异常类型，单个异常直接传入异常类型，多个异常
    以tuple或list传入。
    :param sleep:  实现延迟的方法，默认为time.sleep。
    在一些异步框架，如tornado中，使用time.sleep会导致阻塞，可以传入自定义的方法
    来实现延迟。
    自定义方法函数签名应与time.sleep相同，接收一个参数，为延迟执行的时间。
    :param callback: 回调函数，函数签名应接收一个参数，每次出现异常时，会将异常
    对象传入。
    可用于记录异常日志，中断重试等。
    如回调函数正常执行，并返回True，则表示告知重试装饰器异常已经处理，重试装饰器
    终止重试，并且不会抛出任何异常。
    如回调函数正常执行，没有返回值或返回除True以外的结果，则继续重试。
    如回调函数抛出异常，则终止重试，并将回调函数的异常抛出。
    :param validate: 验证函数，用于验证执行结果，并确认是否继续重试。
    函数签名应接收一个参数，每次被装饰的函数完成且未抛出任何异常时，调用验证函数，
    将执行的结果传入。
    如验证函数正常执行，且返回False，则继续重试，即使被装饰的函数完成且未抛出任何
    异常。如回调函数正常执行，没有返回值或返回除False以外的结果，则终止重试，并将
    函数执行结果返回。
    如验证函数抛出异常，且异常属于被重试装饰器捕获的类型，则继续重试。
    如验证函数抛出异常，且异常不属于被重试装饰器捕获的类型，则将验证函数的异常抛出。
    :return: 被装饰函数的执行结果。
    """

    def wrapper(func):
        @wraps(func)
        def _wrapper(*args, **kwargs):
            delay, step, max_retries = delay_time, step_time, max_retry_times
            result = None
            return_flag = False
            retry_flag = True
            retry_times = 0
            error = None
            while retry_flag:
                try:
                    result = func(*args, **kwargs)
                    # 验证函数返回False时，表示告知装饰器验证不通过，继续重试
                    if callable(validate) and not validate(result):
                        logger.error("Validate result return false.")
                        continue
                    logger.info("Validate result return true.")
                    return_flag = True
                    return result
                except exceptions as ex:
                    # 回调函数返回True时，表示告知装饰器异常已经处理，终止重试
                    if callable(callback) and callback(ex) is True:
                        return result
                    error = ex
                    logger.error(f"Failed to execute {func.__name__} method, "
                                 f"err_msg: {ex}.")
                finally:
                    retry_times += 1
                    if retry_times > max_retries:
                        logger.info("Retry all complete.")
                        retry_flag = False
                        raise error
                    elif return_flag:
                        logger.info("Retry return True, all complete.")
                    else:
                        sleep_retry(delay, retry_times, step)
            return result

        def sleep_retry(delay, retry_times, step):
            if step > 0:
                delay += step
            logger.info(f"Retry after {delay} seconds, "
                        f"retry times: {retry_times}.")
            sleep(delay)

        return _wrapper

    return wrapper


def check_value_null(value):
    if value is None:
        return True
    if not value and value != 0:
        return True
    if type(value) is str and not value.strip:
        return True
    return False


def check_param_ip(host_ip):
    flag, message = checker.check_ip(host_ip)
    if not flag:
        logger.warn(message)
        return False
    return True


def check_param_string(string):
    if not checker.check_param_string(string):
        return False
    return True


def check_param_pwd(pwd):
    # 获取的统一密码开启SSH通道会被设置为'xxx'
    if "xxx" == pwd or "password" == pwd:
        return True
    if not checker.check_param_password(pwd):
        return False
    return True


def check_param_integer(integer):
    if isinstance(integer, int):
        return True
    if isinstance(integer, str) and checker.check_param_integer(integer):
        return True
    return False


def check_host_connection(host):
    if not can_vm_pinged_to(host):
        logger.warn(f"Host {host} unreachable.")
        return False
    return True


def check_param_region_id(region_id, raise_ex=True):
    if check_value_null(region_id):
        logger.error(f"The region id is empty: {region_id}.")
        if raise_ex:
            raise ValueError(f"The region id is empty: {region_id}.")
        return False
    region_naming_rule = re.compile(r"^[a-zA-Z\d-]{5,32}$")
    if not re.match(region_naming_rule, region_id):
        logger.error(
            f"Param {region_id} do not match rule {region_naming_rule}.")
        if raise_ex:
            raise ValueError(
                f"Param {region_id} do not match rule {region_naming_rule}.")
        return False
    logger.info(f"Check param {region_id} success.")
    return True


def check_param_domain_name(domain_name, raise_ex=True):
    if check_value_null(domain_name):
        logger.error(f"The domain name is empty: {domain_name}.")
        if raise_ex:
            raise ValueError(f"The domain name is empty: {domain_name}.")
        return False
    special_char_pattern = re.compile(r"[ ;'\"!&$`*?@#\[\](){}<>|\\\n]")
    match_case = re.findall(special_char_pattern, domain_name)
    if match_case:
        logger.error(
            f"The domain name {domain_name} contains high-risk "
            f"characters {match_case}.")
        if raise_ex:
            raise ValueError(
                f"The domain name {domain_name} contains high-risk "
                f"characters {match_case}.")
        return False
    logger.info(f"Check param {domain_name} success.")
    return True


def check_param_az_id(az_id, raise_ex=True):
    if check_value_null(az_id):
        logger.error(f"The az id is empty: {az_id}.")
        if raise_ex:
            raise ValueError(f"The az id is empty: {az_id}.")
        return False
    az_naming_rule = re.compile(r"^[a-zA-Z\d-]{1,50}\.[a-zA-Z\d-]{1,50}$")
    if not re.match(az_naming_rule, az_id):
        logger.error(f"Param {az_id} do not match rule {az_naming_rule}.")
        if raise_ex:
            raise ValueError(
                f"Param {az_id} do not match rule {az_naming_rule}.")
        return False
    logger.info(f"Check param {az_id} success.")
    return True


def get_param_value(pod_id, module, key, user_key):
    return params.get_param_value(pod_id, module, key, user_key)


def get_ip_address(pod_id, module, key):
    """
    获取一个IP地址,该方法加入了参数校验,会优先从指定的module中去获取,如果获取结果为空或获取到的值不是
    IP地址, 则会从当前服务配置中获取
    :param pod_id:
    :param key: 配置文件中的key值
    :param module: 从哪个模式获取该配置参数值
    :return:
    """
    ip_value = params.get_value_from_cloudparam(
        pod_id, Component.REPLICATION, key)
    logger.info(f"ip_value:{ip_value}.")
    if check_value_null(ip_value) or not check_param_ip(ip_value):
        ip_value = params.get_value_from_cloudparam(pod_id, module, key)
        if check_value_null(ip_value) or not check_param_ip(ip_value):
            logger.warn(f"Get ip address by key {key} return {ip_value}.")
            return ""
    return ip_value


def get_value_string(pod_id, module, key):
    """
    获取一个配置参数字符串,该方法加入了参数校验,会优先从指定的module中去获取,如果获取结果为空或获取到的值不是
    字符串, 则会从当前服务配置中获取
    :param pod_id:
    :param key:
    :param module:
    :return:
    """
    value = params.get_value_from_cloudparam(
        pod_id, Component.REPLICATION, key)
    if check_value_null(value) or not check_param_string(value):
        value = params.get_value_from_cloudparam(pod_id, module, key)
        if check_value_null(value) or not check_param_string(value):
            logger.warn(f"Get string by key {key} return {value}.")
            return ""
    return value


def get_string(pod_id, module, key):
    value = params.get_value_from_cloudparam(
        pod_id, Component.REPLICATION, key)
    if check_value_null(value):
        value = params.get_value_from_cloudparam(pod_id, module, key)
        if check_value_null(value):
            logger.warn(f"Get string by key {key} return {value}.")
            return ""
    return value


def get_value_pwd(pod_id, module, key):
    """
    获取一个配置参数字符串,该方法加入了参数校验,会优先从指定的module中去获取,如果获取结果为空或获取到的值不是
    字符串, 则会从当前服务配置中获取
    :param pod_id:
    :param key:
    :param module:
    :return:
    """
    value = params.get_value_from_cloudparam(
        pod_id, Component.REPLICATION, key)
    if check_value_null(value) or not check_param_pwd(value):
        value = params.get_value_from_cloudparam(pod_id, module, key)
        if check_value_null(value) or not check_param_pwd(value):
            logger.warn(f"Get pwd by key {key} failed.")
            return ""
    return value


def get_value_integer(pod_id, module, key):
    """
    获取一个配置参数数字,该方法加入了参数校验,会优先从指定的module中去获取,如果获取结果为空或获取到的值不是
    数字, 则会从当前服务配置中获取
    :param pod_id:
    :param key:
    :param module:
    :return:
    """
    value = params.get_value_from_cloudparam(
        pod_id, Component.REPLICATION, key)
    if check_value_null(value) or not check_param_integer(value):
        value = params.get_value_from_cloudparam(pod_id, module, key)
        if check_value_null(value) or not check_param_integer(value):
            logger.warn(f"Get integer by key {key} return {value}.")
            return ""
    return value


def get_install_mode(project_id, pod_id):
    """根据场景返回当前eReplication的安装模式

    :param project_id:
    :param pod_id:
    :return:
    """

    condition = Condition(project_id)
    is_primary = condition.is_primary
    second_ip = get_ip_address(
        pod_id, Component.REPLICATION, "eReplication_Second_IP")
    if condition.is_region_con_ha or condition.is_csha:
        # 启用了管理面跨AZ容灾, 或有CSHA, 表示是云服务扩展场景, 且为管理面跨AZ容灾, 此时直接返回HA模式
        return "2"
    if condition.is_global_con_dr or condition.is_csdr:
        # 当前8.0版本，CSDR部署两个server节点（主备region都以HA模式部署），模式返回2
        # 如果主region安装CSDR时只部署了一个server节点，
        # 这时候备region 安装CSDR的时候也只部署一个server节点（主备region以主备模式部署），模式返回1
        # 在安装从Region时,  此时需要用户输入主eReplication的IP地址
        if not is_primary and not second_ip:
            return "1"
        else:
            return "2"
    # 如果既没有启用管理面跨AZ容灾, 也没有启用管理面跨Region容灾, 则默认使用HA模式部署,且eReplication部署在同一个AZ内
    return "2"


def get_local_role(project_id, pod_id):
    """获取当前Region上eReplication的安装角色,

    如果部署角色是HA模式,此时不关心两个eReplication哪一个是主,哪一个是备,直接返回1
    :param project_id:
    :param pod_id:
    :return:
    """

    install_mode = get_install_mode(project_id, pod_id)
    if Condition(project_id).is_primary or install_mode == "2":
        # 首Region区, 不管是CSDR,VHA,还是CSHA,直接返回1, 表示以主端部署, 此时只有CSDR真正关心是部署为主还是备
        # 或者安装模式为2,表示HA部署,此时直接返回1
        return "1"
    return "2"


def get_server_gw(project_id):
    gw_ip = ''
    cloud_service_info = BaseOps().get_lld_params(project_id)
    for info in cloud_service_info:
        if info.get("param_name", None) in \
                ["eReplication_Server01", "eReplication_Server02",
                 "eReplication_FloatIP"]:
            gw_ip = info.get("gateway")
            if gw_ip:
                return gw_ip
    if not gw_ip:
        err_msg = \
            "Get server gateway failed, please check lld parameter."
        logger.error(err_msg)
        raise Exception(err_msg)
    return gw_ip


def get_dest_ip_list(project_id, pod_id):
    """
    获取当前分配的eReplication IP地址,
    公共eReplication IP分配放到了CSHA， 因此由CSHA来分配
    :return:
    """
    ip_1 = get_ip_address(
        pod_id, Component.REPLICATION, "server_physical_ip01")
    ip_2 = get_ip_address(
        pod_id, Component.REPLICATION, "server_physical_ip02")
    ip_3 = get_ip_address(pod_id, Component.REPLICATION, "server_float_ip")
    install_mode = get_install_mode(project_id, pod_id)
    if install_mode == "1":
        # install_mode为1的时候，此时主备region各部署一个server节点，且以主备模式部署
        if check_value_null(ip_1):
            raise HCCIException('663500', 'eReplication_Server01')
        return [ip_1]
    elif install_mode == "2":
        # 安装模式为HA时, 需要同时配置三个IP
        if check_value_null(ip_1) or check_value_null(
                ip_2) or check_value_null(ip_3):
            raise HCCIException(
                '663500',
                'eReplication_Server01,'
                'eReplication_Server02,eReplication_FloatIP')
        return [ip_1, ip_2, ip_3]
    else:
        if check_value_null(ip_1) or check_value_null(ip_2):
            raise HCCIException(
                '663500', "eReplication_Server01, eReplication_Server02")
        return [ip_1, ip_2]


def get_physics_ip(project_id, pod_id):
    """
    获取当前eReplication需要使用到的IP列表(需要操作的虚拟机)
    :param project_id:
    :param pod_id:
    :return:
    """
    ip_list = []
    dest_ip = get_dest_ip_list(project_id, pod_id)
    install_mode = get_install_mode(project_id, pod_id)
    if install_mode == "1":
        # 主region已经安装CSDR服务，且是基于8.0以前版本进行安装，即只安装了一个Server节点
        # install_mode为1的时候，只取一个节点IP
        ip_list.append(dest_ip[0])
    else:
        # install_mode不为1, 说明用户配置过安装模式,此时只能是CSHA或VHA场景或者8.0及其以后版本的CSDR
        # install_mode不为1的时候，取两个节点IP
        ip_list.append(dest_ip[0])
        ip_list.append(dest_ip[1])
    return ip_list


def get_install_services(project_id):
    install_services = list()
    condition = Condition(project_id)
    if condition.is_csdr or condition.is_hcs_global or \
            condition.is_primary_hcs_global:
        install_services += [Component.CSDR_U]
    if condition.is_csha:
        install_services += [Component.CSHA_U]
    if condition.is_vha:
        install_services += [Component.VHA_U]
    if len(install_services) > 0:
        return install_services
    # 兼容单独勾选DMK，但需要在eReplication节点安装maglev-agent
    region_id = ProjectApi().get_regionlist_by_project_Id(project_id)[0]
    cmdb_ins = ManageOneCmdbUtil(project_id)
    cloud_service_info = cmdb_ins.get_cloud_service_info(region_id, Component.REPLICATION)
    extend_info = cloud_service_info[0].get("extendInfos", [])
    for extend_ in extend_info:
        extend_key = extend_.get("key")
        extend_value = extend_.get("value")
        if extend_key in [Component.CSDR_U, Component.CSHA_U, Component.VHA_U] and extend_value == "True":
            install_services.append(extend_key)
    return install_services


def get_project_global_scale(project_id):
    scale = ProjectApi().get_project_info(project_id)['global_scale']
    scale = int(scale.split('_')[-1])
    return scale


def set_param_value(project_id: int, service_name: str,
                    region_id: str, key: str, value: str):
    params.set_service_cloud_param(project_id, service_name, region_id, key, value)
