# -*- coding:utf-8 -*-

from functools import wraps

import utils.common.log as logger
from utils.business.param_util import CheckResult
from utils.business.param_util import HandleParamCheck
from utils.business.param_util import ParamUtil
from utils.common.compat import is_legacy_hcsu_scene
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.message import RESULT_CODE

from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.lib.utils import check_host_connection
from plugins.eReplication.common.lib.utils import check_param_az_id
from plugins.eReplication.common.lib.utils import check_param_domain_name
from plugins.eReplication.common.lib.utils import check_param_integer
from plugins.eReplication.common.lib.utils import check_param_ip
from plugins.eReplication.common.lib.utils import check_param_pwd
from plugins.eReplication.common.lib.utils import check_param_region_id
from plugins.eReplication.common.lib.utils import check_value_null


def handle_task_result(func):
    def handle_func_args(*args, **kwargs):
        dec_class_name = func.__qualname__
        logger.info(f"[eReplication] {dec_class_name} process start.")
        try:
            func(*args, **kwargs)
        except HCCIException as err:
            logger.error(
                f"[eReplication] {dec_class_name} failed: {err}")
            return Message(RESULT_CODE['ERROR'], err)
        except Exception as err:
            logger.error(
                f"[eReplication] {dec_class_name} failed: {err}")
            return Message(
                RESULT_CODE['ERROR'], HCCIException('663600', str(err)))
        logger.info(f"[eReplication] {dec_class_name} return success.")
        return Message(RESULT_CODE['SUCCESS'])

    return handle_func_args


def handle_task_check_result(func):
    def handle_func_args(*args, **kwargs):
        dec_class_name = func.__qualname__
        logger.info(f"[eReplication] {dec_class_name} process start.")
        try:
            result = func(*args, **kwargs)
        except HCCIException as err:
            logger.error(
                f"[eReplication] {dec_class_name} failed: {err}")
            check_result = CheckResult(
                itemname_ch=f"执行{dec_class_name}失败，请检查。",
                itemname_en=f"Failed to run {dec_class_name}. Please check.",
                status="failure", error_msg_cn=err)
            return Message(RESULT_CODE['SUCCESS'], check_results=check_result)
        except Exception as err:
            logger.error(
                f"[eReplication] {dec_class_name} failed: {err}")
            ex = HCCIException('665002', str(err))
            check_result = CheckResult(
                itemname_ch=f"执行{dec_class_name}失败，请检查。",
                itemname_en=f"Failed to run {dec_class_name}. Please check.",
                status="failure", error_msg_cn=ex)
            return Message(RESULT_CODE['SUCCESS'], check_results=check_result)
        logger.info(f"[eReplication] {dec_class_name} return success.")
        return Message(RESULT_CODE['SUCCESS'], check_results=result)

    return handle_func_args


class CheckFuncRes(object):

    @classmethod
    def check_ip_res(cls, func):
        @wraps(func)
        def wrapped_function(*args, **kwargs):
            log_string = f"Check {func.__name__} result."
            logger.debug(log_string)
            res = func(*args, **kwargs)
            if isinstance(res, list):
                for ele in res:
                    if check_param_ip(ele):
                        continue
                    raise ValueError(f"{ele} is not a valid IP address.")
            if isinstance(res, str):
                if not check_param_ip(res):
                    raise ValueError(f"{res} is not a valid IP address.")
            return res

        return wrapped_function

    @classmethod
    def check_string_res(cls, func):
        @wraps(func)
        def wrapped_function(*args, **kwargs):
            log_string = f"Check {func.__name__} result."
            logger.debug(log_string)
            res = func(*args, **kwargs)
            if check_value_null(res):
                raise ValueError(f"{res} is null.")
            return res

        return wrapped_function

    @classmethod
    def check_pwd_res(cls, func):
        @wraps(func)
        def wrapped_function(*args, **kwargs):
            log_string = f"Check {func.__name__} result."
            logger.debug(log_string)
            res = func(*args, **kwargs)
            if not check_param_pwd(res):
                raise ValueError(
                    f"{func.__name__} return is not a valid pwd format.")
            return res

        return wrapped_function

    @classmethod
    def check_region_id_res(cls, func):
        @wraps(func)
        def wrapped_function(*args, **kwargs):
            log_string = f"Check {func.__name__} result."
            logger.debug(log_string)
            res = func(*args, **kwargs)
            if not check_param_region_id(res, raise_ex=False):
                raise ValueError(f"{res} is not a valid region id format.")
            return res

        return wrapped_function

    @classmethod
    def check_az_id_res(cls, func):
        @wraps(func)
        def wrapped_function(*args, **kwargs):
            log_string = f"Check {func.__name__} result."
            logger.debug(log_string)
            res = func(*args, **kwargs)
            if not check_param_az_id(res, raise_ex=False):
                raise ValueError(f"{res} is not a valid az id format.")
            return res

        return wrapped_function

    @classmethod
    def check_domain_res(cls, func):
        @wraps(func)
        def wrapped_function(*args, **kwargs):
            log_string = f"Check {func.__name__} result."
            logger.debug(log_string)
            res = func(*args, **kwargs)
            if not check_param_domain_name(res, raise_ex=False):
                raise ValueError(f"{res} is not a valid domain format.")
            return res

        return wrapped_function

    @classmethod
    def check_integer_res(cls, func):
        @wraps(func)
        def wrapped_function(*args, **kwargs):
            log_string = f"Check {func.__name__} result."
            logger.debug(log_string)
            res = func(*args, **kwargs)
            if not check_param_integer(res):
                raise ValueError(f"{res} is not a valid integer format.")
            return res

        return wrapped_function

    @classmethod
    def check_arb_info_res(cls, func):
        @wraps(func)
        def wrapped_function(*args, **kwargs):
            log_string = f"Check {func.__name__} result."
            logger.debug(log_string)
            res = func(*args, **kwargs)
            for key, val in res.items():
                if "ip" not in key.lower():
                    continue
                for ele in val:
                    if not check_param_ip(ele):
                        raise ValueError(f"{ele} is not a valid IP address.")
                    if not check_host_connection(ele):
                        raise ConnectionError(f"{ele} is unreachable.")
            return res

        return wrapped_function


def handle_param_check_result(func):
    @wraps(func)
    def handle_func_args(*args, **kwargs):
        dec_class_name = func.__qualname__
        logger.info(f"[eReplication] {dec_class_name} process start.")
        project_id = kwargs.get("project_id") if 'params' not in kwargs.keys() else kwargs.get("params")['project_id']
        is_hcsu_scene = is_legacy_hcsu_scene(project_id)
        check_result = HandleParamCheck() if is_hcsu_scene else ()
        params = ParamUtil().get_need_check_cloud_params(project_id, Component.REPLICATION)
        param_keys = [key for key in params.keys()]
        try:
            check_result = func(*args, **kwargs)
        except HCCIException as err:
            logger.error(
                f"[eReplication] {dec_class_name} failed: {err}")
            if is_hcsu_scene:
                check_result.set_check_result(
                    param_keys=param_keys, status=500, error_msg=err)
            else:
                data = {key: err for key in param_keys}
                check_result = False, data
            return check_result
        except Exception as err:
            logger.error(
                f"[eReplication] {dec_class_name} failed: {err}")
            ex = HCCIException('665002', str(err))
            if is_hcsu_scene:
                check_result.set_check_result(
                    param_keys=param_keys, status=500, error_msg=ex)
            else:
                data = {key: ex for key in param_keys}
                check_result = False, data
            return check_result
        logger.info(f"[eReplication] {dec_class_name} return success.")
        return check_result

    return handle_func_args
