# -*- coding:utf-8 -*-
import threading
from copy import deepcopy

import utils.common.log as logger

from plugins.eReplication.common.api.vm_api import API as VM_API
from plugins.eReplication.common.constant import Network
from plugins.eReplication.common.constant import Pkg
from plugins.eReplication.common.constant import VMName
from plugins.eReplication.common.constant import VMScale
from plugins.eReplication.common.lib.base import BaseSubJob
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.decorator import handle_task_result
from plugins.eReplication.common.lib.model import PkgFileLimitInfo
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.lib.utils import get_install_mode
from plugins.eReplication.common.lib.utils import get_project_global_scale
from plugins.eReplication.common.os_api import API as OS_API
from plugins.eReplication.common.scale_api import API as SCALE_API

logger.init("CreateServerVM")


class CreateServerVM(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(CreateServerVM, self).__init__(project_id, pod_id, regionid_list)
        self.file_size_limit_gb = 5
        self.file_num_limit = 1
        self.net_work = Network.OM
        self.condition = Condition(project_id)
        if self.condition.is_upgrade_to_hcs8:
            # 按照25网络平面部署
            self.net_work = Network.PUBLIC
        nodes = Nodes(project_id, pod_id)
        self.hosts = nodes.hosts
        self.sudo_user = nodes.sudo_user
        self.sudo_pwd = nodes.preset_sudo_pwd

    @handle_task_result
    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        # 如果用户配置了eReplication_ip外部参数, 此时认为用户是要在已有的eReplicaton上配置容灾服务,
        # 或要创建一个公共eReplication供所有容灾场景的云服务使用(CSHA/CSDR/VHA)
        pkg_file_limit_info = PkgFileLimitInfo("GB", self.file_size_limit_gb, self.file_num_limit)
        VM_API.extract_os_file(self.project_id, pkg_file_limit_info, Pkg.SERVER_EXTRACT_PATH, Pkg.SERVER_TEMP_NAME)
        scale = get_project_global_scale(project_id)
        vcpus, ram, sys_disk = \
            SCALE_API.get_deploy_vm_scale(project_id, pod_id, scale)
        self._create_vms(vcpus, ram, sys_disk)
        # 虚拟机创建刚结束时，测试是否能正常连接
        check_funcs = list()
        thread_name = threading.current_thread().name
        for host in self.hosts:
            check_func = (
                OS_API.check_os_connection, thread_name,
                (host, self.sudo_user, self.sudo_pwd), {}
            )
            check_funcs.append(check_func)
        ExcThread.exec_func_in_thread(check_funcs)

    def _create_vms(self, vcpus, ram, sys_disk):
        if get_install_mode(self.project_id, self.pod_id) == "1":
            self.hosts = [self.hosts[0]]
            # HCS 6.5.1启用了Global级管理结点跨Region容灾, 此时eReplication
            # 以主备配置eReplication, 因此每一个region内只创建一台虚拟机。
            logger.info(
                f"GlobalConDR is enable, start create vm: {self.hosts}.")
            result = VM_API.create_manage_vm(
                self.pod_id, VMName.SERVICE_NAME, VMName.GROUP_NAME, vcpus,
                ram, VMScale.DISK_0, self.hosts, self.net_work,
                sys_disk=sys_disk, image_file_name=Pkg.SERVER_TEMP_NAME,
                vm_image_name=Pkg.IMAGE_NAME, availability_zone="manage-az",
                volume_type="manager.fsa.csg")
        elif self.condition.is_region_con_ha:
            # 启用了管理结点跨AZ容灾, 或用户已确认启用了管理结点跨AZ容灾,此时
            # eReplication需要分别在Region的两个DC下面分别部署一套eReplication
            logger.info(
                f"RegionConHA is enable, start create vm: {self.hosts}.")
            funcs = list()
            kwargs = {
                "sys_disk": sys_disk, "image_file_name": Pkg.SERVER_TEMP_NAME,
                "vm_image_name": Pkg.IMAGE_NAME,
                "availability_zone": "manage-az",
                "volume_type": "manager.fsa.csg"
            }
            thread_name = threading.current_thread().name
            for index, host in enumerate(self.hosts):
                host_kwargs = deepcopy(kwargs)
                args = (
                    self.pod_id, VMName.SERVICE_NAME,
                    f"{VMName.GROUP_NAME}0{index + 1}", vcpus, ram,
                    VMScale.DISK_0, [host], self.net_work
                )
                if index == len(self.hosts) - 1:
                    host_kwargs.update({
                        "availability_zone": "dr-manage-az",
                        "volume_type": "dr.manager.fsa.csg"})
                func = (
                    VM_API.create_manage_vm, thread_name, args, host_kwargs)
                funcs.append(func)
            result = ExcThread.exec_func_in_thread_and_return(funcs)
        else:
            # 其它场景, 根据服务类型返回的ip_list, 虚拟机全部创建在同一个Region下面的同一个AZ
            logger.info(
                f"Start create common eReplication vm: {self.hosts}.")
            result = VM_API.create_manage_vm(
                self.pod_id, VMName.SERVICE_NAME, VMName.GROUP_NAME, vcpus,
                ram, VMScale.DISK_0, self.hosts, self.net_work,
                sys_disk=sys_disk, image_file_name=Pkg.SERVER_TEMP_NAME,
                vm_image_name=Pkg.IMAGE_NAME, availability_zone="manage-az",
                volume_type="manager.fsa.csg")
        logger.debug(f"VMS id: {result}.")

    def del_vm(self):
        logger.info(f"Rollback create vm: {self.hosts}.")
        if get_install_mode(self.project_id, self.pod_id) == "2" and \
                self.condition.is_region_con_ha:
            delete_funcs = list()
            thread_name = threading.current_thread().name
            for index, host in enumerate(self.hosts):
                args = (
                    self.pod_id, VMName.SERVICE_NAME,
                    f"{VMName.GROUP_NAME}0{index + 1}", [host], self.net_work
                )
                delete_func = (VM_API.delete_manage_vm, thread_name, args, {})
                delete_funcs.append(delete_func)
            ExcThread.exec_func_in_thread(delete_funcs)
            return
        elif get_install_mode(self.project_id, self.pod_id) == "1":
            self.hosts = [self.hosts[0]]
        VM_API.delete_manage_vm(
            self.pod_id, VMName.SERVICE_NAME, VMName.GROUP_NAME, self.hosts,
            self.net_work)

    def retry(self, project_id, pod_id, *args, **kwargs):
        logger.info("Retry create common vm.")
        self.del_vm()
        return self.execute(project_id, pod_id)
