# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.Logic.dmk.dmk_data import DMKChannel

from plugins.eReplication.common.lib.model import DmkDeploymentInfo
from plugins.eReplication.common.api.file_api import API as FILE_API
from plugins.eReplication.common.client.dmk_client import API as DMK_API
from plugins.eReplication.common.constant import Action
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.dr_api import API as DR_API
from plugins.eReplication.common.lib.base import BaseSubJob
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.decorator import handle_task_result
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params

logger.init("StartDRService")


class StartDRService(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(StartDRService, self).__init__(project_id, pod_id, regionid_list)
        self.condition = Condition(self.project_id)
        self.nodes = Nodes(self.project_id, self.pod_id)
        self.params = Params(self.project_id, self.pod_id)
        self.pkg_version = DR_API.get_bcm_version(self.project_id, self.pod_id)

    @handle_task_result
    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        logger.info("Begin Start eReplication service.")
        self._start_service_by_region(self.nodes.primary_hosts, True)
        if len(self.nodes.standby_hosts) > 0:
            self._start_service_by_region(self.nodes.standby_hosts, False)
        is_csdr_or_hcs_global = self.condition.is_csdr or self.condition.is_hcs_global
        if self.condition.is_sub and self.condition.is_global_con_dr and is_csdr_or_hcs_global:
            DR_API.clear_environment(self.project_id, self.pod_id, [self.nodes.primary_ip] + self.nodes.hosts)

    def _start_service_by_region(self, current_host_info, is_primary_region):
        current_region_dmk_float_ip = self.params.primary_region_dmk_float_ip \
            if is_primary_region else self.params.standby_region_dmk_float_ip
        host_info, config_info = self.get_dmk_task_config(current_host_info)
        DMK_API.login_dmk(current_region_dmk_float_ip, self.params.dmk_deploy_user, self.params.dmk_user_new_pwd)
        account_id = DMK_API.get_dmk_account_id(self.params.dmk_deploy_user, self.nodes.ssh_user)
        result = DMK_API.execute_dmk_deployment(
            DmkDeploymentInfo(Component.REPLICATION, True, self.pkg_version, Action.START_SERVER,
                              host_info, config_info, account_id), DMKChannel.AGENT)
        if not result:
            raise HCCIException("663617", "Start Service task return false.")

    def get_dmk_task_config(self, current_host_info):
        # 构造节点信息
        host_info = FILE_API.get_config_content(
            os.path.join(os.path.dirname(os.path.dirname(
                os.path.dirname(os.path.dirname(__file__)))),
                "conf", "server_hosts"))
        host_info = host_info.format(",".join(current_host_info))
        config_info = \
            f"---\nversion: {self.pkg_version}\n\n" \
            f"remote_ssh_user: {self.nodes.ssh_user}"
        logger.info(f"Host info:[{host_info}], Config info:[{config_info}].")
        return host_info, config_info
