# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
from IPy import IP
from tenacity import retry, stop_after_attempt, wait_fixed
from utils.business.arb_util import ArbApi
from utils.common.exception import HCCIException

from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import ArbInstallState
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import Pkg
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.model import ArbInfo
from plugins.eReplication.common.lib.model import SSHClientInfo
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.utils import check_param_az_id

ARB_DEFAULT_ADMIN = "quorumsvr"


class API(object):

    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        nodes = Nodes(self.project_id, self.pod_id)
        self.arb_ip = nodes.arb_ip
        self.ssh_user = nodes.arb_ssh_user
        self.ssh_pwd = nodes.arb_ssh_pwd
        self.sudo_user = nodes.arb_sudo_user
        self.sudo_pwd = nodes.arb_sudo_pwd
        condition = Condition(project_id)
        self.is_arm = condition.is_arm
        self.is_csha = condition.is_csha
        self.software_re = \
            Pkg.QUORUM_RE_ARM if self.is_arm else Pkg.QUORUM_RE_X86
        self.software_flag = Pkg.QUORUM_FLAG
        self.ssh_client = SSH_API.get_ssh_client(SSHClientInfo(self.arb_ip, self.sudo_user, self.sudo_pwd))
        # 用于记录安装仲裁软件失败的位置，重试时，精确卸载
        self.arb_install_state = ArbInstallState.NO_START_INSTALL

    @retry(stop=stop_after_attempt(3), wait=wait_fixed(30), reraise=True)
    def install_arb(self):
        is_installed = self.check_installed()
        if is_installed:
            logger.info(f"[InstallArbitration]The arb software has been installed on {self.arb_ip}.")
            return
        self._unzip_pkg()
        self._clear_installed_arb()
        self._install_arb()

    def _unzip_pkg(self):
        count_pkg_num_cmd = f"ls {Path.TMP_PATH} | " \
                            f"grep {self.software_flag} | wc -l"
        pkg_count_return = SSH_API.exec_command_return_list(
            self.ssh_client, count_pkg_num_cmd)
        logger.info(
            f"Count the number of arb pkg return {pkg_count_return}.")
        if not pkg_count_return:
            logger.error("Count the number of arb pkg failed.")
            raise HCCIException("663601", self.software_re)
        pkg_count = int(pkg_count_return[0])
        if pkg_count == 0:
            logger.error("Count the number of arb pkg failed.")
            raise HCCIException("663601", self.software_re)
        elif pkg_count > 1:
            logger.error("The number of arb pkg is greater than 1.")
            raise HCCIException(
                "663602", Path.TMP_PATH)
        else:
            unzip_cmd = \
                f'file_name=`ls {Path.TMP_PATH} | ' \
                f'grep {self.software_flag}` && unzip -o -d ' \
                f'{Path.TMP_PATH} {Path.TMP_PATH}/' \
                '"${file_name}";echo CMD_RESULT=$?'
            SSH_API.send_command(self.ssh_client, unzip_cmd, "CMD_RESULT=0")
            logger.info("Unzip arb pkg success.")

    def _install_arb(self):
        logger.info(f"Start to install arb software on {self.arb_ip}.")
        try:
            self._do_install_arb()
        except Exception as err:
            logger.error(f"Install arb software failed: {err}.")
            self._clear_installed_arb()
            raise HCCIException("663618", err) from err
        logger.info(f"Install arb software on {self.arb_ip} success.")

    def _do_install_arb(self):
        self.arb_install_state = ArbInstallState.NO_START_INSTALL
        logger.info("Install arb software step 1.")
        install_cmd = f"sh {Path.TMP_PATH}/package/quorum_server.sh -install"
        SSH_API.send_command(self.ssh_client, install_cmd, 'Continue to install')
        logger.info("Install arb software step 2.")
        SSH_API.send_command(self.ssh_client, "Y", "Enter an admin")
        logger.info("Install arb software step 3.")
        self.arb_install_state = ArbInstallState.CREATE_ADMIN_USER
        SSH_API.send_command(self.ssh_client, ARB_DEFAULT_ADMIN, "New password")
        logger.info("Install arb software step 4.")
        SSH_API.send_command(self.ssh_client, self.ssh_pwd, "Retype new password")
        logger.info("Install arb software step 5.")
        self.arb_install_state = ArbInstallState.INSTALLED_FAILED
        SSH_API.send_command(self.ssh_client, self.ssh_pwd, "install success")
        logger.info("Install arb software step 6.")
        self._check_arb_installed()

    def _clear_installed_arb(self):
        logger.info(f"Clear arb on {self.arb_ip} begin.")
        try:
            logger.info(f"Clear arb software arb_install_state={self.arb_install_state}.")
            self.ssh_client = SSH_API.get_ssh_client(SSHClientInfo(self.arb_ip, self.sudo_user, self.sudo_pwd))
            if self.arb_install_state == ArbInstallState.CREATE_ADMIN_USER:
                logger.info("Clear arb software step 1.")
                cmd = f"cat /etc/passwd | grep -q {ARB_DEFAULT_ADMIN} && userdel -r {ARB_DEFAULT_ADMIN};" \
                      f"echo CMD_RESULT=$?"
                SSH_API.send_command(self.ssh_client, cmd, "CMD_RESULT=0")
            if self.arb_install_state in [ArbInstallState.NO_START_INSTALL, ArbInstallState.CREATE_ADMIN_USER]:
                logger.info("Clear arb software step 2.")
                cmd = f"sh {Path.TMP_PATH}/package/quorum_server.sh -uninstall"
                SSH_API.send_command(self.ssh_client, cmd, "uninstall success")
            else:
                logger.info("Clear arb software step 3.")
                cmd = f"sh {Path.TMP_PATH}/package/quorum_server.sh -uninstall"
                SSH_API.send_command(self.ssh_client, cmd, "backup configuration")
                logger.info("Clear arb software step 4.")
                SSH_API.send_command(self.ssh_client, "N", "default account")
                logger.info("Clear arb software step 5.")
                SSH_API.send_command(self.ssh_client, "Y", "uninstall success")
        except Exception as err:
            logger.error(f"Clear arb failed on {self.arb_ip}: {err}.")
            raise HCCIException("663618", err) from err
        logger.info(f"Clear arb on {self.arb_ip} end.")

    def _check_arb_installed(self):
        logger.info(f"Check arb installed on {self.arb_ip} begin.")
        # 验证是否安装成功
        SSH_API.send_command(self.ssh_client, "qsadmin", "admin:/")
        logger.info(f"Install arb software step 7.")
        SSH_API.send_command(self.ssh_client, "exit")
        logger.info(f"Install arb software step 8.")
        SSH_API.send_command(self.ssh_client, f"rm -fr {Path.TMP_PATH}")
        logger.info(f"Check arb installed on {self.arb_ip} end.")

    def _check_installed(self):
        self.arb_install_state = ArbInstallState.NO_START_INSTALL
        self.ssh_client = SSH_API.get_ssh_client(SSHClientInfo(self.arb_ip, self.sudo_user, self.sudo_pwd))
        cmd = f"echo file_count=`ls -A {Path.QUORUM_INSTALL_PATH} | wc -l`"
        result = SSH_API.exec_command_return_list(self.ssh_client, cmd)
        if 'file_count=0' in str(result):
            logger.info(f"The arb software has not been installed on {self.arb_ip}.")
            return False
        cmd = f"echo count=`cat /etc/passwd | grep -q {ARB_DEFAULT_ADMIN} | wc -l`"
        result = SSH_API.exec_command_return_list(self.ssh_client, cmd)
        if 'file_count=0' in str(result):
            logger.info(f"The arb software has not been installed on {self.arb_ip}.")
            return False
        self.arb_install_state = ArbInstallState.CREATE_ADMIN_USER
        self._check_arb_installed()
        logger.info(f"The arb software has been installed on {self.arb_ip}.")
        return True

    def check_installed(self):
        logger.info(f"Check whether the arb software has been installed on {self.arb_ip}.")
        try:
            return self._check_installed()
        except Exception as err:
            logger.error(f"Check whether the arb software failed: {err}, reinstall the software.")
            self.arb_install_state = ArbInstallState.INSTALLED_FAILED
        logger.info(f"The arb software has not been installed on {self.arb_ip}.")
        return False

    def config_arb(self):
        logger.info(f"Start config quorum on {self.arb_ip}.")
        iptables = os.path.join(os.path.dirname(
            os.path.dirname(__file__)), "conf", "iptables")
        SSH_API.put_file(
            self.arb_ip, self.sudo_user, self.sudo_pwd, iptables, Path.NODE_ROOT_PATH)
        mv_cmd = f"echo y|mv {Path.NODE_ROOT_PATH}/iptables {Path.SYS_IPTABLES_PATH}"
        SSH_API.send_command(self.ssh_client, mv_cmd)
        SSH_API.send_command(
            self.ssh_client, "service iptables restart")
        SSH_API.send_command(self.ssh_client, "qsadmin", 'admin:/')
        # 这儿的端口号30002需要与eReplication/conf/iptables中的30002端口一致
        try:
            SSH_API.send_command(
                self.ssh_client,
                f"add server_ip ip={self.arb_ip} port=30002",
                ['successfully', 'already exists'])
        except Exception as err:
            logger.error("CMD add server_ip not success.")
            raise err
        SSH_API.send_command(
            self.ssh_client, "show server_ip", self.arb_ip)
        SSH_API.send_command(
            self.ssh_client, "change cipher list=0,1,2,3,4", 'success')
        SSH_API.send_command(self.ssh_client, "exit")
        logger.info(f"Config quorum on {self.arb_ip} successfully.")

    def upload_pkg(self):
        logger.info(f"Start upload package to {self.arb_ip}.")
        if self.is_csha:
            arch = self.get_vm_arch()
            replaced_to = "ARM_64" if arch == "X86_64" else "X86_64"
            self.software_re = \
                "^OceanStor(.*)QuorumServer(.*?)" \
                rf"(?<!_{replaced_to})\.zip$"
        quorum_file = PKG_API.get_file_abs_path_by_re(self.software_re)
        SSH_API.send_command(
            self.ssh_client,
            f"rm -fr {Path.TMP_PATH} && mkdir -p {Path.TMP_PATH}")
        SSH_API.put_file(
            self.arb_ip, self.sudo_user, self.sudo_pwd, quorum_file,
            Path.TMP_PATH)
        logger.info(f"Upload package to {self.arb_ip} successfully.")

    def get_vm_arch(self):
        result = SSH_API.exec_command_return_list(
            self.ssh_client, "echo arch=`arch`")
        logger.info(f"Get cpu arch on {self.arb_ip} return {result}.")
        if "x86_64" not in str(result) and "aarch64" not in str(result):
            return "ARM_64" if self.is_arm else "X86_64"
        return "X86_64" if "x86_64" in str(result) else "ARM_64"

    @classmethod
    def _init_arb_info(cls, project_id, pod_id):
        local_az = ""
        remote_az = ""
        params = Params(project_id, pod_id)
        condition = Condition(project_id)
        if condition.is_csha:
            az_map_info = params.csha_service_map_info
            for region in az_map_info.replace(' ', '').split('#'):
                value_info = region.split('|')
                az1_info = value_info[0].split(',')[0]
                az2_info = value_info[1].split(',')[0]
                local_az = \
                    f"{local_az},{az1_info}" if local_az else az1_info
                remote_az = \
                    f"{remote_az},{az2_info}" if remote_az else az2_info
        check_param_az_id(local_az)
        check_param_az_id(remote_az)
        arb_info = Params(project_id, pod_id).arb_info
        arb_ips = arb_info['ips']
        arb_port = arb_info['arb_port']
        user_name = arb_info['user_info']['name']
        user_pwd = arb_info['user_info']['password']
        dc1_name = arb_info['dc_name']['dc1_name']
        dc2_name = arb_info['dc_name']['dc2_name']
        arb_list = ""
        for index, arb_ip in enumerate(arb_ips):
            if IP(arb_ip).version() == 6:
                arb_ip = f"[{arb_ip}]"
            if index + 1 != len(arb_ips):
                arb_list = f"{arb_list}{arb_ip}:{arb_port},"
            else:
                arb_list = f"{arb_list}{arb_ip}:{arb_port}"
        logger.info(
            f"Get arb info return: {arb_list}, {dc1_name}, {dc2_name}, "
            f"{user_name}, {local_az}, {remote_az}.")
        arb_info = {
            "arb_list": arb_list, "dc1_name": dc1_name, "dc2_name": dc2_name,
            "user_name": user_name, "user_pwd": user_pwd,
            "local_az": local_az, "remote_az": remote_az}
        return ArbInfo(arb_info).arb_params

    @classmethod
    def config_cloud_arb(cls, project_id, pod_id):
        arb_info = cls._init_arb_info(project_id, pod_id)
        nodes = Nodes(project_id, pod_id)
        all_hosts = nodes.all_hosts if Condition(project_id). \
            is_sub else nodes.hosts
        params = Params(project_id, pod_id)
        region_id = params.project_region_id
        cls._get_arb_cert(
            all_hosts, nodes.ssh_user, nodes.ssh_pwd, pod_id, Path.NODE_TMP_PATH)
        for index, host in enumerate(all_hosts):
            local_dc_name = arb_info.dc1_name
            remote_dc_name = arb_info.dc2_name
            local_az_name = arb_info.local_az
            remote_az_name = arb_info.remote_az
            if index % 2:
                # 当前虚拟机创建是固定的, 第一个IP是创建在manage-az的, 第二个是在dr-manage-az的
                # 因此, 如果配置的是第二个IP, 此时dc与az要与第一个节点的对应配置相反
                local_dc_name, remote_dc_name = remote_dc_name, local_dc_name
                local_az_name, remote_az_name = remote_az_name, local_az_name
            ssh_client = None
            try:
                logger.info(f"Start config arb in {host}.")
                tmp_config = os.path.join(Path.NODE_TMP_PATH, "arb_config")
                cert_file = os.path.join(Path.NODE_TMP_PATH, "ca.crt")
                ssh_client = SSH_API.get_sudo_ssh_client(Auth(host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
                                                              nodes.sudo_pwd))
                cmd = \
                    f"chmod 750 {Path.NODE_TMP_PATH};" \
                    f"echo '{region_id}.arb.ips={arb_info.arb_list}' > " \
                    f"{tmp_config} && " \
                    f"echo '{region_id}.arb.certs={cert_file}' >> " \
                    f"{tmp_config} && " \
                    f"echo '{region_id}.arb.user={arb_info.user_name}' >> " \
                    f"{tmp_config} && " \
                    f"echo '{region_id}.arb.dc1={local_dc_name}' >> " \
                    f"{tmp_config} && " \
                    f"echo '{region_id}.arb.dc2={remote_dc_name}' >> " \
                    f"{tmp_config} && " \
                    f"echo '{region_id}.arb.az1={local_az_name}' >> " \
                    f"{tmp_config} && " \
                    f"echo '{region_id}.arb.az2={remote_az_name}' >> " \
                    f"{tmp_config} && chown -h ICUser:LEGO {cert_file} " \
                    f"{tmp_config} &&" \
                    F"sudo -u ICUser -s sh {Path.BCM_SCRIPTS_PATH}configArbitration.sh " \
                    f"autoConfig {tmp_config} {region_id};rm -f {Path.NODE_TMP_PATH}"
                SSH_API.send_command(
                    ssh_client, cmd, "enter the arbitration server's password")
                SSH_API.send_command(
                    ssh_client, arb_info.user_pwd, "Configuration succeeded.",
                    timeout=120)
            finally:
                SSH_API.close_ssh(ssh_client)

    @classmethod
    def _get_arb_cert(cls, hosts, ssh_user, ssh_pwd, pod_id, dest_path):
        for host in hosts:
            # Step3: 上传仲裁服务器证书到服务器
            ArbApi().get_arb_cers(pod_id, host, ssh_user, ssh_pwd, dest_path)

    def __del__(self):
        if self.ssh_client:
            SSH_API.close_ssh(self.ssh_client)
        logger.info(f"SSH channel is closed: {self.arb_ip}.")
