# -*- coding:utf-8 -*-
import threading
from tenacity import retry, stop_after_attempt, wait_fixed

import utils.common.log as logger
from utils.business.param_util import CheckResult
from utils.common.compat import is_legacy_hcsu_scene
from utils.common.exception import HCCIException

from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import CipherType
from plugins.eReplication.common.constant import Euler
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import VERSION_MAP
from plugins.eReplication.common.dr_api import API as DR_API
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.lib.utils import set_param_value
from plugins.eReplication.common.lib.utils import get_param_value


class API(object):

    def __init__(self, project_id, pod_id, region_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_id = region_id
        self.condition = Condition(self.project_id)
        if self.condition.is_sm_compatible:
            self.to_mode = CipherType.SM_COMPATIBLE
        elif self.condition.is_sm_only:
            self.to_mode = CipherType.SM_ONLY
        else:
            self.to_mode = CipherType.GENERAL
        self.source_mode = get_param_value(pod_id, Component.REPLICATION, 'current_cipher_type', 'current_cipher_type')
        self.nodes = Nodes(self.project_id, self.pod_id)

    @retry(stop=stop_after_attempt(3), wait=wait_fixed(30), reraise=True)
    def cipher_change(self, rollback=False):
        change_funcs = list()
        thread_name = threading.current_thread().name
        hosts = self.nodes.all_hosts if is_legacy_hcsu_scene(
            self.project_id) else self.nodes.hosts
        for host in hosts:
            logger.info(f"Change encrypt algorithm on {host} in thread.")
            auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                        self.nodes.sudo_user,
                        self.nodes.sudo_pwd)
            change_func = (self._judge_and_switch, thread_name,
                           (auth, rollback), {})
            change_funcs.append(change_func)
        result = ExcThread.exec_func_in_thread_and_return(change_funcs)
        logger.info("Change encrypt algorithm process finish.")
        # 重启服务
        if True not in result:
            return False
        return True

    def check_cipher_consistency(self):
        logger.info("Start to check type consistency.")
        funcs = list()
        thread_name = threading.current_thread().name
        cipher_type = self._query_cipher_type_from_cmdb()
        if cipher_type not in CipherType.ALL_CIPHER:
            error_msg = f"Query CMDB cipher type[{cipher_type}] failed."
            logger.error(error_msg)
            raise Exception(error_msg)
        for host in self.nodes.all_hosts:
            logger.info(f"Change encrypt algorithm on {host} in thread.")
            auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                        self.nodes.sudo_user,
                        self.nodes.sudo_pwd)
            func = (self._check_cipher_consistency, thread_name,
                    (auth, cipher_type), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread_and_return(funcs)
        logger.info("Check type consistency finish.")

    def _check_cipher_consistency(self, auth, cipher_type):
        logger.info(f"Start to check type consistency on {auth.host} begin.")
        sudo_client = None
        try:
            sudo_client = SSH_API.get_sudo_ssh_client(auth)
            bcm_is_to_mode = self._check_bcm_encrypt_algorithm(sudo_client, cipher_type)
            os_is_to_mode = self._check_os_encrypt_algorithm(sudo_client, cipher_type)
            if not bcm_is_to_mode or not os_is_to_mode:
                error_msg = f"The CMDB cipher type: {cipher_type}, {Component.REPLICATION} type consistent:" \
                            f" {bcm_is_to_mode}, os type consistent: {os_is_to_mode} on {auth.host}."
                logger.error(error_msg)
                raise Exception(error_msg)
            logger.info(f"Check type consistency on {auth.host} end.")
        finally:
            SSH_API.close_ssh(sudo_client)

    def _query_cipher_type_from_cmdb(self):
        logger.info("Query type from cmdb begin.")
        mo_api = MO_API(self.project_id, self.pod_id)
        cloud_service_info = mo_api.cmdb_ins.get_cloud_service_info(self.region_id, Component.REPLICATION)
        cipher_type = ""
        if len(cloud_service_info) == 0:
            logger.info("The cmdb info is null.")
            return cipher_type
        extend_infos = cloud_service_info[0].get("extendInfos")
        if len(extend_infos) == 0:
            logger.info("The param extend_infos is null.")
            return cipher_type
        cipher_type_list = [extend_info.get("value") for extend_info in extend_infos if
                            extend_info.get("key") == "CipherType"]
        if len(cipher_type_list) == 0:
            logger.info("The param cipher_type is null.")
            return cipher_type
        logger.info(f"Query type from cmdb end, cipher_type: {cipher_type}.")
        return cipher_type_list[0]

    def _judge_and_switch(self, auth, rollback):
        logger.info(f"Judge and switch on {auth.host}, rollback={rollback}.")
        ssh_client = SSH_API.get_sudo_ssh_client(auth)
        to_mode = self._get_to_mode(rollback)
        logger.info(f"Judge and switch on {auth.host} to mode: {to_mode}")
        bcm_is_to_mode = self._check_bcm_encrypt_algorithm(ssh_client, to_mode)
        os_is_to_mode = self._check_os_encrypt_algorithm(ssh_client, to_mode)
        if bcm_is_to_mode and os_is_to_mode:
            logger.info(f"Current install mode is already {to_mode} no need execute change task on {auth.host}.")
            return False
        logger.info(f"Check encrypt algorithm on {auth.host} "
                    f"result: bcm_is_to_mode[{bcm_is_to_mode}], os_is_to_mode[{os_is_to_mode}]")
        # 停止服务
        self._stop_bcm_service(ssh_client)
        if not bcm_is_to_mode:
            self._change_bcm_encrypt_algorithm(ssh_client, to_mode)
        if not os_is_to_mode:
            self._change_os_encrypt_algorithm(ssh_client, to_mode)
        SSH_API.close_ssh(ssh_client)
        logger.info(f"Judge and switch end on {auth.host}.")
        return True

    def _change_bcm_encrypt_algorithm(self, ssh_client, to_mode):
        logger.info(f"Change eReplication encrypt algorithm begin on {ssh_client.get('ip')}.")
        cmd = f"sudo -u ICUser -s sh {Path.BCM_SCRIPTS_PATH}changeEncrypedType.sh " \
              f"{to_mode} changeCipher;echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, cmd, timeout=120)
        if "CMD_RESULT=0" not in str(result):
            err_msg = f"Change encrypt algorithm task return {result}."
            logger.error(err_msg)
            raise Exception(err_msg)
        if not self._check_bcm_encrypt_algorithm(ssh_client, to_mode):
            err_msg = "Check whether change eReplication encrypt algorithm task execute success."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Change eReplication encrypt algorithm success on {ssh_client.get('ip')}.")

    def _change_os_encrypt_algorithm(self, ssh_client, to_mode):
        logger.info(f"Change os encrypt algorithm begin on {ssh_client.get('ip')}.")
        cmd = f"sh {Path.BCM_ROOT_SCRIPTS_PATH}changeOsCipher.sh {to_mode};echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, cmd, timeout=120)
        if "CMD_RESULT=0" not in str(result):
            err_msg = f"Change os encrypt algorithm task return {result}."
            logger.error(err_msg)
            raise Exception(err_msg)
        if not self._check_os_encrypt_algorithm(ssh_client, to_mode):
            err_msg = "Check whether change os encrypt algorithm task execute success."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Change os encrypt algorithm success on {ssh_client.get('ip')}.")

    def clear_environment(self):
        clear_funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            ssh_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                                          self.nodes.sudo_user,
                                                          self.nodes.sudo_pwd))
            logger.info(f"Clear env on {host} in thread.")
            clear_func = (
                self._clear_environment, thread_name, (ssh_client,), {})
            clear_funcs.append(clear_func)
        ExcThread.exec_func_in_thread(clear_funcs)
        logger.info("Clear env success.")

    @staticmethod
    def _clear_environment(ssh_client):
        cmd = f"rm -fr {Path.CIPHER_BACKUP_PATH};echo CMD_RESULT=$?"
        result = \
            SSH_API.exec_command_return_list(ssh_client, cmd, timeout=20)
        if "CMD_RESULT=0" not in str(result):
            raise Exception(
                f"Clear environment task return {result}.")
        logger.info("Clear environment success.")

    @staticmethod
    def _check_bcm_encrypt_algorithm(ssh_client, to_mode):
        logger.info(f"Check bcm encrypt algorithm begin")
        cmd = f"cat {Path.LEGO_FILE_PATH} | grep 'cipher.type' | awk -F'=' '{{print $2}}'"
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        return to_mode in str(result)

    def _get_to_mode(self, rollback):
        return self.source_mode if rollback else self.to_mode

    @staticmethod
    def _check_os_encrypt_algorithm(ssh_client, to_mode):
        logger.info(f"Check os encrypt algorithm begin")
        cmd = f"cat {Path.SYS_OS_FILE} | grep 'crypt_style' | awk -F'=' '{{print $2}}'"
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        if to_mode == CipherType.GENERAL:
            os_is_to_mode = CipherType.SM3 not in str(result)
        else:
            os_is_to_mode = CipherType.SM3 in str(result)
        return os_is_to_mode

    @staticmethod
    def _stop_bcm_service(ssh_client):
        stop_cmd = \
            f"echo y|sudo -u ICUser -s sh {Path.BCM_SCRIPTS_PATH}shutdownSystem.sh"
        stop_result = SSH_API.exec_command_return_list(ssh_client, stop_cmd)
        if "successfully" not in str(stop_result):
            logger.error(
                f"Stop bcm service failed, cmd execute return {stop_result}")
            raise Exception("Stop bcm service failed.")
        logger.info("Stop bcm service success on host.")

    def check_environment(self):
        check_result = list()
        for host in self.nodes.all_hosts:
            ssh_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                                          self.nodes.sudo_user,
                                                          self.nodes.sudo_pwd))
            logger.info(f"Check environment process execute on {host}.")
            check_result = self._check_environment(
                ssh_client, host, check_result)
            auth_provider = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd, self.nodes.sudo_user,
                                 self.nodes.sudo_pwd)
            check_result = self._check_service_version(
                auth_provider, check_result)
        return check_result

    @staticmethod
    def _check_environment(ssh_client, host, result_check_list):
        disk = "/opt/"
        capacity = 4
        backup_path = Path.CIPHER_BACKUP_PATH
        result_check_list = DR_API.check_whether_backup_file_exists(
            ssh_client, host, backup_path, result_check_list)
        result_check_list = DR_API.check_whether_disk_free_space_is_enough(
            ssh_client, host, disk, capacity, result_check_list)
        result_check_list = DR_API.check_whether_disk_can_be_writen(
            ssh_client, host, disk, result_check_list)
        return result_check_list

    @staticmethod
    def _check_service_version(auth_provider, result_check_list):
        host = auth_provider.host
        service_version = DR_API(auth_provider).get_current_version()
        if service_version < VERSION_MAP.get("8.1.0"):
            exception = HCCIException('675037', host)
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查服务版本[{host}]",
                    itemname_en=f"Check Service Version[{host}]",
                    status="failure", error_msg_cn=exception))
        else:
            result_check_list.append(
                CheckResult(
                    itemname_ch="检查服务版本",
                    itemname_en="Check Service Version", status="success"))
        return result_check_list

    def check_cipher_type(self):
        current_cipher_type = self.get_current_cipher_type()
        if current_cipher_type == self.to_mode:
            logger.error(
                f"The current cipher type {current_cipher_type} is "
                "the same as the target cipher type.")
            raise Exception(
                f"The current cipher type {current_cipher_type} is "
                "the same as the target cipher type.")
        set_param_value(self.project_id, Component.REPLICATION, self.region_id,
                        'current_cipher_type', current_cipher_type)
        set_param_value(self.project_id, Component.REPLICATION, self.region_id, 'target_cipher_type', self.to_mode)
        logger.info("Check cipher type success.")

    def check_cpu_arch(self):
        check_result = list()
        for host in self.nodes.all_hosts:
            ssh_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                                          self.nodes.sudo_user,
                                                          self.nodes.sudo_pwd))
            logger.info(f"Check cpu arch on {host}.")
            version_result = self._check_os_version(ssh_client, host)
            if not version_result:
                exception = HCCIException('675037', host)
                check_result.append(
                    CheckResult(
                        itemname_ch=f"检查OS版本[{host}]",
                        itemname_en=f"Check OS Version[{host}]",
                        status="failure", error_msg_cn=exception))
        if not check_result:
            check_result.append(
                CheckResult(
                    itemname_ch="检查CPU和OS", itemname_en="Check CPU And OS",
                    status="success"))
        return check_result

    @staticmethod
    def _is_x86_arch(ssh_client, host):
        result = SSH_API.exec_command_return_list(
            ssh_client, "echo arch=`arch`")
        logger.info(f"Get cpu arch on {host} return {result}.")
        if "x86_64" in str(result):
            return True
        return False

    def _check_os_version(self, ssh_client, host):
        cmd = \
            r"uname -r | grep -oP 'eulerosv\K(\d+)r(\d+)' | " \
            r"awk -F'r' '{print $1,$2}'"
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        if not result:
            raise Exception("Get Euler OS version failed, please check.")
        euler_version = int(result[0].split()[0])
        release_version = int(result[0].split()[1])
        logger.info(f"Get cpu version on {host} return "
                    f"{euler_version}.{release_version}.")
        if euler_version > Euler.LIMIT_VERSION:
            return True
        is_x86_arch = self._is_x86_arch(ssh_client, host)
        # x86操作系统需要系统版本号大于等于2.9
        x86_limit_ok = is_x86_arch and release_version >= Euler.LIMIT_X86_RELEASE
        # arm操作系统需要系统版本号大于等于2.8
        arm_limit_ok = (not is_x86_arch) and release_version >= Euler.LIMIT_ARM_RELEASE
        if euler_version == Euler.LIMIT_VERSION and (x86_limit_ok or arm_limit_ok):
            return True
        return False

    def get_current_cipher_type(self):
        logger.info("Start to get current cipher type.")
        cmd = f"cat {Path.LEGO_FILE_PATH} | grep 'cipher.type' | " \
              f"awk -F'=' '{{print $2}}'"
        for host in self.nodes.all_hosts:
            ssh_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                                          self.nodes.sudo_user,
                                                          self.nodes.sudo_pwd))
            result = SSH_API.exec_command_return_list(ssh_client, cmd)
            if result:
                logger.info(f"Get current cipher type return {result}.")
                return result[0]
        logger.error("Get current cipher type failed.")
        raise Exception("Get current cipher type failed.")

    def refresh_cmdb_extend_info(self, rollback=False):
        target_mode = self.source_mode if rollback else self.to_mode
        mo_api = MO_API(self.project_id, self.pod_id)
        region_info = mo_api.get_region_info()
        for region in region_info:
            region_id = region.get("regionCode")
            cloud_service_info = mo_api.cmdb_ins.get_cloud_service_info(region_id, Component.REPLICATION)
            if not cloud_service_info:
                logger.info(
                    f"The DR service is not installed on {region_id}, "
                    f"no need to update CMDB.")
                continue
            mo_api.update_cloud_service_extend_info(
                region_id, [{"key": "CipherType", "value": target_mode}])
        logger.info(
            f"Refresh cmdb extend info cipher type to {target_mode} success.")
