# -*- coding:utf-8 -*-
import json
import os

import utils.common.log as logger
import yaml
from utils.Logic.ServerOperate import ServerOperate
from utils.PasswordManager.PasswordManager import FSPasswordManager
from utils.business.arb_util import ArbApi
from utils.business.dns_util import DNSApi
from utils.business.dns_util import DNSNodeType
from utils.business.param_util import ParamUtil
from utils.common.compat import is_legacy_hcsu_scene
from utils.common.exception import HCCIException
from utils.common.fic_base import ToolClient
from utils.common.openstack_host import get_openstack_info_by_pod_id

from plugins.eReplication.common.api.file_api import API as FILE_API
from plugins.eReplication.common.client.dmk_client import API as DMK_API
from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Action
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import Network
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import VMName
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.decorator import CheckFuncRes
from plugins.eReplication.common.lib.model import DMKInfo
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.utils import check_value_null
from plugins.eReplication.common.lib.utils import get_dest_ip_list
from plugins.eReplication.common.lib.utils import get_install_services
from plugins.eReplication.common.lib.utils import get_ip_address
from plugins.eReplication.common.lib.utils import get_param_value
from plugins.eReplication.common.lib.utils import get_physics_ip
from plugins.eReplication.common.lib.utils import get_server_gw
from plugins.eReplication.common.lib.utils import get_string
from plugins.eReplication.common.lib.utils import get_value_integer
from plugins.eReplication.common.lib.utils import get_value_pwd
from plugins.eReplication.common.lib.utils import get_value_string
from plugins.eReplication.common.constant import ArbInfo


class Params(object):

    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.service = Component.REPLICATION
        self.params = ParamUtil()
        self.condition = Condition(self.project_id)

    @property
    @CheckFuncRes.check_domain_res
    def apicom_service_uri(self):
        return get_string(self.pod_id, self.service, "apicom_service_uri")

    @property
    @CheckFuncRes.check_domain_res
    def openstack_domain(self):
        return get_string(self.pod_id, self.service, "openstack_domain")

    @property
    @CheckFuncRes.check_az_id_res
    def openstack_region(self):
        return get_string(self.pod_id, self.service, "openstack_region")

    @property
    @CheckFuncRes.check_domain_res
    def iam_address(self):
        if is_legacy_hcsu_scene(self.project_id):
            return ""
        if self.condition.is_cloud_unify:
            return MO_API(self.project_id, self.pod_id).get_central_cloud_iam_domain()
        return get_string(self.pod_id, self.service, "iam_address")

    @property
    @CheckFuncRes.check_integer_res
    def iam_port(self):
        return get_value_integer(self.pod_id, self.service, "iam_port")

    @property
    @CheckFuncRes.check_string_res
    def iam_domain(self):
        return get_string(self.pod_id, self.service, "iam_domain")

    @property
    @CheckFuncRes.check_string_res
    def region_display_name(self):
        return get_string(self.pod_id, self.service, "region_display_name")

    @property
    @CheckFuncRes.check_domain_res
    def global_domain(self):
        if is_legacy_hcsu_scene(self.project_id):
            mo_public_params = self.params.get_service_cloud_param(
                self.project_id, "ManageOne_public_params")
            return mo_public_params.get("ManageOne_global_domain_name")
        if self.condition.is_cloud_unify:
            return MO_API(self.project_id, self.pod_id).get_central_cloud_global_domain()
        return get_string(self.pod_id, self.service, "global_domain_name")

    @property
    @CheckFuncRes.check_domain_res
    def external_global_domain(self):
        return get_string(
            self.pod_id, self.service, "external_global_domain_name")

    @property
    @CheckFuncRes.check_integer_res
    def server_port(self):
        return get_value_integer(
            self.pod_id, self.service, "server_listen_port")

    @property
    @CheckFuncRes.check_pwd_res
    def bcm_keystore_pwd(self):
        return get_value_pwd(self.pod_id, self.service, "bcm_keystore_pwd")

    @property
    @CheckFuncRes.check_pwd_res
    def data_encrypt_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.pod_id, self.service, "eReplication_data_encrypt_pwd")
        return get_value_pwd(
            self.pod_id, self.service, "eReplication_data_encrypt_pwd")

    @property
    @CheckFuncRes.check_string_res
    def time_zone(self):
        return get_string(self.pod_id, self.service, "timezone")

    @property
    @CheckFuncRes.check_integer_res
    def cpu_num(self):
        return get_value_integer(self.pod_id, self.service, "eReplication_cpu")

    @property
    @CheckFuncRes.check_integer_res
    def mem_num(self):
        return get_value_integer(self.pod_id, self.service, "eReplication_mem")

    @property
    @CheckFuncRes.check_string_res
    def csdr_service_map_info(self):
        return get_string(self.pod_id, self.service, "csdr_region_map_info")

    @property
    @CheckFuncRes.check_string_res
    def csdr_service_map_info_for_nas(self):
        return get_string(
            self.pod_id, self.service, "csdr_region_map_info_for_nas")

    @property
    @CheckFuncRes.check_string_res
    def csha_service_map_info(self):
        return get_string(self.pod_id, self.service, "csha_region_map_info")

    @property
    @CheckFuncRes.check_string_res
    def vha_service_map_info(self):
        if self.condition.is_expansion_service or \
                self.condition.is_expansion_az:
            return get_string(
                self.pod_id, self.service, "vha_volumeType_map_info_2")
        else:
            return get_string(
                self.pod_id, self.service, "vha_volumeType_map_info_1")

    @property
    @CheckFuncRes.check_domain_res
    def console_home_address(self):
        if self.condition.is_cloud_unify:
            return MO_API(self.project_id, self.pod_id).get_central_cloud_console_domain()
        return get_string(self.pod_id, self.service, "console_home_address")

    @property
    @CheckFuncRes.check_region_id_res
    def project_region_id(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_region_id_by_project_id(self.project_id)
        if self.condition.installed_dr_service_and_expand_csp:
            return get_param_value(self.pod_id, self.service, 'region_info', 'region0_id')
        return get_string(self.pod_id, self.service, "region_info")

    @property
    @CheckFuncRes.check_pwd_res
    def iam_account_preset_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.pod_id, self.service, "iam_pwd")
        return get_value_pwd(self.pod_id, self.service, "iam_pwd")

    @property
    @CheckFuncRes.check_string_res
    def get_global_scale_by_project_id(self):
        return self.params.get_project_scales_by_project_id(
            self.project_id).get("global_scale")

    @property
    @CheckFuncRes.check_ip_res
    def dmk_float_ip(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, "DMK_public_params", "dmk_floatIp")
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        return DMKInfo(dmk_info).dmk_params.dmk_floatIp

    @property
    @CheckFuncRes.check_ip_res
    def primary_region_dmk_float_ip(self):
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        dmk_params = DMKInfo(dmk_info).dmk_params
        if is_legacy_hcsu_scene(self.project_id):
            return dmk_params.dmk_floatIp
        else:
            return dmk_params.Global_DMK_Float_IP if self.condition.is_sub else dmk_params.dmk_floatIp

    @property
    @CheckFuncRes.check_ip_res
    def standby_region_dmk_float_ip(self):
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        dmk_params = DMKInfo(dmk_info).dmk_params
        if is_legacy_hcsu_scene(self.project_id):
            return dmk_params.standby_region_dmk_floatIp
        else:
            return dmk_params.dmk_floatIp if self.condition.is_sub else dmk_params.StandbyRegion_DMK_Float_IP

    @property
    @CheckFuncRes.check_pwd_res
    def primary_region_dmk_sys_user_pwd(self):
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        dmk_params = DMKInfo(dmk_info).dmk_params
        if is_legacy_hcsu_scene(self.project_id):
            return dmk_params.dmk_ui_password
        else:
            return dmk_params.Global_dmk_ui_password if self.condition.is_sub else dmk_params.dmk_ui_password

    @property
    @CheckFuncRes.check_pwd_res
    def standby_region_dmk_sys_user_pwd(self):
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        dmk_params = DMKInfo(dmk_info).dmk_params
        if is_legacy_hcsu_scene(self.project_id):
            return dmk_params.standby_region_dmk_ui_password
        else:
            return dmk_params.dmk_ui_password if self.condition.is_sub else dmk_params.StandbyRegion_DMK_UI_Password

    @property
    @CheckFuncRes.check_pwd_res
    def primary_region_dmk_os_business_user_pwd(self):
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        dmk_params = DMKInfo(dmk_info).dmk_params
        # 该参数为升级场景专用
        return dmk_params.dmk_os_business_user_password

    @property
    @CheckFuncRes.check_pwd_res
    def standby_region_dmk_os_business_user_pwd(self):
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        dmk_params = DMKInfo(dmk_info).dmk_params
        # 该参数为升级场景专用
        return dmk_params.standby_region_dmk_os_business_user_password

    @property
    @CheckFuncRes.check_ip_res
    def source_dmk_float_ip(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "current_region_dmk_float_ip")

    @property
    @CheckFuncRes.check_ip_res
    def target_dmk_float_ip(self):
        if self.condition.is_primary:
            return self.params.get_value_from_cloud_param(
                self.project_id, "DNS", "to_primary_region_dmk_float_ip")
        return self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "to_standby_region_dmk_float_ip")

    @property
    @CheckFuncRes.check_integer_res
    def dmk_port(self):
        return 8443

    @property
    @CheckFuncRes.check_string_res
    def dmk_sys_user(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, "DMK_public_params", "dmk_ui_username")
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        return DMKInfo(dmk_info).dmk_params.dmk_ui_username

    @property
    @CheckFuncRes.check_pwd_res
    def dmk_sys_user_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, "DMK_public_params", "dmk_ui_password")
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        return DMKInfo(dmk_info).dmk_params.dmk_ui_password

    @property
    @CheckFuncRes.check_string_res
    def source_dmk_sys_user(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DMK_public_params", "dmk_ui_username")

    @property
    @CheckFuncRes.check_pwd_res
    def source_dmk_sys_user_pwd(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DMK_public_params", "dmk_ui_password")

    @property
    @CheckFuncRes.check_string_res
    def target_dmk_sys_user(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DMK_public_params", "to_dmk_ui_username")

    @property
    @CheckFuncRes.check_pwd_res
    def target_dmk_sys_user_pwd(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DMK_public_params", "to_dmk_ui_password")

    @property
    @CheckFuncRes.check_string_res
    def source_dmk_dns_user(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "dns_dmk_username")

    @property
    @CheckFuncRes.check_pwd_res
    def source_dmk_dns_user_pwd(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "dns_dmk_password")

    @property
    @CheckFuncRes.check_string_res
    def target_dmk_dns_user(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "to_dns_dmk_username")

    @property
    @CheckFuncRes.check_pwd_res
    def target_dmk_dns_user_pwd(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "to_dns_dmk_password")

    @property
    @CheckFuncRes.check_string_res
    def target_dmk_nginx_user(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "Nginx", "to_nginx_dmk_username")

    @property
    @CheckFuncRes.check_pwd_res
    def target_dmk_nginx_user_pwd(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, "Nginx", "to_nginx_dmk_password")

    @property
    @CheckFuncRes.check_string_res
    def dmk_os_business_user(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, "DMK_public_params",
                "dmk_os_business_username")
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        return DMKInfo(dmk_info).dmk_params.dmk_os_business_username

    @property
    @CheckFuncRes.check_pwd_res
    def dmk_os_business_user_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, "DMK_public_params",
                "dmk_os_business_user_password")
        dmk_info = DMK_API.get_dmk_info(self.pod_id)
        return DMKInfo(dmk_info).dmk_params.dmk_os_business_user_password

    @property
    @CheckFuncRes.check_string_res
    def dmk_deploy_user(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, self.service, "dmk_server_deploy_user")
        return get_value_string(
            self.pod_id, self.service, "dmk_user_account_server")

    @property
    @CheckFuncRes.check_string_res
    def dmk_deploy_group(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, self.service, "dmk_user_group_server")
        return get_value_string(
            self.pod_id, self.service, "dmk_user_group_server")

    @property
    @CheckFuncRes.check_pwd_res
    def dmk_user_preset_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, self.service, "dmk_deploy_user_pwd")
        return get_value_pwd(
            self.pod_id, self.service, "dmk_user_preset_password")

    @property
    @CheckFuncRes.check_pwd_res
    def dmk_user_new_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.project_id, self.service, "dmk_deploy_new_pwd")
        return get_value_pwd(
            self.pod_id, self.service, "dmk_user_new_password")

    @property
    @CheckFuncRes.check_arb_info_res
    def arb_info(self):
        return ArbApi().get_arb_info(self.pod_id)

    @property
    @CheckFuncRes.check_string_res
    def arb_admin_user(self):
        return ArbApi.ARB_ADMIN_USERNAME

    @property
    @CheckFuncRes.check_pwd_res
    def arb_admin_user_pwd(self):
        return get_value_pwd(self.pod_id, self.service, "arb_admin_password")

    @property
    @CheckFuncRes.check_ip_res
    def dns_ips(self):
        dns_ips = DNSApi().get_dns_ips(DNSNodeType().OM, self.pod_id)
        return dns_ips

    @property
    @CheckFuncRes.check_ip_res
    def sdr_float_ip(self):
        return self.params.get_param_value(
            self.pod_id, Component.SDR, "ctrl_float_ip", "sdr_ctrl_float_ip")

    @property
    @CheckFuncRes.check_ip_res
    def apig_float_ip(self):
        """参数项：API_Gateway_shubao_lb_float_ip"""
        return self.params.get_param_value(
            self.pod_id, Component.APIG, 'shubao_lb_float_ip',
            'shubao_lb_float_ip')

    @property
    @CheckFuncRes.check_ip_res
    def apig_internal_float_ip(self):
        """参数项：API_Gateway_shubao_internal_lb_float_ip"""
        return self.params.get_param_value(self.pod_id, Component.APIG,
                                           'shubao_internal_lb_float_ip',
                                           'shubao_internal_lb_float_ip')

    @property
    @CheckFuncRes.check_pwd_res
    def env_os_pwd(self):
        fsp_password_manager = FSPasswordManager(self.pod_id)
        return fsp_password_manager.get_env_os_password()

    @property
    @CheckFuncRes.check_ip_res
    def openstack_reverse_ip(self):
        return self.params.get_param_value(
            self.pod_id, Component.OPENSTACK, "reverse_proxy_ip",
            "openstack_reverse_proxy_ip")

    @property
    @CheckFuncRes.check_pwd_res
    def openstack_fsp_pwd(self):
        fsp_password = "openstack_fsp_pwd" if is_legacy_hcsu_scene(self.project_id) else "openstack_fsp_password"
        return self.params.get_param_value(self.pod_id, Component.OPENSTACK, "fsp_password", fsp_password)

    @property
    @CheckFuncRes.check_pwd_res
    def openstack_sudo_pwd(self):
        root_password = "openstack_root_pwd" if is_legacy_hcsu_scene(self.project_id) else "openstack_root_password"
        return self.params.get_param_value(self.pod_id, Component.OPENSTACK, "root_password", root_password)

    @property
    @CheckFuncRes.check_string_res
    def nginx_dmk_user(self):
        return self.params.get_param_value(
            self.pod_id, Component.NGINX, "nginx_dmk_username",
            "nginx_dmk_username")

    @property
    @CheckFuncRes.check_ip_res
    def standby_dmk_float_ip(self):
        return self.params.get_param_value(
            self.pod_id, Component.NGINX, "standby_region_dmk_float_ip",
            "standby_region_dmk_float_ip")

    @property
    @CheckFuncRes.check_pwd_res
    def nginx_dmk_user_pwd(self):
        return self.params.get_param_value(
            self.pod_id, Component.NGINX, "nginx_dmk_password",
            "nginx_dmk_password")

    @property
    @CheckFuncRes.check_pwd_res
    def nginx_standby_dmk_user_pwd(self):
        return self.params.get_param_value(
            self.pod_id, Component.NGINX, "standby_region_nginx_dmk_password",
            "standby_region_nginx_dmk_password")


class Nodes(object):

    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.service = Component.REPLICATION
        self.params = ParamUtil()
        self.condition = Condition(self.project_id)
        config_path = os.path.join(
            os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
            'conf', 'env.ini')
        self.env = FILE_API(config_path)

    @property
    @CheckFuncRes.check_ip_res
    def hosts(self):
        hosts = self.env.get_value_by_key_and_sub_key(self.service, "hosts")
        if hosts and not self.condition.install_dr_service:
            return json.loads(hosts)
        if is_legacy_hcsu_scene(self.project_id):
            hosts = self.get_server_nodes_by_project()
        elif self.condition.is_current_dr_installed:
            hosts = self.get_server_nodes_by_project(
                Params(self.project_id, self.pod_id).project_region_id)
        else:
            hosts = get_physics_ip(self.project_id, self.pod_id)
        self.env.set_value_by_key_and_sub_key(
            self.service, "hosts", json.dumps(hosts))
        return hosts

    @property
    @CheckFuncRes.check_ip_res
    def primary_hosts(self):
        primary_hosts = self.env.get_value_by_key_and_sub_key(
            self.service, "primary_hosts")
        if primary_hosts:
            return json.loads(primary_hosts)
        if is_legacy_hcsu_scene(self.project_id):
            primary_hosts = self.get_ha_all_server_nodes(
                self.server_ip_01, self.ssh_user, self.ssh_pwd, self.sudo_user,
                self.sudo_pwd)
            if len(primary_hosts) not in [2, 4]:
                raise Exception("The number of cluster nodes is abnormal.")
        elif self.condition.is_primary and \
                not self.condition.is_current_dr_installed:
            primary_hosts = self.hosts
        else:
            primary_hosts = [
                get_ip_address(
                    self.pod_id, self.service, "eReplication_Primary_IP"),
                get_ip_address(
                    self.pod_id, self.service, "eReplication_Second_IP")
            ]
        self.env.set_value_by_key_and_sub_key(
            self.service, "primary_hosts", json.dumps(primary_hosts))
        return primary_hosts

    @property
    @CheckFuncRes.check_ip_res
    def standby_hosts(self):
        if self.condition.is_dr_site:
            return self.hosts
        return \
            [host for host in self.all_hosts if host not in self.primary_hosts]

    @property
    @CheckFuncRes.check_ip_res
    def all_hosts(self):
        all_hosts = self.env.get_value_by_key_and_sub_key(
            self.service, "all_hosts")
        if all_hosts:
            return json.loads(all_hosts)
        if is_legacy_hcsu_scene(self.project_id):
            all_hosts = self.get_all_server_nodes(
                self.server_ip_01, self.ssh_user, self.ssh_pwd, self.sudo_user,
                self.sudo_pwd)
            if len(all_hosts) not in [2, 4]:
                raise Exception("The number of cluster nodes is abnormal.")
        elif self.condition.is_current_dr_installed or (
                self.condition.is_sub and not self.condition.is_global_con_dr):
            all_hosts = MO_API(
                self.project_id, self.pod_id).get_all_server_nodes_from_cmdb()
        else:
            all_hosts = self.get_server_nodes_by_project()
        self.env.set_value_by_key_and_sub_key(
            self.service, "all_hosts", json.dumps(all_hosts))
        return all_hosts

    @property
    @CheckFuncRes.check_ip_res
    def server_ip_01(self):
        server_ip_01 = self.env.get_value_by_key_and_sub_key(
            self.service, "server_ip_01")
        if server_ip_01:
            return server_ip_01
        # HCS 6.5.1 CSDR为主备单节点部署，虚拟机名称不带数字编号，如果有此名称对应的虚拟机，返回此虚拟机IP地址
        vm_info = self.params.get_vm_info_by_vm_name(
            self.project_id, VMName.VM_NAME_PREFIX)
        if vm_info:
            server_ip_01 = vm_info[0].get('ip')
            self.env.set_value_by_key_and_sub_key(
                self.service, "server_ip_01", server_ip_01)
            return server_ip_01
        # 考虑VHA改造扩容CSDR场景，从DNS配置文件中查询到浮动IP，如果浮动IP地址跟其中一个节点地址一致，返回改地址即可，否则，返回vm1的地址
        vm1_info = self.params.get_vm_info_by_vm_name(
            self.project_id, VMName.VM_NAME_01)
        vm2_info = self.params.get_vm_info_by_vm_name(
            self.project_id, VMName.VM_NAME_02)
        if vm1_info and vm2_info:
            service_type = get_install_services(self.project_id)[0]
            float_ip = Nodes._get_float_ip_from_dns(self.pod_id, service_type)
            if float_ip == vm2_info[0].get('ip'):
                server_ip_01 = vm2_info[0].get('ip')
            else:
                server_ip_01 = vm1_info[0].get('ip')
        elif vm1_info and not vm2_info:
            server_ip_01 = vm1_info[0].get('ip')
        elif not vm1_info and vm2_info:
            server_ip_01 = vm2_info[0].get('ip')
        else:
            raise Exception("Server VMs not found.")
        self.env.set_value_by_key_and_sub_key(
            self.service, "server_ip_01", server_ip_01)
        return server_ip_01

    @property
    @CheckFuncRes.check_ip_res
    def local_ip(self):
        return get_physics_ip(self.project_id, self.pod_id)[0]

    @property
    @CheckFuncRes.check_ip_res
    def peer_ip(self):
        return get_physics_ip(self.project_id, self.pod_id)[1]

    @property
    @CheckFuncRes.check_ip_res
    def gw_ip(self):
        if self.condition.is_project_deploy:
            return get_ip_address(
                self.pod_id, self.service, "external_om_gateway_ip")
        elif self.condition.is_expansion_service:
            return get_server_gw(self.project_id)
        return ""

    @property
    @CheckFuncRes.check_ip_res
    def primary_ip(self):
        return get_ip_address(
            self.pod_id, self.service, "eReplication_Primary_IP")

    @property
    @CheckFuncRes.check_ip_res
    def second_ip(self):
        return get_ip_address(
            self.pod_id, self.service, "eReplication_Second_IP")

    @property
    @CheckFuncRes.check_string_res
    def ssh_user(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.pod_id, self.service, "server_business_user")
        if self.condition.installed_dr_service_and_expand_csp:
            return "DRManager"
        return get_value_string(
            self.pod_id, self.service, "server_node_ssh_user")

    @property
    @CheckFuncRes.check_pwd_res
    def ssh_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_param_value(
                self.pod_id, self.service, "server_business_user_pwd",
                "eReplication_server_node_ssh_password")
        if self.condition.expand_service_or_install_in_sub:
            return get_value_pwd(self.pod_id, self.service, "server_node_ssh_password")
        if self.condition.installed_dr_service_and_expand_csp:
            return self.params.get_param_value(
                self.pod_id, self.service, 'server_node_ssh_password',
                'eReplication_server_node_ssh_password')
        return get_value_pwd(
            self.pod_id, self.service, "server_node_ssh_password")

    @property
    @CheckFuncRes.check_string_res
    def sudo_user(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.pod_id, self.service, "server_administrator_user")
        return get_value_string(
            self.pod_id, self.service, "server_node_sudo_user")

    @property
    @CheckFuncRes.check_pwd_res
    def sudo_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_param_value(
                self.pod_id, self.service, "server_root_pwd",
                "eReplication_server_node_ssh_sudo_password")
        if self.condition.expand_service_or_install_in_sub:
            return self.preset_sudo_pwd
        if self.condition.installed_dr_service_and_expand_csp:
            return self.params.get_param_value(
                self.pod_id, self.service, 'server_node_ssh_sudo_password',
                'eReplication_server_node_root_password')
        return self.preset_sudo_pwd

    @property
    @CheckFuncRes.check_pwd_res
    def preset_sudo_pwd(self):
        return get_value_pwd(
            self.pod_id, self.service, "server_node_ssh_sudo_password")

    @property
    @CheckFuncRes.check_ip_res
    def service_ip(self):
        if self.condition.is_service_installed:
            return get_ip_address(self.pod_id, self.service, "eReplication_ip")
        return self.float_ip

    @property
    @CheckFuncRes.check_string_res
    def service_name(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.pod_id, self.service, "sync_admin_name")
        return get_value_string(self.pod_id, self.service, "server_ui_user")

    @property
    @CheckFuncRes.check_integer_res
    def service_port(self):
        if is_legacy_hcsu_scene(self.project_id):
            return 9443
        if self.condition.is_service_installed:
            return get_value_integer(
                self.pod_id, self.service, "eReplication_port")
        return get_value_integer(
            self.pod_id, self.service, "server_listen_port")

    @property
    @CheckFuncRes.check_pwd_res
    def service_pwd(self):
        if is_legacy_hcsu_scene(self.project_id):
            return self.params.get_value_from_cloud_param(
                self.pod_id, self.service, "sync_admin_pwd")
        if self.condition.is_service_installed:
            return get_value_pwd(self.pod_id, self.service, "eReplication_pwd")
        return get_value_pwd(self.pod_id, self.service, "server_ui_pwd")

    @property
    @CheckFuncRes.check_ip_res
    def float_ip(self):
        if self.condition.is_current_dr_installed:
            return get_ip_address(self.pod_id, self.service, "eReplication_ip")
        return get_dest_ip_list(self.project_id, self.pod_id)[2]

    @property
    @CheckFuncRes.check_ip_res
    def arb_ip(self):
        if self.condition.is_csha:
            return get_ip_address(self.pod_id, self.service, ArbInfo.ARB_IP)
        return get_ip_address(self.pod_id, self.service, "vha_quorum_ip")

    @property
    @CheckFuncRes.check_string_res
    def arb_ssh_user(self):
        return get_value_string(self.pod_id, self.service, "quorum_user_name")

    @property
    @CheckFuncRes.check_pwd_res
    def arb_ssh_pwd(self):
        return get_value_pwd(self.pod_id, self.service, "quorum_user_pwd")

    @property
    @CheckFuncRes.check_string_res
    def arb_sudo_user(self):
        return get_value_string(self.pod_id, self.service, "quorum_sudo_user")

    @property
    @CheckFuncRes.check_pwd_res
    def arb_sudo_pwd(self):
        if self.condition.is_csha:
            return get_value_pwd(
                self.pod_id, self.service, ArbInfo.ARB_USER_PWD)
        return get_value_pwd(
            self.pod_id, self.service, "vha_quorum_vm_sudo_pwd")

    @property
    @CheckFuncRes.check_ip_res
    def source_ip(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, self.service,
            "source_site_eReplication_physical_ip")

    @property
    @CheckFuncRes.check_ip_res
    def target_ip(self):
        return self.params.get_value_from_cloud_param(
            self.project_id, self.service,
            "target_site_eReplication_physical_ip")

    @property
    @CheckFuncRes.check_pwd_res
    def source_ssh_pwd(self):
        return self.params.get_param_value(
            self.pod_id, self.service, "source_site_server_business_user_pwd",
            "source_site_server_business_user_pwd")

    @property
    @CheckFuncRes.check_pwd_res
    def source_sudo_pwd(self):
        return self.params.get_param_value(
            self.pod_id, self.service, "source_site_server_root_pwd",
            "source_site_server_root_pwd")

    @property
    @CheckFuncRes.check_pwd_res
    def target_ssh_pwd(self):
        return self.params.get_param_value(
            self.pod_id, self.service, "target_site_server_business_user_pwd",
            "target_site_server_business_user_pwd")

    @property
    @CheckFuncRes.check_pwd_res
    def target_sudo_pwd(self):
        return self.params.get_param_value(
            self.pod_id, self.service, "target_site_server_root_pwd",
            "target_site_server_root_pwd")

    @property
    @CheckFuncRes.check_ip_res
    def source_all_hosts(self):
        source_all_hosts = self.env.get_value_by_key_and_sub_key(
            self.service, "source_all_hosts")
        if source_all_hosts:
            return json.loads(source_all_hosts)
        source_all_hosts = self.get_all_server_nodes(
            self.source_ip, self.ssh_user, self.source_ssh_pwd, self.sudo_user,
            self.source_sudo_pwd)
        if len(source_all_hosts) not in [2, 4]:
            raise Exception("The number of cluster nodes is abnormal.")
        self.env.set_value_by_key_and_sub_key(
            self.service, "source_all_hosts", json.dumps(source_all_hosts))
        return source_all_hosts

    @property
    @CheckFuncRes.check_ip_res
    def source_primary_hosts(self):
        source_primary_hosts = self.env.get_value_by_key_and_sub_key(
            self.service, "source_primary_hosts")
        if source_primary_hosts:
            return json.loads(source_primary_hosts)
        source_primary_hosts = self.get_ha_all_server_nodes(
            self.source_ip, self.ssh_user, self.source_ssh_pwd, self.sudo_user,
            self.source_sudo_pwd)
        if len(source_primary_hosts) not in [2, 4]:
            raise Exception("The number of cluster nodes is abnormal.")
        self.env.set_value_by_key_and_sub_key(
            self.service, "source_primary_hosts",
            json.dumps(source_primary_hosts))
        return source_primary_hosts

    @property
    @CheckFuncRes.check_ip_res
    def source_standby_hosts(self):
        return [host for host in self.source_all_hosts if
                host not in self.source_primary_hosts]

    @property
    @CheckFuncRes.check_ip_res
    def target_all_hosts(self):
        target_all_hosts = self.env.get_value_by_key_and_sub_key(
            self.service, "target_all_hosts")
        if target_all_hosts:
            return json.loads(target_all_hosts)
        target_all_hosts = self.get_all_server_nodes(
            self.target_ip, self.ssh_user, self.target_ssh_pwd, self.sudo_user,
            self.target_sudo_pwd)
        if len(target_all_hosts) not in [2, 4]:
            raise Exception("The number of cluster nodes is abnormal.")
        self.env.set_value_by_key_and_sub_key(
            self.service, "target_all_hosts", json.dumps(target_all_hosts))
        return target_all_hosts

    @property
    @CheckFuncRes.check_ip_res
    def target_primary_hosts(self):
        target_primary_hosts = self.env.get_value_by_key_and_sub_key(
            self.service, "target_primary_hosts")
        if target_primary_hosts:
            return json.loads(target_primary_hosts)
        target_primary_hosts = self.get_ha_all_server_nodes(
            self.target_ip, self.ssh_user, self.target_ssh_pwd, self.sudo_user,
            self.target_sudo_pwd)
        if len(target_primary_hosts) not in [2, 4]:
            raise Exception("The number of cluster nodes is abnormal.")
        self.env.set_value_by_key_and_sub_key(
            self.service, "target_primary_hosts",
            json.dumps(target_primary_hosts))
        return target_primary_hosts

    @property
    @CheckFuncRes.check_ip_res
    def target_standby_hosts(self):
        return [host for host in self.target_all_hosts if
                host not in self.target_primary_hosts]

    @classmethod
    def _get_float_ip_from_dns(cls, pod_id, service_type):
        dmk_float_ip = ParamUtil().get_param_value(pod_id, "DMK", "dmk_floatIp", "DMK_dmk_floatIp")
        dmk_ui_username = ParamUtil().get_param_value(pod_id, "DMK", "dmk_ui_username", "DMK_dmk_ui_username")
        dmk_ui_password = ParamUtil().get_param_value(pod_id, "DMK", "dmk_ui_password", "DMK_dmk_ui_password")
        DMK_API.login_dmk(dmk_float_ip, dmk_ui_username, dmk_ui_password)
        try:
            (old_config, _, _) = DMK_API.get_dmk_new_deploy_config("DNS-Internal", Action.REGISTER_OM_DNS)
        except HCCIException as err:
            logger.error(f"Get dmk new deploy config failed, error: {err}")
            (old_config, _, _) = DMK_API.get_dmk_new_deploy_config("DNS-Internal", Action.REGISTER_DNS)
            raise err
        dns_config = yaml.safe_load(old_config)
        om_config = dns_config.get("om_config")
        domains = om_config.get("domains")
        for domain in domains:
            rrsets = domain.get("rrsets")
            for rrset in rrsets:
                if rrset.get("name") == service_type.lower():
                    server_listen_ip = rrset.get("records")[0]
                    return server_listen_ip
        return ""

    def get_server_nodes_by_project(self, region_id=None):
        logger.info("Get server node ips start.")
        upgraded_to_hcs80 = self.condition.is_upgrade_to_hcs8
        net_work = Network.OM
        if upgraded_to_hcs80:
            # 按照25网络平面部署
            net_work = Network.PUBLIC
        server_ip_lst = list()
        cmdb_server_ip_lst = MO_API(self.project_id, self.pod_id). \
            get_all_server_nodes_from_cmdb(region_id=region_id)
        if not is_legacy_hcsu_scene(self.project_id) and (
                self.condition.is_primary or self.condition.is_global_con_dr):
            server_name_lst = ["Service-eReplication01",
                               "Service-eReplication02"]
            server_operate = ServerOperate()
            host_and_tool = self._get_host_and_tool(
                self.pod_id, ToolClient.OPENSTACK_COMPUTE)
            ret = server_operate.list_server(host_and_tool)
            if not ret:
                logger.error("Query server info failed.")
                raise Exception("Query server info failed.")
            vm_list = ret['servers']
            vm_ip = None
            for vm_info in vm_list:
                if vm_info['name'] in server_name_lst:
                    addresses = vm_info["addresses"]
                    vm_ip = addresses[net_work][0]["addr"]
                    server_ip_lst.append(vm_ip)
            if not vm_ip:
                logger.error("Get Server ip failed.")
                raise Exception("Get Server ip failed.")
        for server_ip in cmdb_server_ip_lst:
            if server_ip not in server_ip_lst:
                server_ip_lst.append(server_ip)
        logger.info(f"Get Server node ips return {server_ip_lst}.")
        return server_ip_lst

    @staticmethod
    def _get_host_and_tool(pod_id, tool_client):
        auth_provider = get_openstack_info_by_pod_id(pod_id)
        host_and_tool = auth_provider, tool_client
        return host_and_tool

    @staticmethod
    def get_ips_from_execute_cmd_output(output):
        ip_list = list()
        for res in output:
            if 'IP_LIST=' not in res or "echo IP_LIST=" in res:
                continue
            logger.info(f"Find the all dest str: {res}.")
            ips = res.replace('\n', '').split('=')[1].split(' ')
            for host_ip in ips:
                if not check_value_null(host_ip) and host_ip not in ip_list:
                    ip_list.append(host_ip)
            break
        return ip_list

    def get_all_server_nodes(
            self, host, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        sudo_client = SSH_API.get_sudo_ssh_client(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd))
        is_single = self._check_whether_the_system_is_single_node(sudo_client)
        if is_single:
            return [host]
        logger.info(f"Start get all server nodes info in {host}.")
        ha_ip_list = self.get_ha_all_server_nodes(
            host, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
        # 非单机模式, 从当前节点nodeAgent.properties查出当前环境的所有Server节点
        result = SSH_API.exec_command_return_list(
            sudo_client,
            f"[ -f '{Path.NODE_AGENT_PROPERTIES_PATH}' ] && "
            "echo CMD_RESULT=$?")
        logger.info(f"Check exists result: {str(result)}.")
        if 'CMD_RESULT=0' not in str(result):
            logger.error(
                "Can not find the nodeAgent.properties file in "
                f"{host}.")
            ip_lst = ha_ip_list if len(ha_ip_list) > len([host]) else [host]
            return ip_lst
        cmd = f"echo IP_LIST=`cat {Path.NODE_AGENT_PROPERTIES_PATH} | " \
              "grep -E '^node' | awk -F '=' '{print $2}' | " \
              "awk -F ';' '{print $1}'`"
        results = SSH_API.exec_command_return_list(sudo_client, cmd)
        node_agent_ip_list = self.get_ips_from_execute_cmd_output(results)
        ip_lst = ha_ip_list if len(ha_ip_list) > len(
            node_agent_ip_list) else node_agent_ip_list
        logger.info(f"Get all ips return: {ip_lst}.")
        return ip_lst

    def get_ha_all_server_nodes(
            self, host, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        ip_list = list()
        # 非单机模式, 从当前节点查询出对端节点信息
        sudo_client = SSH_API.get_sudo_ssh_client(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd))
        result = SSH_API.exec_command_return_list(
            sudo_client,
            f"[ -f '{Path.HA_SYN_XML_PATH}' ] && echo CMD_RESULT=$?")
        logger.info(f"Check exists result: {str(result)}.")
        if 'CMD_RESULT=0' not in str(result):
            logger.error(f"Can not find the hasync.xml file in {host}.")
            return ip_list
        cmd = f"echo IP_LIST=`cat {Path.HA_SYN_XML_PATH} | " \
              "grep 'node name' | awk -F'ip=\"|\" port' '{print $2,$4}'`"
        results = SSH_API.exec_command_return_list(sudo_client, cmd)
        ip_list = self.get_ips_from_execute_cmd_output(results)
        return ip_list

    @staticmethod
    def _check_whether_the_system_is_single_node(sudo_client):
        check_single_cmd = \
            f"echo iNodeCount=`cat {Path.HA_COM_XML_PATH} | grep -E " \
            "'cascade.*disable|hamode.*single' | wc -l`"
        check_result = SSH_API.exec_command_return_list(
            sudo_client, check_single_cmd)
        if "iNodeCount=2" in str(check_result):
            logger.info("Current server is single mode.")
            return True
        return False
