# -*- coding:utf-8 -*-
import datetime
import os
import time

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.PasswordManager.OsUserPasswdManager import COMMON_CODE
from utils.PasswordManager.OsUserPasswdManager import ERROR_CODE
from utils.PasswordManager.OsUserPasswdManager import EXCEPT_CODE
from utils.PasswordManager.OsUserPasswdManager import SSH_RET_LENGTH
from utils.PasswordManager.OsUserPasswdManager import getPasswdExpires

from plugins.eReplication.common.lib.model import SSHClientInfo
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.lib.utils import check_param_ip


class API(object):

    def __init__(self, host, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        self.host = host
        self.ssh_user = ssh_user
        self.ssh_pwd = ssh_pwd
        self.sudo_user = sudo_user
        self.sudo_pwd = sudo_pwd
        self.ssh_client = SSH_API.get_sudo_ssh_client(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd))

    @classmethod
    def check_os_connection(cls, host, ssh_user, ssh_pwd):
        # 当前容灾用到的所有虚拟机都是用的相同的root密码，因此，当前连接虚拟机的密码在此统一获取
        logger.info(f"Check whether the host {host} can be logged in.")
        err_msg = None
        ssh_client = None
        for item in range(0, 20):
            # 重试20次
            logger.info(f"Retry check times: {item + 1}.")
            try:
                ssh_client = SSH_API.get_ssh_client(SSHClientInfo(host, ssh_user, ssh_pwd))
            except Exception as err:
                err_msg = f"{host} logged in failed: {err}"
                logger.error(err_msg)
                time.sleep(10)
                continue
            finally:
                SSH_API.close_ssh(ssh_client)
            logger.info(f"{host} logged in success.")
            err_msg = None
            break
        if err_msg is not None:
            raise Exception(err_msg)

    @classmethod
    def check_result_in_os(cls, host, sudo_user, sudo_pwd):
        try:
            return cls._do_check_result_in_os(host, sudo_pwd, sudo_user)
        except Exception as err:
            logger.error("Check result failed[{}].".format(str(err)))
            return False

    @classmethod
    def _do_check_result_in_os(cls, host, sudo_pwd, sudo_user):
        ssh_client = SSH_API.get_ssh_client(SSHClientInfo(host, sudo_user, sudo_pwd))
        cmd = "ls /home/ICUser/RDInstalled.xml && echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        logger.info(f"Execute cmd {cmd} on {host} return {result}.")
        if "CMD_RESULT=0" in str(result):
            return True
        return False

    @classmethod
    def check_os_password_expired_day(cls, auth_provider, user_lst):
        return_data = dict()
        sudo_client = SSH_API.get_sudo_ssh_client(auth_provider)
        for user in user_lst:
            cmd = f"chage -l {user}"
            output = SSH_API.exec_command_return_list(sudo_client, cmd)
            logger.info(f"Execute command success: {output}.")
            if not output or len(output) < SSH_RET_LENGTH:
                logger.error("Execute command did not get expect output.")
                raise Exception("Execute command did not get expect output.")
            code, expires, inactive = getPasswdExpires("\n".join(output))
            if code == ERROR_CODE:
                logger.error("Execute command did not get expect output.")
                raise Exception("Execute command did not get expect output.")
            elif code == EXCEPT_CODE:
                err_msg = "Analysis command output failed."
                logger.error(err_msg)
                raise HCCIException(663551, auth_provider.host, user, err_msg)
            elif code == COMMON_CODE:
                expire_days = (expires - datetime.datetime.now()).days
                logger.info(
                    f"The {user} password will expire in {expire_days} days.")
                if not expire_days:
                    raise Exception(
                        f"Failed to obtain the {user} expiration time.")
                return_data[user] = expire_days
        logger.info(f"Get users expire day return {return_data}.")
        return return_data

    @classmethod
    def init_os(cls, auth, time_zone):
        logger.info(f"Start init os system on {auth.host}.")
        script_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), "shell_tools", "os_init.sh")
        SSH_API.put_file(auth.host, auth.sudo_user, auth.sudo_pwd, script_file, Path.NODE_ROOT_PATH)
        # 判断一下当前管理面网络为IPV4还是IPV6,如果为ipv4就删除IPV6端口配置以减少初始化时间
        ip_version = 6 if ":" in auth.host else 4
        ssh_client = None
        try:
            ssh_client = SSH_API.get_ssh_client(SSHClientInfo(auth.host, auth.sudo_user, auth.sudo_pwd))
            result = SSH_API.send_command(
                ssh_client,
                f"dos2unix {Path.NODE_ROOT_PATH}/os_init.sh && "
                f"sh {Path.NODE_ROOT_PATH}/os_init.sh init {auth.host} {time_zone} {ip_version}",
                "init successfully", timeout=3600)
            logger.info(f"Start create user on {auth.host}.")
            SSH_API.send_command(
                ssh_client,
                f"sh {Path.NODE_ROOT_PATH}/os_init.sh createUser {auth.ssh_user} && "
                f"rm -f {Path.NODE_ROOT_PATH}/os_init.sh", "Please enter user password")
            SSH_API.send_command(ssh_client, auth.ssh_pwd, "Create user successfully")
        except Exception as err:
            logger.error(f"Init os system on {auth.host} failed: {err}")
            raise err
        finally:
            SSH_API.close_ssh(ssh_client)
        logger.info(f"Init os system on {auth.host} return {result}.")

    @classmethod
    def _check_system_root_pwd_changed(cls, host, sudo_user, new_sudo_pwd):
        ssh_client = None
        logger.info(f"Try to log in {host} with new password.")
        try:
            ssh_client = SSH_API.get_ssh_client(SSHClientInfo(host, sudo_user, new_sudo_pwd))
        except Exception as err:
            logger.warn(f"Log in {host} with new password failed: {err}")
            if "Authentication failed" in str(err):
                return False
            raise err
        finally:
            SSH_API.close_ssh(ssh_client)
        return True

    @classmethod
    def change_system_root_pwd(
            cls, host, sudo_user, old_sudo_pwd, new_sudo_pwd):
        logger.info("Begin to change system root passwd.")
        if cls._check_system_root_pwd_changed(host, sudo_user, new_sudo_pwd):
            logger.info(f"{host} sudo password already changed.")
            return
        ssh_client = SSH_API.get_ssh_client(SSHClientInfo(host, sudo_user, old_sudo_pwd))
        cmd = "passwd root"
        SSH_API.send_command(ssh_client, cmd, "New password")
        SSH_API.send_command(ssh_client, new_sudo_pwd, "Retype new password")
        SSH_API.send_command(ssh_client, new_sudo_pwd, "updated successfully")
        logger.info("Change system root passwd success.")

    @staticmethod
    def create_user(auth):
        script_file = os.path.join(os.path.dirname(
            os.path.dirname(__file__)), "shell_tools", "os_init.sh")
        SSH_API.put_file(
            auth.host, auth.sudo_user, auth.sudo_pwd, script_file, Path.NODE_ROOT_PATH)
        sudo_client = SSH_API.get_ssh_client(SSHClientInfo(auth.host, auth.sudo_user, auth.sudo_pwd))
        SSH_API.send_command(
            sudo_client,
            f"sh {Path.NODE_ROOT_PATH}/os_init.sh createUser {auth.ssh_user} && "
            f"rm -f {Path.NODE_ROOT_PATH}/os_init.sh", "Please enter user password")
        result = SSH_API.send_command(sudo_client, auth.ssh_pwd, "Create user successfully")
        SSH_API.close_ssh(sudo_client)
        logger.info(f"Create user on {auth.host} return {result}.")

    def enforce_os(self, user="DRManager", reset=True):
        logger.info(f"Start reinforce os system on {self.host}.")
        script_file = os.path.join(os.path.dirname(
            os.path.dirname(__file__)), "shell_tools", "reinforce_os.sh")
        SSH_API.put_file(self.host, self.sudo_user, self.sudo_pwd, script_file, Path.NODE_ROOT_PATH)
        result = SSH_API.send_command(
            self.ssh_client,
            f"dos2unix {Path.NODE_ROOT_PATH}/reinforce_os.sh && "
            f"sh {Path.NODE_ROOT_PATH}/reinforce_os.sh {user} "
            f"{reset} && rm -rf {Path.NODE_ROOT_PATH}/reinforce_os.sh",
            "reinforce successfully", timeout=300)
        logger.info(f"Reinforce os system on {self.host} return {result}.")

    def config_dns_and_ntp(self, dns_ips, ntp_ips, time_zone):
        self._config_dns(dns_ips)
        self._config_ntp(ntp_ips, time_zone)
        logger.info("Config dns and ntp successfully.")

    def _config_dns(self, dns_ips):
        logger.info("Start config dns.")
        dns_file = "/etc/resolv.conf"
        SSH_API.send_command(
            self.ssh_client, f"[ ! -f '{dns_file}' ] && touch {dns_file}")
        SSH_API.send_command(
            self.ssh_client,
            f"sed -i '/.*nameserver .*/d' {dns_file} && echo CMD_RESULT=$?",
            "CMD_RESULT=0")
        for host_ip in dns_ips:
            if not check_param_ip(host_ip):
                raise ValueError(f"Incorrect IP format: {host_ip}.")
            SSH_API.send_command(
                self.ssh_client,
                f"echo 'nameserver {host_ip}' >> {dns_file} && echo CMD_RESULT=$?",
                "CMD_RESULT=0")
        logger.info(f"Config dns on {self.host} success.")

    def _config_ntp(self, ntp_ips, time_zone):
        logger.info("Start config ntp.")
        time_zone_file = f"/usr/share/zoneinfo/{time_zone}"
        SSH_API.send_command(
            self.ssh_client,
            f"[ -f '{time_zone_file}' ] && "
            f"echo y | cp '{time_zone_file}' /etc/localtime")
        SSH_API.send_command(
            self.ssh_client,
            "cp /etc/ntp.conf /etc/ntp.conf.`date +'%s'` && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(
            self.ssh_client,
            "sed -i '/server.*maxpoll.*minpoll.*/d' /etc/ntp.conf && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        for index, ntp_ip in enumerate(ntp_ips):
            if not check_param_ip(ntp_ip):
                raise ValueError(f"Incorrect format: {ntp_ip}.")
            if index == 0:
                cmds = \
                    f"echo 'server {ntp_ip} maxpoll 4 minpoll 3 " \
                    "prefer' >> /etc/ntp.conf && echo CMD_RESULT=$?"
            else:
                cmds = \
                    f"echo 'server {ntp_ip} maxpoll 4 minpoll 3' " \
                    ">> /etc/ntp.conf && echo CMD_RESULT=$?"
            SSH_API.send_command(self.ssh_client, cmds, "CMD_RESULT=0")
        if len(ntp_ips) > 0:
            logger.info("Restart ntpd service.")
            SSH_API.send_command(
                self.ssh_client,
                "systemctl enable ntpd && service ntpd stop && "
                f"ntpdate {ntp_ips[0]} && service ntpd restart && "
                f"echo CMD_RESULT=$?", "CMD_RESULT=0")
        logger.info(f"Config ntp on {self.host} success.")

    def __del__(self):
        if self.ssh_client:
            SSH_API.close_ssh(self.ssh_client)
        logger.info(f"SSH channel is closed: {self.host}.")
