import paramiko
import json
import time
import sys
import os
import re
import traceback
import requests
from functools import wraps
from urllib.parse import quote
from collections import namedtuple, defaultdict

import utils.common.log as logger
from datetime import datetime, timedelta
from utils.business.param_util import ParamUtil
from utils.common.message import Message
from utils.security.crypt import decrypt
from platforms.param.ParamManager import ParamService
from plugins.AcceptanceTest.basic_mo_test.common.login_sc import LoginSC
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client

sys.path.append(os.path.realpath(__file__ + '/../../../../..'))
logger.init("CSBSLOGS")

CSBS_PROVIDER_ID = 'fc4d5750-22e7-4798-8a46-f48f62c4c1da'
KARBOR_USER = 'djmanager'
KARBOR_USER_PASSWD = 'wd4V/YyIpD+ykbz42ngl2JPvHq1duoLlz5MOhMwyLqYrh7+SaozWF03MSolZiDP2'
SERVER1_NAME = 'csbs_vm01'
VPC_NAME = 'csbs_vpc'
SUBNET_NAME = 'csbs_subnet'
KEYPAIR_NAME = 'csbs_keypair'
ECS_PRODUCT_TYPE = 'ecs'
EVS_PRODUCT_TYPE = 'evs'
VPC_PRODUCT_TYPE = 'vpc'
OPT_TYPE_ON = "start"
OPT_TYPE_OFF = "stop"


def get_region_info(project_id):
    RegionInfo = namedtuple('RegionInfo', ['region_id', 'region_type', 'region_name'])
    db_obj = ParamService()
    project_info = db_obj.get_project_info(project_id)
    region_id = project_info.get('region_id')
    region_type = project_info.get('region_type')
    region_name = LoginSC(project_id, region_id, region_type).get_region_name_and_az_name()[0]
    return RegionInfo(region_id, region_type, region_name)


def get_az_id(project_id, region_info):
    region_id = region_info.region_id
    region_type = region_info.region_type
    az_id = LoginSC(project_id, region_id, region_type).get_az_id()
    return az_id


def remove_sensitive_info(info):
    pattern = r'([a-zA-Z]{3}\d{6})|([a-zA-z]\d{8})'
    return re.sub(pattern, '****', str(info))


def get_dummy_vpc(vpc_list):
    for item in vpc_list:
        if item.get("name") == "dummy_external_network":
            logger.info("external network has dummy_external_network, "
                        "id=%s" % item["id"])
            return True
    return False


def retry(retry_nums=3):
    def decorator(fn):
        @wraps(fn)
        def iner_func(*args, **kwargs):
            retry_count = 0
            while retry_count <= retry_nums:
                try:
                    return fn(*args, **kwargs)
                except Exception as e:
                    retry_count += 1
                    time.sleep(5)
                    logger.error('Failed to run {}, errMsg:{},'
                                 ' start to retry: {}'.format(fn.__name__, str(e), retry_count))
                    continue
            else:
                raise Exception('Over retry error')

        return iner_func

    return decorator


class Login_SC(object):
    def __init__(self, project_id, region_id, region_type="type1"):
        # SC首页的headers，包括cftk,agencid,jsessionid等，在登录过程中更新
        self.bak_global_headers = {"Accept": "*/*",
                                   "X-Language": "zh-cn",
                                   "X-Requested-With": "XMLHttpRequest",
                                   "Connection": "keep-alive",
                                   "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 "
                                                 "(KHTML, like Gecko) Chrome/59.0.3071.104 Safari/537.36",
                                   "Content-Type": "application/json; charset=UTF-8"
                                   }
        # VPC服务的headers，包括cftk,agencid,jsessionid等，在登录过程中更新
        self.global_headers = {
            "Accept": "*/*",
            "X-Language": "zh-cn",
            "X-Requested-With": "XMLHttpRequest",
            "Connection": "keep-alive",
            "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) "
                          "Chrome/59.0.3071.104 Safari/537.36",
            "Content-Type": "application/json; charset=UTF-8"
        }
        self.pro_id = project_id
        self.region_id = region_id
        self.type = region_type
        self.get_test_params(project_id, region_id, region_type)

    def get_test_params(self, project_id, region_id, region_type):
        params = ParamUtil()
        self.console_host = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                              '%s_console_host'
                                                              % region_type, region_id)
        self.iam_host = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                          '%s_iam_host'
                                                          % region_type, region_id)
        self.manage_console_host = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                                     '%s_manage_console_host'
                                                                     % region_type, region_id)
        self.manage_iam_host = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                                 '%s_manage_iam_host'
                                                                 % region_type, region_id)

        self.is_b2b = False
        if self.manage_console_host != self.console_host:
            self.is_b2b = True

    @retry(retry_nums=3)
    def login_sc_on_web(self, console_host, iam_host, user, pwd, region_name):
        try:
            if self.is_b2b and 'bss_admin' in user:
                self.access_url = 'moserviceaccesswebsite'
                user_id, project_id, region_id, \
                self.bak_global_headers, self.global_headers = self.b2bAdmin_login_sc_on_web(console_host, iam_host,
                                                                                             user, pwd, region_name)
            else:
                self.access_url = 'motenantconsolehomewebsite'
                user_id, project_id, region_id, \
                self.bak_global_headers, self.global_headers = self.nonb2b_login_sc_on_web(console_host, iam_host,
                                                                                           user, pwd, region_name)
            return user_id, project_id, region_id, self.bak_global_headers, self.global_headers
        except Exception as e:
            logger.error('Login SC failed!')
            raise e

    def nonb2b_login_sc_on_web(self, console_host, iam_host, user, pwd, region_name):
        try:
            # tenantAuth接口
            tenantAuth_url = "https://{host}/rest/mologin/v3.0/tenantAuth".format(host=iam_host)
            basic_header = {
                "Content-Type": "application/x-www-form-urlencoded; charset=UTF-8",
                "Accept": "*/*"
            }
            quote_pwd = quote(pwd)
            request_body = "userName={username}&password={password}&" \
                           "verifyCode=&locale=zh-cn&service=https%253A%252F%252F{host}" \
                           "%252Fmotenantconsolehomewebsite%252F".format(username=user,
                                                                         password=quote_pwd,
                                                                         host=console_host)
            resp = requests.post(tenantAuth_url, data=request_body, verify=False, headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(response_body)
            if statu_code == 200:
                user = response_body.get("user")
                logger.info(user)
                id = user.get("id")
                logger.info(id)
                thirdLoginurl = response_body.get("url")
                logger.info(thirdLoginurl)
            else:
                raise Exception("login sc failed,/tenantAuth/ response is %s" % response_body)
            logger.info("***** /tenantAuth/ finish *****")
            # thirdLoginurl接口
            basic_header.update(
                {"Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8"})
            resp = requests.get(thirdLoginurl, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 302:
                thirdLogin_setcookie = resp.headers.get('set-cookie')
                ticket_location = resp.headers.get('Location')
            else:
                raise Exception("login sc failed,/thirdLoginurl/ statu_code is %s ,expect 302" % (statu_code))
            logger.info("***** /thirdLoginurl/ finish *****")
            # ticket接口
            resp = requests.get(ticket_location, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 302:
                ticket_setcookie = resp.headers.get('set-cookie')
            else:
                raise Exception("login sc failed,/ticket/ statu_code is %s ,expect 302" % (statu_code))
            logger.info("***** /ticket/ finish *****")
            # motenantconsolehomewebsite接口
            basic_header.update({"Cookie": ticket_setcookie})
            motenantconsolehomewebsite = "https://{host}/motenantconsolehomewebsite/".format(host=console_host)
            resp = requests.get(motenantconsolehomewebsite, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            if statu_code != 200:
                raise Exception(
                    "login sc failed,/motenantconsolehomewebsite/ statu_code is %s ,expect 200" % (statu_code))
            # console me接口
            basic_header.update({
                "Cookie": ticket_setcookie,
                "Accept": "*/*",
                "X-Language": "zh-cn",
                "X-Requested-With": "XMLHttpRequest",
                "Content-Type": "application/json; charset=UTF-8",
                "Connection": "keep-alive",
                "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) "
                              "Chrome/59.0.3071.104 Safari/537.36",
            })
            me_url = "https://{host}/motenantconsolehomewebsite/rest" \
                     "/me?salt=0.4070987459363351".format(host=console_host)
            resp = requests.get(me_url, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(response_body)
            if statu_code == 200:
                me_cftk = response_body.get('cftk')
                project_id = response_body.get('projectId')
                user_id = response_body.get('userId')
                region = response_body.get('region')
                aggencyID = response_body.get('id')
            else:
                raise Exception("login sc failed,/rest/me response is %s" % response_body)
            logger.info("***** /rest/me finish *****")
            basic_header.update({"cftk": me_cftk})
            self.bak_global_headers[
                "Cookie"] = ticket_setcookie
            self.bak_global_headers["cftk"] = me_cftk
            self.bak_global_headers["ProjectName"] = region
            logger.info(self.bak_global_headers)
            cloud_infras = self.get_cloud_infra_and_regionid_by_regionname(console_host, region_name)
            logger.info(cloud_infras)
            self.bak_global_headers["region"] = cloud_infras[0][0]
            region_id = cloud_infras[0][0]
            logger.info(self.bak_global_headers)
            # project CSDR切换project
            projects_url = "https://{host}/motenantconsolehomewebsite/rest/vdc/v3.1/users/" \
                           "{agencyid}/projects?start=0&limit=100&rel_projects=true".format(host=console_host,
                                                                                            agencyid=aggencyID)
            resp = requests.get(projects_url, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 200 and resp.content:
                response_body = json.loads(resp.content)
                logger.info(response_body)
                projects = response_body.get('projects')
                for project in projects:
                    region_temp = project.get('regions')
                    region_temp_id = region_temp[0].get('region_id')
                    if region_temp_id == region_id:
                        project_id = project.get('id')
                        region = project.get('name')
                        logger.info("***** reset project_id and project_name success")
            # vpc login接口：
            vpc_login_url = "https://{host1}/authui/login?service=https%3A%2F%2F{host}" \
                            "%2Fvpc%2F%3FagencyId%3D{agencyid}%26region%3D" \
                            "{Region}%26locale%3Dzh-cn".format(host1=iam_host, host=console_host,
                                                               agencyid=aggencyID, Region=region)
            basic_header.update({"Cookie": thirdLogin_setcookie})
            resp = requests.get(vpc_login_url, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            logger.info(resp.headers)
            if statu_code == 302:
                vpc_location = resp.headers.get('Location')
            else:
                raise Exception("login sc failed,/vpc authui/login/ statu_code is %s ,expect 302" % (statu_code))
            logger.info("***** /vpc authui/login/ finish *****")
            # vpc_Jsession获取
            basic_header.update({"Cookie": ticket_setcookie})
            resp = requests.get(vpc_location, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            logger.info(statu_code)
            if statu_code == 302:
                vpc_setcookie = resp.headers.get('set-cookie')
                vpc_J_SESSION_ID_temp1 = vpc_setcookie.split('J_SESSION_ID=')
                vpc_J_SESSION_ID_temp2 = vpc_J_SESSION_ID_temp1[1].split('; Version=1; Path=/; Secure;')
                vpc_J_SESSION_ID = vpc_J_SESSION_ID_temp2[0]
            else:
                raise Exception("login sc failed,/vpc_Jsession/ statu_code is %s ,expect 302" % (statu_code))
            logger.info("***** /vpc_Jsession/ finish *****")
            # vpc me接口
            vpc_me_cookie = ticket_setcookie + ';J_SESSION_ID=' + vpc_J_SESSION_ID
            basic_header.update({
                "Cookie": vpc_me_cookie,
                "AgencyId": aggencyID})
            vpc_me_url = "https://{host}/vpc/rest/me?salt=0.8809960674639905".format(
                host=console_host)
            resp = requests.get(vpc_me_url, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(response_body)
            if statu_code == 200:
                vpc_me_cftk = response_body.get('cftk')
            else:
                raise Exception("login sc failed,/vpc/rest/me response is %s" % response_body)
            logger.info("***** /vpc/rest/me finish *****")
            self.global_headers[
                "Cookie"] = vpc_me_cookie
            self.global_headers["cftk"] = vpc_me_cftk
            self.global_headers["AgencyId"] = aggencyID
            self.global_headers["ProjectName"] = region
            self.global_headers["region"] = region_id
            logger.info(self.global_headers)
            logger.info("***** login sc finish *****")
            return user_id, project_id, region_id, self.bak_global_headers, self.global_headers
        except Exception as e:
            logger.error('Login SC failed!')
            raise e

    def b2bAdmin_login_sc_on_web(self, console_host, iam_host, user, pwd, region_name):
        try:
            # tenantAuth接口
            tenantAuth_url = "https://{host}/rest/mologin/v3.0/tenantAuth".format(host=iam_host)
            basic_header = {
                "Content-Type": "application/x-www-form-urlencoded; charset=UTF-8",
                "Accept": "*/*"
            }
            quote_pwd = quote(pwd)
            request_body = "userName={username}&password={password}&verifyCode=&" \
                           "locale=zh-cn&service=https%253A%252F%252F" \
                           "{host}%252Fmoserviceaccesswebsite%252F".format(username=user,
                                                                           password=quote_pwd,
                                                                           host=console_host)
            resp = requests.post(tenantAuth_url, data=request_body, verify=False, headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(response_body)
            if statu_code == 200:
                user = response_body.get("user")
                logger.info(user)
                id = user.get("id")
                logger.info(id)
                thirdLoginurl = response_body.get("url")
                logger.info(thirdLoginurl)
            else:
                raise Exception("login sc failed,/tenantAuth/ response is %s" % response_body)
            logger.info("***** /tenantAuth/ finish *****")
            # thirdLoginurl接口
            basic_header.update(
                {"Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8"})
            resp = requests.get(thirdLoginurl, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 302:
                thirdLogin_setcookie = resp.headers.get('set-cookie')
                ticket_location = resp.headers.get('Location')
            else:
                raise Exception("login sc failed,/thirdLoginurl/ statu_code is %s ,expect 302" % (statu_code))
            logger.info("***** /thirdLoginurl/ finish *****")
            # ticket接口
            resp = requests.get(ticket_location, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 302:
                ticket_setcookie = resp.headers.get('set-cookie')
                aggencyID_temp1 = ticket_setcookie.split('agencyID=')
                aggencyID_temp2 = aggencyID_temp1[1].split(';Path=/;Secure, J_SESSION_ID=')
                aggencyID = aggencyID_temp2[0]
                J_SESSION_ID_temp1 = aggencyID_temp2[1].split(';Path=/;Secure;HttpOnly')
                J_SESSION_ID = J_SESSION_ID_temp1[0]
            else:
                raise Exception("login sc failed,/ticket/ statu_code is %s ,expect 302" % (statu_code))
            logger.info("***** /ticket/ finish *****")
            # console me接口
            basic_header.update({
                "Cookie": "theme=default; SID=Set2; agencyID={agencyid}; J_SESSION_ID={j_session_id}; "
                          "locale=zh-cn; browserCheckResult=A".format(agencyid=aggencyID,
                                                                      j_session_id=J_SESSION_ID),
                "Accept": "*/*",
                "X-Language": "zh-cn",
                "X-Requested-With": "XMLHttpRequest",
                "Connection": "keep-alive",
                "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) "
                              "Chrome/59.0.3071.104 Safari/537.36",
            })
            me_url = "https://{host}/moserviceaccesswebsite/rest/me?salt=0.4070987459363351".format(host=console_host)
            resp = requests.get(me_url, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(response_body)
            if statu_code == 200:
                me_cftk = response_body.get('cftk')
                project_id = response_body.get('projectId')
                user_id = response_body.get('userId')
                region = response_body.get('region')
            else:
                raise Exception("login sc failed,/rest/me response is %s" % response_body)
            logger.info("***** /rest/me finish *****")
            basic_header.update({"cftk": me_cftk})
            self.bak_global_headers[
                "Cookie"] = "theme=default; SID=Set2; agencyID={agencyid}; J_SESSION_ID={j_session_id}; " \
                            "locale=zh-cn; browserCheckResult=A".format(agencyid=aggencyID,
                                                                        j_session_id=J_SESSION_ID)
            self.bak_global_headers["cftk"] = me_cftk
            self.bak_global_headers["AgencyId"] = aggencyID
            self.bak_global_headers["ProjectName"] = region
            logger.info(self.bak_global_headers)
            cloud_infras = self.get_cloud_infra_and_regionid_by_regionname(console_host, region_name)
            logger.info(cloud_infras)
            self.bak_global_headers["region"] = cloud_infras[0][0]
            region_id = cloud_infras[0][0]
            logger.info(self.bak_global_headers)
            # vpc login接口：
            vpc_login_url = "https://{host1}/authui/login?service=https%3A%2F%2F{host}" \
                            "%2Fvpc%2F%3FagencyId%3D{agencyid}%" \
                            "26region%3D{Region}%26locale%3Dzh-cn".format(host1=iam_host, host=console_host,
                                                                          agencyid=aggencyID, Region=region)
            basic_header.update({"Cookie": thirdLogin_setcookie})
            resp = requests.get(vpc_login_url, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            logger.info(resp.headers)
            if statu_code == 302:
                vpc_location = resp.headers.get('Location')
            else:
                raise Exception("login sc failed,/vpc authui/login/ statu_code is %s ,expect 302" % (statu_code))
            logger.info("***** /vpc authui/login/ finish *****")
            # vpc_Jsession获取
            resp = requests.get(vpc_location, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            logger.info(statu_code)
            if statu_code == 302:
                vpc_setcookie = resp.headers.get('set-cookie')
                vpc_J_SESSION_ID_temp1 = vpc_setcookie.split('Path=/; Secure, J_SESSION_ID="')
                vpc_J_SESSION_ID_temp2 = vpc_J_SESSION_ID_temp1[1].split(';Path=/;Secure;HttpOnly')
                vpc_J_SESSION_ID = vpc_J_SESSION_ID_temp2[0]
            else:
                raise Exception("login sc failed,/vpc_Jsession/ statu_code is %s ,expect 302" % (statu_code))
            logger.info("***** /vpc_Jsession/ finish *****")
            # vpc me接口
            # vpc 接口的jsessionid需要加双引号
            vpc_J_SESSION_ID_format = '"' + vpc_J_SESSION_ID + '"'
            basic_header.update({
                "Cookie": "theme=default; SID=Set2; agencyID={agencyid}; J_SESSION_ID={j_session_id}; "
                          "locale=zh-cn; browserCheckResult=A".format(agencyid=aggencyID,
                                                                      j_session_id=vpc_J_SESSION_ID_format),
                "AgencyId": aggencyID})
            vpc_me_url = "https://{host}/vpc/rest/me?salt=0.8809960674639905".format(host=console_host)
            resp = requests.get(vpc_me_url, verify=False, allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(response_body)
            if statu_code == 200:
                vpc_me_cftk = response_body.get('cftk')
            else:
                raise Exception("login sc failed,/vpc/rest/me response is %s" % response_body)
            logger.info("***** /vpc/rest/me finish *****")
            self.global_headers[
                "Cookie"] = "theme=default; SID=Set2; agencyID={agencyid}; J_SESSION_ID={j_session_id}; " \
                            "locale=zh-cn; browserCheckResult=A".format(agencyid=aggencyID,
                                                                        j_session_id=vpc_J_SESSION_ID_format)
            self.global_headers["cftk"] = vpc_me_cftk
            self.global_headers["AgencyId"] = aggencyID
            self.global_headers["ProjectName"] = region
            self.global_headers["region"] = region_id
            logger.info(self.global_headers)
            logger.info("***** login sc finish *****")
            return user_id, project_id, region_id, self.bak_global_headers, self.global_headers
        except Exception as e:
            logger.error('Login SC failed!')
            raise e

    def get_cloud_infra_and_regionid_by_regionname(self, console_host, region_name="ALL"):
        # 根据region名称获取region_id,cloud_infra_id,cloud_infra_name
        try:
            region_name_filed = "" if region_name == "ALL" else region_name
            url = "https://{host}/{access_url}/goku/" \
                  "rest/serviceaccess/v3.0/regions?name={region}".format(host=console_host,
                                                                         access_url=self.access_url,
                                                                         region=region_name_filed)
            resp = requests.get(url, verify=False, allow_redirects=False, headers=self.bak_global_headers)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(response_body)
            if statu_code == 200:
                records = response_body.get("records")
                cloud_infra_list = []
                for region_info in records:
                    name = region_info["name"]
                    if region_name == name:
                        cloud_infras = region_info["cloud_infras"]
                        for cloud_infra in cloud_infras:
                            cloud_infra_temp = []
                            if cloud_infra["type"] == "FUSION_CLOUD":
                                region_id = cloud_infra["region_id"]
                                cloud_infra_id = cloud_infra["id"]
                                cloud_infra_name = cloud_infra["name"]
                                cloud_infra_temp.append(region_id)
                                cloud_infra_temp.append(cloud_infra_id)
                                cloud_infra_temp.append(cloud_infra_name)
                                cloud_infra_list.append(cloud_infra_temp)
                                return cloud_infra_list
                        else:
                            return None
                    elif region_name == "ALL":
                        cloud_infras = region_info["cloud_infras"]
                        for cloud_infra in cloud_infras:
                            cloud_infra_temp = []
                            if cloud_infra["type"] == "FUSION_CLOUD":
                                region_id = cloud_infra["region_id"]
                                cloud_infra_id = cloud_infra["id"]
                                cloud_infra_name = cloud_infra["name"]
                                cloud_infra_temp.append(region_id)
                                cloud_infra_temp.append(cloud_infra_id)
                                cloud_infra_temp.append(cloud_infra_name)
                                cloud_infra_list.append(cloud_infra_temp)
                return cloud_infra_list
            else:
                raise Exception("get cloud infra list failed! response is %s" % response_body)
        except Exception as e:
            logger.error('Get cloud infra list failed!')
            raise e


class CSBSBaseAWAdapter(object):
    """
    CSBS 基础AW
    """

    def __init__(self, project_id):
        console = CSBSBaseConsoleAdapter(project_id)
        console.login_sc()
        self.region_type = console.region_type
        self.region_id = console.region
        self.az = console.az_id
        self.region_name = console.region_name
        self.project_id = console.project_id
        self.project_name = console.project_name
        self.product_name = console.product_name
        self.server1_name = console.server1_name
        self.server1_id = console.server1_id
        self.server1_system_volume_name = console.server1_system_volume_name
        self.server1_system_volume_id = console.server1_system_volume_id
        self.console = console
        params = ParamUtil()
        self.headers = {"Content-Type": "application/json;charset=UTF-8"}
        self.openstack_api_version = "v2"
        self.iam_version = "v3"
        self.session = requests.Session()
        self.session.trust_env = False
        self.session.proxies = None
        self.session.verify = False
        self.token = None
        self.auth_domain = "https://" + params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                                          '%s_iam_host' % self.region_type,
                                                                          self.region_id)
        self.karbor_user = params.get_value_from_cloud_param(project_id, 'CSBS-VBS',
                                                             'karbor_djmanager_username',
                                                             self.region_id) or KARBOR_USER
        self.karbor_user_passwd = params.get_value_from_cloud_param(project_id,
                                                                    'CSBS-VBS',
                                                                    'karbor_djmanager_pwd',
                                                                    self.region_id) or decrypt(KARBOR_USER_PASSWD)
        self.karbor_root_passwd = params.get_value_from_cloud_param(project_id,
                                                                    'CSBS-VBS',
                                                                    'karbor_root_pwd',
                                                                    self.region_id)
        self.set_karbor_ips(project_id)
        self.port = "8799"
        self.userName = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                          '%s_vdc1_vsm_user1' % self.region_type,
                                                          self.region_id)
        self.userPwd = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                         '%s_sc_user_new_password' % self.region_type,
                                                         self.region_id)
        self.tenantName = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                            '%s_vdc1_name' % self.region_type,
                                                            self.region_id)
        self.cascading_region = params.get_value_from_cloud_param(project_id, "OpenStack",
                                                                  "openstack_region",
                                                                  self.region_id)
        self.cascading_domain = params.get_value_from_cloud_param(project_id, "OpenStack",
                                                                  "openstack_domain",
                                                                  self.region_id)
        self.nova_url = "https://compute." + self.cascading_region + "." + self.cascading_domain + ":443"
        self.cinder_url = "https://volume." + self.cascading_region + "." + self.cascading_domain + ":443"
        temp_host = '[' + self.host + ']' if self.host.__contains__(':') else self.host
        self.base_url = "https://{host}:{port}".format(host=temp_host, port=self.port)
        self.provider_id = params.get_value_from_cloud_param(project_id, 'CSBS-VBS',
                                                             'csbs_provider_id',
                                                             self.region_id) or CSBS_PROVIDER_ID

        self.KarBor_Info = [self.karbor_host_ip, self.karbor_user, self.karbor_user_passwd]
        self.product_id = None
        self.karbor_timezone = None

    def set_karbor_ips(self, project_id):
        params = Params(project_id, project_id, 'CSBS-VBS', self.region_id)
        karbor1_ip = params.get_vm_ips()['karbor_host0']
        logger.info('Ssh to karbor node: {}'.format(karbor1_ip))
        node = karbor_client.CBSNode(karbor1_ip, self.karbor_user,
                                     self.karbor_user_passwd, self.karbor_root_passwd)
        ssh_client = karbor_client.get_sshclient_user_su_root(node)
        cmds = 'get_info.py --tenant_float_ip'
        logger.info('Cmd: {}'.format(cmds))
        result = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        if not karbor_client.is_ssh_cmd_executed(result):
            msg = "Failed to execute cmd: {}".format(cmds)
            logger.error(msg)
            raise Exception(msg)
        logger.info('Get karbor float ip info: {}'.format(result))
        karbor_float_ip = result[0].strip()
        self.host = karbor_float_ip
        self.karbor_host_ip = karbor1_ip

    def close_session(self):
        """关闭session

        返 回 值:None。

        """
        if self.session is None:
            logger.info("Currently no Session can be closed")
        else:
            self.session.close()
            logger.info("Currently Session has successfully closed")

    def _pre_process(self, kwargs):
        """HTTP请求提交之前的预处理，主要是设置headers"""
        if self.headers is None:
            raise Exception("Please Login first.")
        return kwargs.update({"headers": self.headers})

    @staticmethod
    def _post_process(response):
        """对HTTP响应进行处理"""
        if response.ok:
            if len(response.text) > 0:
                ret = json.loads(response.text)
            else:
                ret = response.text
            return ret
        else:
            if "errorCode" in response.text:
                text_dict = json.loads(response.text)
                error_code = text_dict.get('errorCode')
                logger.info(error_code)
                raise Exception("Failed requesting, error code:{i};response:{j}".format(i=error_code, j=text_dict))
            else:
                error_msg = response.text
                logger.info(error_msg)
                raise Exception("Failed requesting, response: {j}".format(j=error_msg))

    def _get(self, url, **kwargs):
        self._pre_process(kwargs)
        response = self.session.get(url, **kwargs)
        resp_body = self._post_process(response)
        return resp_body

    def _post(self, url, data=None, json=None, **kwargs):
        """submit a request with POST method."""
        self._pre_process(kwargs)
        response = self.session.post(url, data=data, json=json, **kwargs)
        return self._post_process(response)

    def _put(self, url, data, **kwargs):
        """submit a request with PUT method."""
        self._pre_process(kwargs)
        response = self.session.put(url, data, **kwargs)
        return self._post_process(response)

    def _delete(self, url, **kwargs):
        """submit a request with DELETE method."""
        self._pre_process(kwargs)
        response = self.session.delete(url, **kwargs)
        return self._post_process(response)

    def login_iam(self):
        iam_url = "%s/%s/auth/tokens" % (self.auth_domain, self.iam_version)
        params = '{"auth": { "identity": { "methods": [ "password" ], "password": {  "user": {'
        params = params + '"name":"' + self.userName + '",'
        params = params + '"password":"' + self.userPwd + '",'
        params = params + '"domain":{' + '"name":"' + self.tenantName + '"}}}}' + ","
        if self.project_name is not None:
            # 不推荐project Name来认证是因为传参的project Name并不是SC界面上显示的，而是“STD_{region ID}_名称”.
            projectName = self.project_name
            params = params + '"scope":{' + '"project":{' + '"name":"' + projectName + '"}}' + ","
        elif self.project_id is not None:
            # 推荐方式
            params = params + '"scope":{' + '"project":{' + '"id":"' + self.project_id + '"}}' + ","
        else:
            # 某些接口需要指定project的权限的token才可以调用（VDC）
            params = params + '"scope":{' + '"domain":{' + '"name":"' + self.tenantName + '"}}' + ","
        params = params.rstrip(',')
        params = params + '}}'

        ret = self.session.request(method="post", url=iam_url, headers=self.headers, data=params)
        self.token = ret.headers['X-Subject-Token']
        logger.debug(self.token)
        self.headers['X-Auth-Token'] = ret.headers['X-Subject-Token']
        res = json.loads(ret.text)
        # 获取product_id
        self.product_id = self.get_product_id_from_name(self.product_name)
        return ret.status_code, res

    def Structure_params_quota(self, type, number):
        """构造申请配额空间的params参数

        :param type:要么是backup_capacity(备份容量)，要么是copy_capacity（复制容量）
        :param number:数量，MB为单位
        :return:
        """
        c = None
        if type == "backup_capacity":
            c = {"count": 1,
                 "param_items": [{"type": "backup_capacity", "quota": number}],
                 "display": {"en_US": [{"label": "Backup", "type": "string", "value": "xx"}],
                             "zh_CN": [{"label": "备份", "type": "string", "value": "xx"}]}
                 }
        elif type == "copy_capacity":
            c = {"count": 1,
                 "param_items": [{"type": "copy_capacity", "quota": number}],
                 "display": {"en_US": [{"label": "Replication", "type": "string", "value": "xx"}],
                             "zh_CN": [{"label": "复制", "type": "string", "value": "xx"}]}
                 }
        m = json.dumps(c)
        return m

    @retry(retry_nums=3)
    def modify_quota(self, service_type, params):
        """修改配额容量

        :param project_id:
        :param service_type:
        :param region_id:
        :param product_id:
        :return:
        """
        url = "%s/v1/{project_id}/products/{product_id}/action".format(project_id=self.project_id,
                                                                       product_id=self.product_id) % self.base_url
        body = {
            "apply": {
                "service_type": service_type,
                "region_id": self.region_id,
                "params": params
            }
        }

        return self._post(url, json=body)

    def ADD_QUOTA(self, quota_num):
        logger.info("Start increasing backup space")
        param = self.Structure_params_quota("backup_capacity", quota_num)
        self.modify_quota("csbs", param)
        logger.info("Succeed increasing backup space")

    def Mu_QUOTA(self, quota_num):
        logger.info("Start decreasing backup space")
        param = self.Structure_params_quota("backup_capacity", -quota_num)
        self.modify_quota("csbs", param)
        logger.info("Succeed decreasing backup space")

    def ADD_QUOTA_copy(self, quota_num):
        logger.info("Start increasing replication space")
        param = self.Structure_params_quota("copy_capacity", quota_num)
        self.modify_quota("csbs", param)
        logger.info("Succeed increasing replication space")

    def Mu_QUOTA_copy(self, quota_num):
        logger.info("Start decreasing replication space")
        param = self.Structure_params_quota("copy_capacity", -quota_num)
        self.modify_quota("csbs", param)
        logger.info("Succeed decreasing replication space")

    def show_region_list(self):
        """获取region信息

        :return:
        """
        url = "{auth_url}/silvan/rest/v1.0/regions".format(auth_url=self.auth_domain)
        ret = self.session.get(url=url)

        logger.info(ret.status_code)
        res = json.loads(ret.text)
        return ret.status_code, res

    def show_product_list(self):
        """获取产品信息

        :param user_id:
        :param service_type:
        :return:
        """
        # 展示可申请的产品列表
        url = "%s/rest/product/v3.0/apply/products?service_type={service_type}" \
              "&start=1&limit=20&region_id={region_id}".format(service_type="csbs",
                                                               region_id=self.region_id) % self.auth_domain

        ret = self.session.get(url=url, headers=self.headers)

        logger.info(ret.status_code)
        res = json.loads(ret.text)
        return ret.status_code, res

    def show_volume(self, volume_id):
        """查询单独卷信息

        输入参数:
        - tenant_id: 租户id
        - volume_id: 卷id

        返 回 值:查询卷得详细信息
        """
        url = "%s/v2/{tenant_id}/volumes/{volume_id}".format(tenant_id=self.project_id,
                                                             volume_id=volume_id) % self.cinder_url
        return self._get(url)

    def list_servers(self):
        """查询当前租户的所有虚拟机
        """
        url = "%s/v2/{tenant_id}/servers".format(tenant_id=self.project_id) % self.nova_url
        return self._get(url)

    def get_all_servers(self):
        """获取指定租户所有虚拟机的名称及id
        """
        servers_info = self.list_servers()
        servers_dict = {}
        for server in servers_info["servers"]:
            servers_dict[server["name"]] = server["id"]
        return servers_dict

    def list_all_volumes(self):
        """获取租户下所有卷
        """
        url = "%s/v2/{tenant_id}/volumes".format(tenant_id=self.project_id) % self.cinder_url
        return self._get(url)

    def list_attached_volumes(self, server_id):
        """查询虚拟机上所挂卷的信息
        """
        url = "%s/v2/{tenant_id}/servers/{server_id}/os-volume_attachments".format(tenant_id=self.project_id,
                                                                                   server_id=server_id) % self.nova_url
        return self._get(url)

    def create_plan_include_backupandcopy_auto(self, plan_name, resources,
                                               max_backups_backup=10, max_backups_copy=10,
                                               Scheduling_name_backup="", Scheduling_name_copy="",
                                               is_exe_policy=False, expired_time="",
                                               trigger_pattern_backup="", trigger_pattern_copy="",
                                               is_enabled_backup=False, is_enabled_copy=False,
                                               operation_type=["backup", "copy"]):
        """创建包含备份和复制策略的计划
        此方法更加完善，可以对备份和复制策略分别设置不同的调度周期
        operation_type: backup 增量备份
                        copy 复制
                        full_backup 全量
        :param project_id:
        :param provider_id:
        :param plan_name:
        :param resources:
        :param max_backups_backup:
        :param max_backups_copy:
        :param Scheduling_name_backup:
        :param Scheduling_name_copy:
        :param is_exe_policy:
        :param expired_time:
        :param trigger_pattern_backup:
        :param trigger_pattern_copy:
        :param is_enabled_backup:
        :param is_enabled_copy:
        :return:


        """

        url = "%s/v1/{project_id}/policies".format(project_id=self.project_id) % self.base_url
        if is_exe_policy:
            body_dict = {
                "policy": {
                    "name": plan_name,
                    "description": "",
                    "provider_id": self.provider_id,
                    "parameters": {
                        "common": {
                            "az": self.az,
                            "expired_at": expired_time
                        }
                    },
                    "scheduled_operations": [
                        {
                            "name": Scheduling_name_backup,
                            "description": "",
                            "enabled": is_enabled_backup,
                            "operation_definition": {
                                "max_backups": max_backups_backup
                            },
                            "trigger": {
                                "properties": {
                                    "pattern": trigger_pattern_backup
                                }
                            },
                            "operation_type": operation_type[0]
                        }, {
                            "name": Scheduling_name_copy,
                            "description": "",
                            "enabled": is_enabled_copy,
                            "operation_definition": {
                                "max_backups": max_backups_copy
                            },
                            "trigger": {
                                "properties": {
                                    "pattern": trigger_pattern_copy
                                }
                            },
                            "operation_type": operation_type[1]
                        }
                    ],
                    "resources": resources
                }
            }
        else:
            body_dict = {
                "policy": {
                    "name": plan_name,
                    "description": "my plan",
                    "provider_id": self.provider_id,
                    "parameters": {},
                    "resources": resources
                }
            }
        logger.debug(body_dict)
        return self._post(url, json=body_dict)

    def change_backupplan(self, policy_id, modify_body):

        """修改备份计划

        输入参数:

        - project_id:租户ID
        - policy_id:备份计划ID
        - modify_body:修改的内容结构体

        异常描述:返回错误码
        使用方法及范例:
        修改历史:

        """
        url = "%s/v1/{project_id}/policies/{policy_id}".format(project_id=self.project_id,
                                                               policy_id=policy_id) % self.base_url
        body_dict = json.dumps(modify_body)
        return self._put(url, body_dict)

    def delete_image(self, checkpoint_id):
        """删除副本

        输入参数:
        - project_id:
        - provider_id:
        - checkpoint_id:
        返 回 值:状态码200 ok
        异常描述:参数异常
        使用方法及范例:
        修改历史:

        """
        url = "%s/v1/{project_id}/providers/{provider_id}/" \
              "checkpoints/{checkpoint_id}".format(project_id=self.project_id,
                                                   provider_id=self.provider_id,
                                                   checkpoint_id=checkpoint_id) % self.base_url
        return self._delete(url)

    def backup(self, policy_id, resources, auto_trigger=False, incremental=True):
        """执行备份
        输入参数：
        - project_id:租户id
        - provider_id:提供商id
        - policy_id:备份计划id
        - resources:类型：列表，需要备份的内容id列表
        - auto_trigger：True或者False
        - incremental:周期性全备参数，True为不周期性全备，False为周期性全备

        使用方法及范例:
        修改历史:

        """

        url = "%s/v1/{project_id}/providers/{provider_id}/" \
              "checkpoints".format(project_id=self.project_id,
                                   provider_id=self.provider_id) % self.base_url
        body_dict = {
            "checkpoint": {
                "plan_id": policy_id,
                "parameters": {
                    "auto_trigger": auto_trigger,
                    "resources": resources,
                    "incremental": incremental
                }
            }
        }
        return self._post(url, json=body_dict)

    def copy_backupimage(self, policy_id, auto_trigger=False):
        """拷贝备份副本

        输入参数:
        - project_id:租户id
        - provider_id:提供商id
        - policy_id:备份计划id
        - auto_trigger:true or false


        异常描述:返回错误码

        使用方法及范例:



        """
        url = "%s/v1/{project_id}/providers/{provider_id}/" \
              "checkpoints/action".format(project_id=self.project_id,
                                          provider_id=self.provider_id) % self.base_url

        body_dict = {
            "copy": {
                "policy_id": policy_id,
                "auto_trigger": auto_trigger
            }
        }

        return self._post(url, json=body_dict)

    def restore(self, checkpoint_id, checkpoint_item_id, targets):
        """恢复备份副本

        输入参数:

        - project_id：
        - provider_id：
        - checkpoint_id：
        - checkpoint_item_id：
        - server_id：
        - volume_list：恢复的卷列表
        | "volumes": [{
        |                     "backup_id": backup_id,
        |                     "volume_id": volume_id
        |                 }，……]

        返 回 值:返回成功

        异常描述:返回错误码

        使用方法及范例:


        """
        url = "%s/v1/{project_id}/restores".format(project_id=self.project_id) % self.base_url

        body_dict = {
            "restore": {
                "provider_id": self.provider_id,
                "checkpoint_id": checkpoint_id,
                "parameters": {
                    "checkpoint_item_id": checkpoint_item_id,
                    "power_on": True,
                    "targets": targets
                }
            }
        }
        return self._post(url, json=body_dict)

    def delete_backupplan(self, policy_id):
        """删除备份计划

        输入参数:
        - poject_id:租户id
        - policy_id:备份计划id

        返 回 值:返回成功

        异常描述:返回错误码

        使用方法及范例:



        """
        url = "%s/v1/{project_id}/policies/{policy_id}".format(project_id=self.project_id,
                                                               policy_id=policy_id) % self.base_url
        return self._delete(url)

    def get_image_detail(self, checkpoint_item_id):
        """获取拷贝副本信息

        输入参数:

        - project_id：
        - checkpoint_item_id:备份副本的id，可以从执行备份的返回值中获取


        异常描述:返回错误码

        使用方法及范例:



        """
        url = "%s/v1/{project_id}/checkpoint_items/" \
              "{checkpoint_item_id}".format(project_id=self.project_id,
                                            checkpoint_item_id=checkpoint_item_id) % self.base_url
        return self._get(url)

    def get_image_list(self):
        """获取拷贝副本列表

        输入参数:

        - project_id:租户id

        返 回 值:

        异常描述:返回错误码

        使用方法及范例:


        """
        url = "%s/v1/{project_id}/checkpoint_items".format(project_id=self.project_id) % self.base_url

        return self._get(url)

    def SBS_Create_Plan_Incremental_Include_TrueAndFalse_auto(self, source_list, plan_name,
                                                              max_backups_true=20, max_backups_false=20,
                                                              Scheduling_name_true="Scheduling_name_backup",
                                                              Scheduling_name_false="Scheduling_name_copy",
                                                              expired_time=5,
                                                              pattern_FREQ_TRUE="WEEKLY", pattern_FREQ_FALSE="WEEKLY",
                                                              pattern_BYDAY_TRUE="MO", pattern_BYDAY_FALSE="MO",
                                                              pattern_BYHOUR_TRUE="10", pattern_BYHOUR_FALSE="10",
                                                              pattern_BYMINUT_TRUE="20", pattern_BYMINUT_FALSE="20",
                                                              is_enable_true=True, is_enable_false=True):
        """创建包含增量备份策略和全备策略的计划
        PS:该方法更加完善，可以

        :param tenant_id:租户ID
        :param provider_id:供应商ID
        :param vmname:虚拟机名称
        :param plan_name:计划名称
        :param max_backups_backup:备份策略最大副本数
        :param max_backups_copy:复制策略最大副本数
        :param Scheduling_name_backup:备份策略的名字
        :param Scheduling_name_copy:复制策略的名字
        :param expired_time:
        :param pattern_FREQ_backup:
        :param pattern_FREQ_copy:
        :param pattern_BYDAY_backup:
        :param pattern_BYDAY_copy:
        :param pattern_BYHOUR_backup:
        :param pattern_BYHOUR_copy:
        :param pattern_BYMINUT_backup:
        :param pattern_BYMINUT_copy:
        :param is_enable_backup:
        :param is_enable_copy:
        :param vmtype:
        :return:

        """
        # 构造计划的过期时间
        dt = datetime.now() + timedelta(expired_time)
        expired_time_str = dt.strftime("%Y-%m-%dT%H:%M:%S")
        # 构造备份执行侧策略：
        trigger_pattern_true = "BEGIN:VCALENDAR\r\nBEGIN:VEVENT\r\nRRULE:FREQ={FREQ};BYDAY={BYDAY};" \
                               "BYHOUR={BYHOUR};BYMINUTE={BYMINUTE}\r\nEND:VEVENT\r\n" \
                               "END:VCALENDAR\r\n".format(FREQ=pattern_FREQ_TRUE, BYDAY=pattern_BYDAY_TRUE,
                                                          BYHOUR=pattern_BYHOUR_TRUE, BYMINUTE=pattern_BYMINUT_TRUE)
        # 构造复制执行侧策略：
        trigger_pattern_false = "BEGIN:VCALENDAR\r\nBEGIN:VEVENT\r\nRRULE:FREQ={FREQ};BYDAY={BYDAY};" \
                                "BYHOUR={BYHOUR};BYMINUTE={BYMINUTE}\r\nEND:VEVENT\r\n" \
                                "END:VCALENDAR\r\n".format(FREQ=pattern_FREQ_FALSE, BYDAY=pattern_BYDAY_FALSE,
                                                           BYHOUR=pattern_BYHOUR_FALSE, BYMINUTE=pattern_BYMINUT_FALSE)
        # 判断构建计划的类型
        # 创建备份计划
        try:
            logger.info("Start creating a backup plan that sets up a scheduling policy...")
            ret = self.create_plan_include_backupandcopy_auto(plan_name, source_list,
                                                              max_backups_backup=max_backups_true,
                                                              max_backups_copy=max_backups_false,
                                                              Scheduling_name_backup=Scheduling_name_true,
                                                              Scheduling_name_copy=Scheduling_name_false,
                                                              expired_time=expired_time_str,
                                                              trigger_pattern_backup=trigger_pattern_true,
                                                              trigger_pattern_copy=trigger_pattern_false,
                                                              is_enabled_backup=is_enable_true,
                                                              is_enabled_copy=is_enable_false,
                                                              is_exe_policy=True,
                                                              operation_type=["backup", "full_backup"]
                                                              )
            logger.info("Expected results: The backup plan was successfully created")
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed to create the backup plan, errMsg:{}".format(e))
        if ret["policy"]["name"] == plan_name and ret["policy"]["resources"] == source_list:
            logger.debug("Backup plan creating results information is consistent with expectations"
                         "The backup plan name is: {i}, The name of the backeup's resource: {j},"
                         "Backup schedule expiration time: {k}".format(i=plan_name,
                                                                       j=source_list,
                                                                       k=expired_time_str))
        else:
            raise Exception("The backup plan created doesn't match the actual backup plan information, "
                            "please check!")
        # 获取计划id
        plan_id = ret["policy"]["id"]
        scheduled_id_list = []
        if ret["policy"]["scheduled_operations"][0]["operation_type"] == "backup":
            scheduled_id_list.append(ret["policy"]["scheduled_operations"][0]["id"])
            scheduled_id_list.append(ret["policy"]["scheduled_operations"][1]["id"])
        else:
            scheduled_id_list.append(ret["policy"]["scheduled_operations"][1]["id"])
            scheduled_id_list.append(ret["policy"]["scheduled_operations"][0]["id"])
        created_time = ret["policy"]["created_at"]
        server_id_list = []
        for source in source_list:
            server_id_list.append(source["id"])

        return plan_id, server_id_list, expired_time_str, scheduled_id_list, created_time

    def common_delete_image_normal(self, checkpoint_item_id, checkpoint_id):
        """删除副本并且等待副本删除完成(包括备份副本和拷贝副本，具体是那一种根据你所传递的副本id来定)

        输入参数:
        - tenant_id: 租户id
        - provider_id:提供商，参数类型string
        - checkpoint_item_id:副本的itemid,参数类型string
        - checkpoint_id: 副本id，参数类型string

        返 回 值:无
        异常描述:描述可能抛出的各种异常，若无，则删除本行。
        使用方法及范例:
        """
        logger.info("Start deleting copies...")
        if isinstance(checkpoint_item_id, list):
            checkpoint_item_id1 = checkpoint_item_id[0]
        else:
            checkpoint_item_id1 = checkpoint_item_id
            # 查看副本状态
        try:
            ret1 = self.get_image_detail(checkpoint_item_id1)
            logger.info("succeed getting copy information")
        except Exception as err:
            logger.error(err)
            self.close_session()
            if "CSBS.3001" in str(err):
                logger.info("The backup copy does not exist, "
                            "deleting successfully.")
                return True
            raise Exception("Failed to get the information of copy {},"
                            " errMsg:{}".format(checkpoint_item_id1, err))
        if ret1["checkpoint_item"]["status"] not in ["error", "available"]:
            raise Exception("The copy status is {i},which can not to be deleted, "
                            "please check!".format(i=ret1["checkpoint_item"]["status"]))
        try:
            self.delete_image(checkpoint_id)
            logger.info("Deleting copy task issued successfully")
        except Exception as e:
            logger.debug(e)
            self.close_session()
            raise Exception("Failed deleting copy, errMsg:{}".format(e))
        logger.info("Waiting for the copy to be deleted successfully...")
        # 获取副本的状态值
        count = 0
        delete_result = False
        try:
            while count < 100:
                copy_list_info = self.get_image_list()
                copy_checkpoint_id_list = []
                for copy in copy_list_info["checkpoint_items"]:
                    copy_checkpoint_id_list.append(copy["checkpoint_id"])
                if checkpoint_id not in copy_checkpoint_id_list:
                    delete_result = True
                    logger.info("Succeed deleting the copy")
                    break
                time.sleep(30)
                count = count + 1
        except Exception as e:
            logger.error("Deleting item error:%s" % traceback.format_exc())
            self.close_session()
            raise Exception("There is a problem in waiting for deleting the copy, errMsg:{}".format(e))
        return delete_result

    def SBS_DelBackupPlan_DelNormal(self, service_id):
        """正常情况下删除备份计划
        输入参数:
        - tenant_id: 租户ID
        - service_id: 备份计划ID

        返 回 值:无

        异常描述:获取的信息与预期不符
        使用方法及范例:

        """
        # 2、删除备份任务
        try:
            logger.info("Start deleting the backup plan")
            self.delete_backupplan(service_id)
            logger.info("Succeed deleting the backup plan")
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed deleting the backup plan, errMsg:{}".format(e))

    def SBS_ChangeBackupPlan_ChangeBackupPlanNormal(self, service_id, **kwargs):
        """修改备份计划

        输入参数:

        - dj_ip: 登录dj的IP地址
        - dj_user: 登录dj的用户名
        - dj_password: 登录dj的密码
        - tenant_id: 租户ID bn
        - service_id: 备份计划ID

        kwargs:可选参数对象,以键值对的形式输入“key=value”:
        - name:备份计划的名称
        - description:备份计划的描述信息
        - parameters:备份参数（备份计划过期时间）
        - resources:备份资源
        - maxbackups:自动备份的最大副本数，整数
        - duration:备份副本保留的天数，取值为1-99999整数
        - pattern:触发时间设置"BEGIN:VCALENDAR\r\nBEGIN:VEVENT\r\nRRULE:FREQ={FREQ};BYDAY={BYDAY};BYHOUR={BYHOUR};BYMINUTE={BYMINUTE}\r\nEND:VEVENT\r\nEND:VCALENDAR\r\n"，其中FREQ，BYDAY，BYHOUR，BYMINUTE是需要修改的项
        - operationtype:备份类型backup或者copy或者full_backup
        注意：
        选择修改对象取值:
        - 修改name或者description:传值方式"name=wf_plan"
        - 修改parameters:传值方式“expired_at=2017-04-12T14:20:02
        - 修改resources:需要修改的备份主机，可以通过self.common_get_server_type获取对应的值“resources=获取的值”
        - 修改scheduled_operation对象:可输入的参数值有:max_backups,pattern,retention_duration_days,operation_type
        | maxbackups=["id",20] | # id为调度策略的id，创建备份计划时返回
        | duration=[id,20] |
        | pattern=[id,"BEGIN:VCALENDAR\r\nBEGIN:VEVENT\r\nRRULE:FREQ={FREQ};BYDAY={BYDAY};BYHOUR={BYHOUR};BYMINUTE={BYMINUTE}\r\nEND:VEVENT\r\nEND:VCALENDAR\r\n"] |
        返 回 值:无
        异常描述:获取的信息与预期不符
        使用方法及范例:
        """
        # 参数体构造
        modify_dict = {}
        if kwargs:
            for key in kwargs:
                if key == "parameters":
                    parameters = {
                        "common": {
                            "az": self.az,
                            "expired_at": kwargs[key]
                        }
                    }
                    modify_dict[key] = parameters
                elif key == "resources":
                    modify_dict[key] = kwargs[key]
                elif key == "maxbackups":
                    modify_dict = {"scheduled_operations": [
                        {
                            'enabled': True,
                            "operation_definition": {
                                "max_backups": kwargs[key][1],
                            },
                            "id": kwargs[key][0]
                        }
                    ]}
                elif key == "duration":
                    modify_dict = {
                        "parameters": {
                            "common": {
                                "az": self.az,
                            }
                        },
                        "scheduled_operations": [
                            {
                                'enabled': True,
                                "operation_definition": {
                                    "retention_duration_days": kwargs[key][1],
                                },
                                "id": kwargs[key][0]
                            }
                        ]}
                elif key == "operationtype":
                    modify_dict = {"scheduled_operations": [
                        {'enabled': True, "operation_type": kwargs[key][1], "id": kwargs[key][0]}]}
                elif key == "pattern":
                    modify_dict = {
                        "scheduled_operations": [
                            {
                                'enabled': True,
                                "trigger": {
                                    "properties": {
                                        "pattern": kwargs[key][1]
                                    }
                                },
                                "id": kwargs[key][0]
                            }
                        ]
                    }
                else:
                    modify_dict[key] = kwargs[key]
        modify_body = {"policy": modify_dict}
        logger.info("The backup plan is modified to be: {i}".format(i=modify_body))
        # 1、修改备份
        try:
            logger.info("Start modifying the backup plan...")
            self.change_backupplan(service_id, modify_body)
            logger.info("Succeed modifying the backup plan")
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed modifying the backup plan, errMsg:{}".format(e))

    def SBS_BackupManualorAuto_BackupManualorAuto(self, plan_id, plan_resource_id_list,
                                                  auto_trigger=False, except_status="available",
                                                  incremental=True, type=None):

        """云主机备份（手工执行）

        输入参数:
        - tenant_id：租户id
        - provider_id：
        - plan_id:备份计划id
        - plan_resource_id_list:备份计划中的资源id信息，创建备份计划时返回
        - auto_trigger:是否自动触发False或者True,当为False时表示手动备份，当为True时为自动备份
        - except_status:备份计划完成后的预期状态值

        返 回 值:checkpoint_id, checkpoint_item_id
        异常描述:备份不成功
        使用方法及范例:
        """
        time.sleep(15)
        # 执行备份
        try:
            logger.info("Start performing backup...")
            ret = self.backup(plan_id, plan_resource_id_list, auto_trigger, incremental=incremental)
            if ret["checkpoint"]["status"] == "protecting":
                logger.info("Expected result: Backup task issued successfully")
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed performing backup, errMsg:{}".format(e))
        logger.info("Wait for the backup task to be complete...")
        time.sleep(10)
        checkpoint_id = ret["checkpoint"]["id"]
        if type is None:
            checkpoint_item_id = self.common_Wait_Image_Status_Except(checkpoint_id,
                                                                      except_status)
            return checkpoint_id, checkpoint_item_id
        else:
            checkpoint_item_id_list = self.common_Wait_Image_Status_Except_morehost(checkpoint_id,
                                                                                    except_status)
            return checkpoint_id, checkpoint_item_id_list

    def common_Wait_Image_Status_Except(self, checkpoint_id, except_status="available"):
        """等待副本状态为期望值并返回checkpoint_item_id
        输入参数:
        - tenant_id：租户ID
        - check_point_id: 执行备份时返回。
        - except_status: 副本状态，默认为available

        返 回 值:checkpoint_item_id。
        异常描述:描述可能抛出的各种异常，若无，则删除本行。
        使用方法及范例:
        """
        time.sleep(5)
        checkpoint_item_id = None
        image_info = self.get_image_list()
        for n in range(len(image_info["checkpoint_items"])):
            if image_info["checkpoint_items"][n]["checkpoint_id"] == checkpoint_id:
                checkpoint_item_id = image_info["checkpoint_items"][n]["id"]
        image_info = self.get_image_detail(checkpoint_item_id)
        ac_plan_status = image_info["checkpoint_item"]["status"]
        count = 0
        try:
            while ac_plan_status != except_status:
                image_info = self.get_image_detail(checkpoint_item_id)
                ac_plan_status = image_info["checkpoint_item"]["status"]
                if ac_plan_status in ["protecting", "restoring", "copying", "available"]:
                    logger.info("Current task progress value: {i}"
                                .format(i=image_info["checkpoint_item"]["extend_info"]["progress"]))
                logger.info("Current copy's status:{i}".format(i=ac_plan_status))
                count = count + 1
                if ac_plan_status == "error":
                    raise Exception("Copy's status is error, task failed, please check")
                if ac_plan_status == except_status or count > 200:
                    logger.info("The state value of the copy is expected,"
                                " and the end state value: {i}".format(i=ac_plan_status))
                    break
                time.sleep(40)
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed performing the cloud server's backup plan, errMsg:{}".format(e))

        if ac_plan_status != except_status:
            raise Exception("Failed performing the cloud server's backup plan,"
                            " errMsg: the state value of the copy is not expected")
        return checkpoint_item_id

    def common_Wait_Image_Status_Except_morehost(self, checkpoint_id, except_status="available"):
        """等待副本状态为期望值并返回checkpoint_item_id
        输入参数:
        - tenant_id：租户ID
        - check_point_id: 执行备份时返回。
        - except_status: 副本状态，默认为available

        返 回 值:checkpoint_item_id。
        异常描述:描述可能抛出的各种异常，若无，则删除本行。
        使用方法及范例:
        """

        image_info = self.get_image_list()
        checkpoint_item_id = []
        for n in range(len(image_info["checkpoint_items"])):
            if image_info["checkpoint_items"][n]["checkpoint_id"] == checkpoint_id:
                checkpoint_item_id.append(image_info["checkpoint_items"][n]["id"])
        for i in range(len(checkpoint_item_id)):
            image_info = self.get_image_detail(checkpoint_item_id[i])
            ac_plan_status = image_info["checkpoint_item"]["status"]
            count = 0
            try:
                while ac_plan_status != except_status:
                    image_info = self.get_image_detail(checkpoint_item_id[i])
                    ac_plan_status = image_info["checkpoint_item"]["status"]
                    if ac_plan_status in ["protecting", "restoring", "copying", "available"]:
                        logger.info("Current task progress value:{i}"
                                    .format(i=image_info["checkpoint_item"]["extend_info"]["progress"]))
                    logger.info("Current copy's status:{i}".format(i=ac_plan_status))
                    count = count + 1
                    if ac_plan_status == "error":
                        raise Exception("Copy's status is error, task failed, please check")
                    if ac_plan_status == except_status or count > 200:
                        logger.info("The state value of the copy is expected,"
                                    " and the end state value: {i}".format(i=ac_plan_status))
                        break
                    time.sleep(30)

            except RuntimeError as e:
                logger.info(e)
                self.close_session()
                raise Exception("Failed performing the cloud server's backup plan, errMsg:{}".format(e))
            if ac_plan_status != except_status:
                raise Exception("Failed performing the cloud server's backup plan, "
                                "errMsg: the state value of the copy is not expected")
        return checkpoint_item_id

    def SBS_CopyBackupPlanImage_Manual(self, policy_id, auto_trigger=False,
                                       except_status="available", type=None):

        """手动执行拷贝

        输入参数:

        - dj_ip: dj服务地址
        - dj_user: 鉴权用户名
        - dj_password: 鉴权的密码
        - tenant_id: 租户ID
        - provider_id:
        - plan_id: 备份计划ID
        - auto_trigger:ture or false
        - except_status:拷贝副本的结果状态

        返 回 值:checkpoint_id，checkpoint_item_id
        异常描述:获取的信息与预期不符
        使用方法及范例:
        """
        if type is None:
            # 执行拷贝任务
            try:
                logger.info("Start Performing replicate")
                ret = self.copy_backupimage(policy_id, auto_trigger)
                logger.info("Performing replicate command was issued successfully")
                checkpoint_item_id = ret["checkpoint_items"][0]
            except RuntimeError as e:
                logger.info(e)
                self.close_session()
                raise Exception("Failed to issue performing replicate command, errMsg:{}".format(e))
            # 等待任务执行完成
            logger.info("Waiting for the replication task to be complete...")
            image_info = self.get_image_detail(checkpoint_item_id)
            ac_plan_status = image_info["checkpoint_item"]["status"]
            count = 0
            try:
                while ac_plan_status != except_status:
                    image_info = self.get_image_detail(checkpoint_item_id)
                    ac_plan_status = image_info["checkpoint_item"]["status"]
                    count = count + 1
                    if ac_plan_status == except_status or ac_plan_status == "error" or count > 31:
                        logger.info("The state value of the copy is expected,"
                                    " and the end state value: {i}".format(i=ac_plan_status))
                        break
                    time.sleep(60)
            except RuntimeError as e:
                logger.info(e)
                self.close_session()
                raise Exception("Failed performing the cloud server's backup plan, errMsg:{}".format(e))
            checkpoint_id = image_info["checkpoint_item"]["checkpoint_id"]
            return checkpoint_id, checkpoint_item_id
        else:
            # 执行拷贝任务
            try:
                logger.info("Start Performing replicate")
                ret = self.copy_backupimage(policy_id, auto_trigger)
                logger.info("Performing replicate command was issued successfully")
                checkpoint_item_id = ret["checkpoint_items"]
            except RuntimeError as e:
                logger.info(e)
                self.close_session()
                raise Exception("Failed to issue performing replicate command, errMsg:{}".format(e))
            # 等待任务执行完成
            logger.info("Waiting for the replication task to be complete...")
            checkpoint_id = None
            for i in range(len(checkpoint_item_id)):
                image_info = self.get_image_detail(checkpoint_item_id[i])
                ac_plan_status = image_info["checkpoint_item"]["status"]
                count = 0
                try:
                    while ac_plan_status != except_status:
                        image_info = self.get_image_detail(checkpoint_item_id[i])
                        ac_plan_status = image_info["checkpoint_item"]["status"]
                        count = count + 1
                        if ac_plan_status == except_status or ac_plan_status == "error" or count > 31:
                            logger.info("The state value of the copy is expected,"
                                        " and the end state value: {i}".format(i=ac_plan_status))
                            break
                        time.sleep(60)
                except RuntimeError as e:
                    logger.info(e)
                    self.close_session()
                    raise Exception("Failed performing the cloud server's backup plan, errMsg:{}".format(e))
                checkpoint_id = image_info["checkpoint_item"]["checkpoint_id"]
            return checkpoint_id, checkpoint_item_id

    def common_wait_autoplan_start_complete(self, plan_resource_id_list,
                                            operation_type="backup", time1=None):
        """等待自动触发备份任务开始，并执行结束（存在返回checkpoint_item_id，checkpoint_id）
        ps:副本列表中不存在
        输入参数:
        - tenant_id:租户id
        - plan_resource_id:备份主机的的id
        - operation_type:任务类型backup或者copy
        :return:checkpoint_id,checkpoint_item_id
        """
        if time1 is not None:
            logger.info("Start waiting for {time} seconds....".format(time=time1))
            time.sleep(time1)
        else:
            pass
        plan_resource_id = plan_resource_id_list[0]
        logger.info("Checking if an auto-triggered backup task exists")
        count = 0
        checkpoint_item_id = None
        checkpoint_id = None
        try:
            while count < 100:
                copy_list_info = self.get_image_list()
                logger.info("Query the list of backup's copies %s times" % count)
                for copy in copy_list_info["checkpoint_items"]:
                    if copy["resource_id"] == plan_resource_id \
                            and copy["extend_info"]["image_type"] == operation_type \
                            and (copy["status"] == "waiting_protect" or copy["status"] == "protecting"):
                        checkpoint_item_id = copy["id"]
                        checkpoint_id = copy["checkpoint_id"]
                        logger.info("Succeed performing Auto-triggered backup task")
                        logger.info("checkpoint_item_id is: {i},"
                                    "checkpoint_id is:{k}".format(i=checkpoint_item_id, k=checkpoint_id))
                        break
                if checkpoint_item_id:
                    break
                time.sleep(12)
                count = count + 1
            else:
                raise Exception("Failed to trigger Auto-backup task")
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("There was an error in querying of the copys list, errMsg:{}".format(e))

        self.common_Wait_Image_Status_Except(checkpoint_id, except_status="available")
        return checkpoint_id, checkpoint_item_id

    def common_get_server_list(self, vmname, retur_type="1"):
        """按某种模板类型获取需要的虚拟机信息
        输入参数:
        - tenant_id:租户id
        - vmname:需要查找虚拟机的名称，参数类型是列表
        - type:需要返回资源组成结构:1、[{"id":"XX","type":"xx":"name":"xxx"},……]2、["id","id"],

        返回值:需要查询的虚拟机值列表
        """
        # 获取当前租户下虚拟机的id
        try:
            logger.info("Get the ids of all vms under the current tenant")
            server_list = self.get_all_servers()
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed getting the ids of all vms under the current tenant, "
                            "please check! errMsg: {}".format(e))

        source_list = []
        if retur_type == "1":
            if isinstance(vmname, list):
                for i in vmname:
                    source_item = {"type": "OS::Nova::Server"}
                    source_item["name"] = i
                    source_item["id"] = server_list[i]
                    source_list.append(source_item)
            else:
                raise Exception("The entered vmname is not a list, please check the parameter type!")
        if retur_type == "2":
            if isinstance(vmname, list):
                for i in vmname:
                    source_list.append(server_list[i])
            else:
                for i in [vmname]:
                    source_list.append(server_list[i])
        logger.info("vms information required: {i}".format(i=source_list))
        return source_list

    def common_get_volume_id(self, volume_name):
        """按某种模板类型获取需要的虚拟机信息

        输入参数:
        - tenant_id:租户id，参数类型string
        - vmname:需要查找虚拟机的名称，参数类型是list
        - type:需要返回资源组成结构:1["id","id"],2[{"id":"XX","type":"xx":"name":"xxx"},……]

        返回值:需要查询的卷id列表

        """
        # 获取当前租户下虚拟机的id
        try:
            logger.info("Get the ids of all vms under the current tenant")
            volume_info = self.list_all_volumes()
            servers_dict = {}
            for volume in volume_info["volumes"]:
                servers_dict[volume["name"]] = volume["id"]
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed getting the ids of all vms under the current tenant, "
                            "please check!, errMsg: {}".format(e))
        source_list = []
        if isinstance(volume_name, list):
            for i in volume_name:
                source_list.append(servers_dict[i])
        else:
            for i in [volume_name]:
                source_list.append(servers_dict[i])
        logger.info("vms information required: {i}".format(i=source_list))
        return source_list

    def common_get_all_volume(self, server_id, return_type="1"):
        """查询虚拟机上所挂卷的信息
        备注:系统卷的名称中包含sys，数据卷的大小一致

        输入参数:
        - tenant_id:租户id，参数类型string
        - server_id:虚拟机id，参数类型string
        - return_type:返回值类型，参数string，取值为1的时候返回 卷id列表，取值为2的时候返回的是，卷名称列表,取值为3的时候返回值是，卷详情

        返 回 值:返回卷id和卷name列表，系统卷在第一位
        """

        # 获取当前虚拟机所有的volume
        try:
            logger.info("Get all the volumes of the current vm")
            volume_info = self.list_attached_volumes(server_id)
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed getting all the volumes of the current vm, errMsg: {}".format(e))
        # 筛选需要的磁盘id
        volume_all_id_list = []
        volume_id_list = []
        volume_name_list = []
        for volume in volume_info["volumeAttachments"]:
            volume_all_id_list.append(volume["volumeId"])
        # 查看卷详情
        for volume_id in volume_all_id_list:
            try:
                logger.info("View the volume's details")
                volume_id_info = self.show_volume(volume_id)
                if "sys" in volume_id_info["volume"]["name"] \
                        and "a" in volume_id_info["volume"]["attachments"][0]["device"]:
                    volume_id_list.insert(0, volume_id)
                    volume_name_list.insert(0, volume_id_info["volume"]["name"])
                else:
                    volume_id_list.append(volume_id)
                    volume_name_list.append(volume_id_info["volume"]["name"])
            except RuntimeError as e:
                logger.info(e)
                self.close_session()
                raise Exception("Failed to view the volume's details, errMsg: {}".format(e))
        if return_type == "1":
            return volume_id_list
        if return_type == "2":
            return volume_name_list
        if return_type == "3":
            return volume_info

    def SBS_Use_Backup_RestoreToHost_RestoreNormal(self, checkpoint_id, checkpoint_item_id,
                                                   backup_sys_volume_id, des_vmname, des_volume_name):

        """根据指定副本恢复到指定云主机
        PS：云主机中是否有数据，需要手工
        :param tenant_id:租户ID
        :param provider_id:供应商ID
        :param checkpoint_id:
        :param checkpoint_item_id:
        :param backup_sys_volume_id:备份对象虚拟机的系统盘ID
        :param des_vmname:恢复的目标虚拟机名称
        :param des_volume_name:目标系统卷卷名称
        :return:无
        """
        new_resource_id = self.common_get_server_list(des_vmname, retur_type="2")[0]
        new_sys_volume_id = self.common_get_volume_id(des_volume_name)[0]
        new_volume_detail_list = self.common_get_all_volume(new_resource_id, return_type="3")
        logger.info("All volumes' information of the current vm:\n{i}".format(i=new_volume_detail_list))
        volumeAttachments = new_volume_detail_list["volumeAttachments"]
        # 拼装目标主机volume id 列表
        new_volume_id_list = []
        for volume_info in volumeAttachments:
            if volume_info['volumeId'] == new_sys_volume_id:
                new_volume_id_list.insert(0, volume_info['volumeId'])
            else:
                new_volume_id_list.append(volume_info['volumeId'])
        logger.info("List of all volumes' ids for the current vm:\n{}".format(new_volume_id_list))
        # 5、执行恢复任务
        self.common_Restore_to_Host(checkpoint_item_id,
                                    backup_sys_volume_id, new_resource_id, new_volume_id_list)
        # 6、等待恢复任务完成
        self.common_Wait_Image_Status_Except(checkpoint_id)

    def make_targets(self, restore_server_id, backup_id_list, volume_id_list):
        """
        公共函数，来组织target的结构
        :param restore_server_id:
        :param backup_id_list:
        :param volume_id_list:
        :return:
        """
        targets = {"server_id": restore_server_id,
                   "volumes": []}
        for i in range(len(backup_id_list)):
            target = dict()
            target['backup_id'] = backup_id_list[i]
            target['volume_id'] = volume_id_list[i]
            targets['volumes'].append(target)
        logger.info('make targets result is: \n {i}'.format(i=targets))
        return targets

    def common_Restore_to_Host(self, checkpoint_item_id, backup_sys_volume_id, resource_id,
                               volume_id_list, backup_id=None):

        """执行恢复到指定云主机

        备注:恢复的目标卷列表，系统卷id放置在列表第一位，数据卷大小一致

        输入参数:
        - tenant_id：租户ID
        - provider_id:
        - checkpoint_item_id：副本ID，一般由执行备份返回
        - backup_sys_volume_id：备份主机的系统卷id，参数类型string
        - resource_id：恢复的目标主机id，参数类型string
        - volume_id_list:恢复的目标主机卷列表，参数类型list

        返 回 值:无
        使用方法及范例:
        """
        try:
            logger.info("Start getting copy information")
            retinfo = self.get_image_detail(checkpoint_item_id)
            checkpoint_id = retinfo["checkpoint_item"]["checkpoint_id"]
            backup_id_item_list = []
            volume_backups = retinfo["checkpoint_item"]["extend_info"]["volume_backups"]
            for volume_backup in volume_backups:
                temp_dict = dict()
                temp_dict['source_volume_id'] = volume_backup['source_volume_id']
                temp_dict['id'] = volume_backup['id']
                backup_id_item_list.append(temp_dict)
            # 找出系统盘，放到第一个位置
            backup_id_list = []
            for backup_id_item in backup_id_item_list:
                if backup_id_item['source_volume_id'] == backup_sys_volume_id:
                    backup_id_list.insert(0, backup_id_item['id'])
                else:
                    backup_id_list.append(backup_id_item['id'])
            if len(backup_id_list) > len(volume_id_list):
                raise ValueError('The number of backup disks is inconsistent with the number of target disks')
            target = self.make_targets(resource_id, backup_id_list, volume_id_list)
            logger.info("the information of vm that needs to be restored")
            logger.info(target)
            logger.info("Perform restoring")
            self.restore(checkpoint_id, checkpoint_item_id, target)
            logger.info("Succeed to issue performing restore")
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed to issue performing restore, errMsg:{}".format(e))

    def common_Restore(self, checkpoint_item_id, original_volume="", replace_volume=""):
        """执行恢复

        输入参数:
        - tenant_id：租户ID
        - provider_id:
        - checkpoint_item_id：副本ID，一般由执行备份返回
        - original_volume 源卷id（适用于源主机删除一个卷后再添加卷并执行恢复操作)
        - replace_volume  替换卷id（适用于源主机删除一个卷后再添加卷并执行恢复操作)
        - power：是否启动虚拟机，默认为"on"

        返 回 值:无。
        异常描述:描述可能抛出的各种异常，若无，则删除本行。
        使用方法及范例:
        """
        logger.info('Sleep 60 seconds')
        time.sleep(60)
        try:
            logger.info("Start getting copy information")
            retinfo = self.get_image_detail(checkpoint_item_id)
            resource_id = retinfo["checkpoint_item"]["resource_id"]
            checkpoint_id = retinfo["checkpoint_item"]["checkpoint_id"]
            volume_backups = retinfo["checkpoint_item"]["extend_info"]["volume_backups"]
            backup_id_list = []
            volume_id_list = []
            for volume_backup in volume_backups:
                backup_id_list.append(volume_backup['id'])
                volume_id_list.append(volume_backup['source_volume_id'])
            if original_volume and replace_volume:
                volume_id_list[volume_id_list.index(original_volume)] = replace_volume
            target = self.make_targets(resource_id, backup_id_list, volume_id_list)
            logger.info("the information of vm that needs to be restored")
            logger.info(target)
            logger.info("Perform restoring")
            ret = self.restore(checkpoint_id, checkpoint_item_id, target)
            logger.info("Succeed to issue performing restore, the response: {}".format(ret))
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed to issue performing restore, errMsg:{}".format(e))

    def get_product_id_from_name(self, evs_product_name):
        """根据product名字获取product_id

        :param region_name:
        :return:
        """
        code, res = self.show_product_list()
        total = res["total"]
        product_list = res["products"]
        for n in range(0, total):
            if evs_product_name in product_list[n]["name"]:
                logger.info(product_list[n]["product_id"])
                return product_list[n]["product_id"]
        else:
            logger.info("{i} Product not exists".format(i=evs_product_name))
            raise Exception("{i} Product not exists".format(i=evs_product_name))

    def get_region_id_from_name(self, region_name):
        """根据region名字获取region_id

        :param region_name:
        :return:
        """
        code, res = self.show_region_list()
        total = res["total"]
        regions_list = res["regions"]
        logger.info("Get the region id according to the region name")
        for n in range(0, total):
            if regions_list[n]["name"] == region_name:
                logger.info(regions_list[n]["id"])
                return regions_list[n]["id"]
        else:
            logger.info("{i} region not exists".format(i=region_name))
            raise Exception("{i} region not exists".format(i=region_name))

    def Bind_Resource(self, policy_id, resource_list):
        """为指定备份策略绑定资源

        :param project_id:
        :param policy_id:
        :return:
        {
          "bind_results" : [ {
            "id" : "6507cb66-90dc-4a12-a573-c9f3398f899d",
            "result" : true,
            "error_msg" : "",
            "error_code" : ""
          } ]
        }

        """
        url = "%s/v1/{project_id}/policies/{policy_id}/action".format(project_id=self.project_id,
                                                                      policy_id=policy_id) % self.base_url
        logger.info(url)
        body_dict = {
            "bind_resources": resource_list
        }
        return self._post(url, json=body_dict)

    def Untied_Resource1(self, policy_id, unbind_resources):
        """为指定策略解绑资源

        :param policy_id:
        :param unbind_resources:
        :return:
        """
        url = "%s/v1/{project_id}/policies/{policy_id}/action".format(project_id=self.project_id,
                                                                      policy_id=policy_id) % self.base_url
        body_dict = {
            "unbind_resources": unbind_resources
        }
        body_body = json.dumps(body_dict)
        return self._post(url, body_body)

    def SBS_Bind_Resourcfe(self, policy_id, vmname_list, vmtype="OS::Nova::Server"):
        """给指定策略绑定云主机

        :param tenant_id:
        :param policy_id:
        :param vmname_list:
        :return:
        """
        # 获取当前租户下虚拟机的id
        try:
            logger.info("Get the ids of all vms under the current tenant")
            server_list = self.get_all_servers()
        except RuntimeError as e:
            logger.info(e)
            self.close_session()
            raise Exception("Failed getting the ids of "
                            "all vms under the current tenant, errMsg:{}".format(e))

        source_list = []
        server_id_list = []
        source_item = {"name": "", "id": "", "type": ""}
        # 创建备份计划需要的资源信息（即虚拟机信息）
        if isinstance(vmname_list, list):
            for i in vmname_list:
                source_item = {"name": "", "id": "", "type": ""}
                server_id = server_list[i]
                source_item["name"] = i
                source_item["id"] = server_id
                source_item["type"] = vmtype
                source_list.append(source_item)
                server_id_list.append(server_id)
        else:
            server_id = server_list[vmname_list]
            source_item["name"] = vmname_list
            source_item["id"] = server_id
            source_item["type"] = vmtype
            source_list.append(source_item)
            server_id_list.append(server_id)
        logger.info("vm information that needs to be backed up: %s" % source_list)
        try:
            logger.info("Start binding a new cloud server to a policy...")
            ret = self.Bind_Resource(policy_id, source_list)
            logger.info("Binding successful")
        except RuntimeError as e:
            logger.info(e)
            logger.info("The binding failed, errMsg: {}".format(e))
        resource_id_list = []
        for resources in ret["bind_results"]:
            resource_id_list.append(resources["id"])
        return resource_id_list

    def SBS_unbind_Resourcfe(self, policy_id, vmname_list, vmtype="OS::Nova::Server"):
        # 获取当前租户下虚拟机的id
        try:
            logger.info("Get the ids of all vms under the current tenant")
            server_list = self.get_all_servers()
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting the ids of "
                            "all vms under the current tenant, errMsg:{}".format(e))

        source_list = []
        server_id_list = []
        source_item = {"name": "", "id": "", "type": ""}
        # 创建备份计划需要的资源信息（即虚拟机信息）
        if isinstance(vmname_list, list):
            for i in vmname_list:
                source_item = {"name": "", "id": "", "type": ""}
                server_id = server_list[i]
                source_item["name"] = i
                source_item["id"] = server_id
                source_item["type"] = vmtype
                source_list.append(source_item)
                server_id_list.append(server_id)
        else:
            server_id = server_list[vmname_list]
            source_item["name"] = vmname_list
            source_item["id"] = server_id
            source_item["type"] = vmtype
            source_list.append(source_item)
            server_id_list.append(server_id)
        logger.info("vm information that needs to be backed up: %s" % source_list)
        try:
            logger.info("Start unbinding a new cloud server to a policy...")
            self.Untied_Resource1(policy_id, source_list)
            logger.info("unbinding success")
        except RuntimeError as e:
            logger.info(e)
            logger.info("Failed unbinding, errMsg: {}".format(e))

    def get_backupplan_detail(self, policy_id):
        """获取备份计划详情

        输入参数:
        - project_id：
        - policy_id：备份计划ID

        异常描述:返回错误码

        使用方法及范例:

        修改历史:

        """
        url = "%s/v1/{project_id}/policies/{policy_id}".format(project_id=self.project_id,
                                                               policy_id=policy_id) % self.base_url
        return self._get(url)

    def Check_polices_host(self, policy_id, vmname):
        '''
        检查某个虚拟机是否存在策略中
        :return:
        '''
        try:
            logger.info("Start getting backup plan's details")
            re = self.get_backupplan_detail(policy_id=policy_id)
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting backup plan's details, errMsg:{}".format(e))
        if vmname in str(re):
            logger.info("Vm is present in the policy")
        else:
            raise Exception("Vm is not present in the policy, please check!")

    def Check_deleteResion_host(self, policy_id, vmname):
        '''
        检查某个虚拟机是否从策略中删除
        :return:
        '''
        try:
            logger.info("Start getting backup plan's details")
            re = self.get_backupplan_detail(policy_id=policy_id)
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting backup plan's details, errMsg:{}".format(e))
        if vmname not in str(re):
            logger.info("Vm is not present in the policy")
        else:
            raise Exception("Vm is present in the policy, please check!")

    def wait_next_auto_run(self, plan_id, plan_resource_list, scheduled_id, operation_type="backup", **kwargs):
        """等待下一次自动调度任务执行（给定下一次调度的时间）

        :return:
        """
        time1 = None
        time_info1 = [None] * 3
        if "minutes" in kwargs:
            time_info1 = self.GetFutureTimeParameters(minutes=kwargs["minutes"])
            time1 = kwargs["minutes"] * 60
        elif "hours" in kwargs:
            time_info1 = self.GetFutureTimeParameters(hours=kwargs["hours"])
            time1 = kwargs["hours"] * 60 * 60
        dreq = "WEEKLY"
        pattern = "BEGIN:VEVENT\r\nRRULE:FREQ=" + dreq + ";" + "BYDAY=" \
                  + time_info1[0] + ";" + "BYHOUR=" + time_info1[1] + \
                  ";" + "BYMINUTE=" + time_info1[2] + "\r\nEND:VEVENT\r\n"
        trigger = [scheduled_id, pattern]
        self.SBS_ChangeBackupPlan_ChangeBackupPlanNormal(plan_id, pattern=trigger)
        re = self.common_wait_autoplan_start_complete(plan_resource_list,
                                                      operation_type=operation_type,
                                                      time1=time1)
        return re

    def teardown(self, checkpoint_item_id=None, checkpoint_id=None, plan_id=None):
        """
        删除副本，再删除备份计划
        :param checkpoint_id: 副本ID
        :param checkpoint_item_id:副本ID列表
        :param plan_id: 备份计划ID
        :return:

        """
        # 检查是否有副本需要删除
        if checkpoint_id and checkpoint_item_id is not None:
            x = 0
            while x < 5:
                try:
                    # 删除副本
                    result = self.common_delete_image_normal(
                        checkpoint_item_id, checkpoint_id)
                    if result:
                        x = 6
                except Exception as err:
                    logger.error("Deleting checkpoint_item_id: %s, "
                                 "retry time: %s, err: %s"
                                 % (checkpoint_item_id, x, err))
                    x = x + 1
                    time.sleep(100)
            else:
                if x == 6:
                    pass
                else:
                    raise Exception("Failed deleting the copy, please check!")
        # 检查是否有计划需要删除
        if plan_id is not None:
            y = 0
            while y < 5:
                try:
                    try:
                        # 删除备份计划
                        self.SBS_DelBackupPlan_DelNormal(plan_id)
                        y = 6
                    except:
                        raise Exception("Failed deleting policy")
                except:
                    y = y + 1
                    time.sleep(150)
            else:
                if y == 6:
                    pass
                else:
                    raise Exception("Failed deleting policy, please check!")

    def change_current_time(self, now, type=None, days=None, hours=None, minutes=None, type1=None):
        """改变当前时间

        :param type:为空表示是减少时间，不为空表示增加时间
        :param days:以天为单位变化
        :param hours:以小时为单位变化
        :param minutes:以分钟为单位变化
        :return:返回一个修改后的当前时间

        """
        now1 = now - timedelta(hours=self.karbor_timezone)
        changetime = now1
        if type1 is None:
            if type is not None:
                if days is not None:
                    changetime = now + timedelta(days=days)
                elif hours is not None:
                    changetime = now + timedelta(hours=hours)
                elif minutes is not None:
                    changetime = now + timedelta(minutes=minutes)
            else:
                if days is not None:
                    changetime = now - timedelta(days=days)
                elif hours is not None:
                    changetime = now - timedelta(hours=hours)
                elif minutes is not None:
                    changetime = now - timedelta(minutes=minutes)
        return changetime

    def get_week_day(self, date):
        """
        获取当前日期是星期几，主要是为了给SCHEDLE_BYDAY参数赋值用。
        :param date:使用datetime.datetime.now()获取出来的当前时间
        :return: MO,TU,WE,TH,FR,SA,SU中的一个

        """

        week_day_dict = {
            0: "MO",
            1: "TU",
            2: "WE",
            3: "TH",
            4: "FR",
            5: "SA",
            6: "SU"
        }
        day = date.weekday()
        return week_day_dict[day]

    def GetFutureTimeParameters(self, **kwargs):
        """构造，创建自动调度计划需要的byday,hour,minute3个参数
        【这里构造的时间参数，只是用于构造未来时间】
        kwargs 可以是 hours = ??  minutes = ?? days = ?? 分别表示 获取 多久 以后的时间参数byday hour minute
        :return:byday,hour,minute
        """
        nowtime = self.connect_to_karbor()
        byday_tmp1 = self.change_current_time(nowtime, type1=2)
        if "hours" in kwargs:
            futuretime = self.change_current_time(byday_tmp1, type=1, hours=kwargs["hours"])
            byday = self.get_week_day(futuretime)
            tmp0 = list(str(futuretime))
            hour = tmp0[11] + tmp0[12]
            minute = tmp0[14] + tmp0[15]
        elif "days" in kwargs:
            futuretime = self.change_current_time(byday_tmp1, type=1, days=kwargs["days"])
            byday = self.get_week_day(futuretime)
            tmp0 = list(str(futuretime))
            hour = tmp0[11] + tmp0[12]
            minute = tmp0[14] + tmp0[15]
        else:
            futuretime = self.change_current_time(byday_tmp1, type=1, minutes=kwargs["minutes"])
            byday = self.get_week_day(futuretime)
            tmp0 = list(str(futuretime))
            hour = tmp0[11] + tmp0[12]
            minute = tmp0[14] + tmp0[15]
        return byday, hour, minute

    def __handle_timezone(self, timezone):
        tz = 0
        if timezone.__contains__('+'):
            _, time_zone = timezone.rsplit('+', 1)
            tz = int(time_zone[:2])
        elif timezone.__contains__('-'):
            _, time_zone = timezone.rsplit('-', 1)
            tz = -int(time_zone[:2])
        return tz

    def connect_to_karbor(self, cmd="date +%Y-%m-%d%H:%M:%S", cmd1='date -R'):
        """获取KARBOR的系统时间

        :param host:
        :param username:
        :param passwd:
        :param cmd:
        :return:
        """
        conn = paramiko.SSHClient()
        conn.load_system_host_keys()
        conn.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        conn.connect(self.KarBor_Info[0], 22, self.KarBor_Info[1], self.KarBor_Info[2])
        stdin, stdout, stderr = conn.exec_command(cmd)
        stdin1, stdout1, stderr1 = conn.exec_command(cmd1)
        local_time = stdout.read()
        logger.info(local_time)
        local_time1 = datetime.strptime(local_time, "%Y-%m-%d%H:%M:%S\n")
        timezone = stdout1.read()
        logger.info('Current time zone: %s' % timezone)
        self.karbor_timezone = self.__handle_timezone(timezone)
        conn.close()
        return local_time1

    def Query_quota(self):
        """获取配额信息

        :return:
        """
        url = "%s/v1/{project_id}/quotas".format(project_id=self.project_id) % self.base_url
        return self._get(url)

    def Check_quota_CSBS(self):
        try:
            logger.info("Start getting CSBS quota information...")
            quota_info = self.Query_quota()
            logger.info("Get successfully")
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting CSBS quota information, errMsg:{}".format(e))
        backup = None
        backup_quota = 0
        for quota in quota_info["quotas"]["resources"]:
            if quota["type"] == "backup_capacity":
                backup_quota = quota["quota"]
                if backup_quota < 102400:
                    logger.info("The CSBS backup capacity is {quota}M, "
                                "less than the established capacity".format(quota=quota["quota"]))
                    backup = 1
                else:
                    logger.info("The CSBS backup capacity is {quota}M, "
                                "which can be used normally".format(quota=quota["quota"]))

        if backup is not None:
            try:
                logger.info("Start applying for CSBS backup space")
                self.ADD_QUOTA(102400 - backup_quota)
                logger.info("Succeed applying for CSBS backup space")
            except Exception as e:
                raise Exception("Failed applying for CSBS backup space, errMsg:{}".format(e))

    def Check_teardown_quota_CSBS(self):
        try:
            logger.info("Start getting CSBS quota information...")
            quota_info = self.Query_quota()
            logger.info("Get successfully")
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting CSBS quota information, errMsg:{}".format(e))
        backup = None
        backup_quota = 0
        for quota in quota_info["quotas"]["resources"]:
            if quota["type"] == "backup_capacity":
                backup_quota = quota["quota"]
                if backup_quota != 0:
                    logger.info("The CSBS backup capacity is {}M, ready to reduce space".format(backup_quota))
                    backup = 1
                else:
                    logger.info("The CSBS backup capacity is {}M".format(backup_quota))
        if backup is not None:
            try:
                logger.info("Start requesting to reduce CSBS backup space")
                self.Mu_QUOTA(backup_quota)
                logger.info("Succeed requesting to reduce CSBS backup space")
            except Exception as e:
                raise Exception("Failed requesting to reduce CSBS backup space, errMsg:{}".format(e))

    def Check_setup(self):
        """任务开始前的检查
        1. 检查租户下是否有策略存在，如果有，则提前删除
        2. 检查租户下备份复制空间是否有容量，如果没有，则自动申请
        :return:
        """
        logger.info('Check and delete copys in the tenant')
        image_info_list = self.get_image_list()
        if not image_info_list['checkpoint_items']:
            logger.info('there are no copys in the tenant')
        else:
            checkpoint_dict = defaultdict(list)
            for image_info in image_info_list['checkpoint_items']:
                # 一条备份记录对应多个副本
                checkpoint_dict[image_info['checkpoint_id']].append(image_info['id'])
            logger.info('Current copy info: {}'.format(str(checkpoint_dict)))
            for checkpoint_id, checkpoint_item_ids in checkpoint_dict.items():
                self.teardown(checkpoint_item_id=checkpoint_item_ids, checkpoint_id=checkpoint_id)
            logger.info('Succeed to delete all copys')

        logger.info("Start checking for policy under the tenant and delete it ahead of time")
        plan_info = self.Check_plan()
        if plan_info is None:
            pass
        else:
            logger.info("Start deleting the policy")
            for plan_id in plan_info:
                self.SBS_DelBackupPlan_DelNormal(plan_id)
        logger.info("Check if there is capacity in the backup space under the tenant,"
                    " and if not, automatically request")
        self.Check_quota_CSBS()

    def Check_Teardown(self):
        """任务结束前的检查
        1. 检查租户下是否有策略存在，如果有，则提前删除
        2. 检查租户下备份复制空间是否有容量，如果没有，则自动申请
        :return:
        """
        logger.info("Start checking for policy under the tenant and delete it ahead of time")
        plan_info = self.Check_plan()
        if plan_info is None:
            pass
        else:
            logger.info("Start deleting the policy")
            for plan_id in plan_info:
                self.SBS_DelBackupPlan_DelNormal(plan_id)
        logger.info("Check if there is capacity in the backup space under the tenant,"
                    "and if so, automatically clear")
        self.Check_teardown_quota_CSBS()

    def Check_plan(self):
        """检查租户下是否有策略存在，如果存在的话，返回策略的ID列表

        :return:
        """
        try:
            logger.info("Get information of all policies under the tenant")
            re = self.get_plan_info()
            logger.info("Succeed getting information of all policies:{i}".format(i=re))
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting information of all policies, errMsg: {}".format(e))
        plan_id_list = []
        if not re["policies"]:
            logger.info("After checking, there is no policy under the tenant")
            plan_id_list = None
        else:
            logger.info("After checking, there is some policies under the tenant")
            for policies in re["policies"]:
                plan_id_list.append(policies["id"])
            logger.info("All policies ids under tenant: %s" % plan_id_list)
        return plan_id_list

    def get_plan_info(self):
        """获取租户下所有策略的信息

        :return:
        """
        url = "%s/v1/{project_id}/policies?provider_id=fc4d5750-22e7-4798-8a46-f48f62c4c1da".format(
            project_id=self.project_id) % self.base_url
        return self._get(url)


class CSBSBaseConsoleAdapter(object):
    def __init__(self, project_id):
        self.pro_id = project_id
        region_info = get_region_info(project_id)
        self.region_type = region_info.region_type
        self.region = region_info.region_id
        self.region_name = region_info.region_name
        self.az_id = get_az_id(project_id, region_info)
        params = ParamUtil()
        self.product_id = None
        self.product_name = "云服务器备份"
        self.project_name = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                              '%s_project1_name' % self.region_type,
                                                              self.region)
        self.provider_id = params.get_value_from_cloud_param(project_id,
                                                             'CSBS-VBS',
                                                             'csbs_provider_id',
                                                             self.region) or CSBS_PROVIDER_ID
        self.user_name = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                           '%s_vdc1_vsm_user1' % self.region_type,
                                                           self.region)
        self.password = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                          '%s_sc_user_new_password' % self.region_type,
                                                          self.region)
        self.console_host = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                              '%s_console_host' % self.region_type,
                                                              self.region)
        self.console_port = "443"
        self.auth_port = "443"
        self.auth_host = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                           '%s_iam_host' % self.region_type,
                                                           self.region)
        self.server1_name = SERVER1_NAME
        self.project_id = None
        self.console_sc_headers = {
            "Accept": "*/*",
            "Accept-Encoding": "gzip, deflate, br",
            "Accept-Language": "zh-CN,zh;q=0.8",
            "AgencyId": None,
            "cftk": None,
            "Connection": "keep-alive",
            "Content-Type": "application/json; charset=UTF-8",
            "Cookie": None,
            "Host": self.console_host,
            "ProjectName": None,
            "Referer": None,
            "region": None,
            "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) "
                          "Chrome/59.0.3071.104 Safari/537.36",
            "X-Language": "zh-CN,zh;q=0.8,en-US;q=0.5,en;q=0.3",
            "X-Requested-With": "XMLHttpRequest"
        }
        self.session = requests.session()

    def _pre_process(self, kwargs):
        """HTTP请求提交之前的预处理，主要是设置headers"""
        self.session.headers = self.console_sc_headers

    @staticmethod
    def _post_process(response):
        """对HTTP响应进行处理"""
        if response.ok:
            if len(response.text) > 0:
                ret = json.loads(response.text)
            else:
                ret = response.text
            return ret
        else:
            if "errorCode" in response.text:
                text_dict = json.loads(response.text)
                error_code = text_dict.get('errorCode')
                logger.info(error_code)
                raise Exception("Request sent failed, error code:{i}, "
                                "response: {j}".format(i=error_code, j=text_dict))
            else:
                error_msg = response.text
                logger.info(error_msg)
                raise Exception("Request sent failed, response: {j}".format(j=error_msg))

    def _get(self, url, **kwargs):
        """Sends a GET request.

        :param url: URL for the new :class:`Request` object.
        :param params: (optional) Dictionary or bytes to be sent in the query string for the :class:
         `Request`.
        :param \*\*kwargs: Optional arguments that ``request`` takes.
        :return: :class:`Response <Response>` object
        :rtype: requests.Response
        """
        self._pre_process(kwargs)
        response = self.session.get(url, **kwargs)
        resp_body = self._post_process(response)
        return resp_body

    def _post(self, url, data=None, json=None, **kwargs):
        """submit a request with POST method."""
        self._pre_process(kwargs)
        response = self.session.post(url, data=data, json=json, **kwargs)
        return self._post_process(response)

    def _put(self, url, data, **kwargs):
        """submit a request with PUT method."""
        self._pre_process(kwargs)
        response = self.session.put(url, data, **kwargs)
        return self._post_process(response)

    def _delete(self, url, **kwargs):
        """submit a request with DELETE method."""
        self._pre_process(kwargs)
        response = self.session.delete(url, **kwargs)
        return self._post_process(response)

    def login_sc(self):
        logion_obj = Login_SC(self.pro_id, self.region, self.region_type)
        user_id, self.project_id, self.region, \
        bak_global_headers, global_headers = logion_obj.login_sc_on_web(self.console_host,
                                                                        self.auth_host,
                                                                        self.user_name,
                                                                        self.password,
                                                                        self.region_name)
        self.console_sc_headers.update(global_headers)
        refer = "https://{console_host}/motenantconsolehomewebsite/".format(console_host=self.console_host)
        self.console_sc_headers['Referer'] = refer

        self.product_id = self.get_product_id_from_name(self.product_name)
        self.server1_id = self.get_id_through_name_service(self.server1_name)
        sys1_info = self.get_sysvol_info_us_serverid(self.server1_id)
        self.server1_system_volume_id = sys1_info[0]
        self.server1_system_volume_name = sys1_info[1]

        self.Check_product_status()

    def get_cloud_infra_and_regionid_by_regionname(self, console_host, region_name="ALL"):
        # 根据region名称获取region_id,cloud_infra_id,cloud_infra_name
        try:
            region_name_filed = "" if region_name == "ALL" else region_name
            url = "https://{host}/motenantconsolehomewebsite/goku/" \
                  "rest/serviceaccess/v3.0/regions?name={region}".format(host=console_host,
                                                                         region=region_name_filed)
            resp = requests.get(url, verify=False, allow_redirects=False, headers=self.console_sc_headers)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(response_body)
            if statu_code == 200:
                records = response_body.get("records")
                cloud_infra_list = []
                for region_info in records:
                    name = region_info["name"]
                    if region_name == name:
                        cloud_infras = region_info["cloud_infras"]
                        for cloud_infra in cloud_infras:
                            cloud_infra_temp = []
                            if cloud_infra["type"] == "FUSION_CLOUD":
                                region_id = cloud_infra["region_id"]
                                cloud_infra_id = cloud_infra["id"]
                                cloud_infra_name = cloud_infra["name"]
                                cloud_infra_temp.append(region_id)
                                cloud_infra_temp.append(cloud_infra_id)
                                cloud_infra_temp.append(cloud_infra_name)
                                cloud_infra_list.append(cloud_infra_temp)
                                return cloud_infra_list
                        else:
                            return None
                    elif region_name == "ALL":
                        cloud_infras = region_info["cloud_infras"]
                        for cloud_infra in cloud_infras:
                            cloud_infra_temp = []
                            if cloud_infra["type"] == "FUSION_CLOUD":
                                region_id = cloud_infra["region_id"]
                                cloud_infra_id = cloud_infra["id"]
                                cloud_infra_name = cloud_infra["name"]
                                cloud_infra_temp.append(region_id)
                                cloud_infra_temp.append(cloud_infra_id)
                                cloud_infra_temp.append(cloud_infra_name)
                                cloud_infra_list.append(cloud_infra_temp)
                return cloud_infra_list
            else:
                raise Exception("get cloud infra list failed! response is %s" % response_body)
        except Exception as e:
            logger.error('Get cloud infra list failed!')
            raise e

    def show_azone_list_of_region(self):
        url = "https://%s/moserviceaccesswebsite/goku/rest/serviceaccess/v3.0/available-zones?region_id=%s" % \
              (self.console_host, self.region)
        ret = self.session.get(url=url, headers=self.console_sc_headers, allow_redirects=False, verify=False)
        logger.info("Search all available zones under region")
        logger.info('Current request Status:[{}]'.format(ret.status_code))
        res = json.loads(ret.text)
        return ret.status_code, res

    def show_product_list(self):
        """获取产品信息

        :param user_id:
        :param service_type:
        :return:
        """
        # 展示可申请的产品列表
        url = "https://{console_url}/moproductwebsite/goku/rest/" \
              "product/v3.0/products?service_type={service_type}" \
              "&start=1&limit=20&region_id={region_id}".format(console_url=self.console_host,
                                                               service_type="csbs",
                                                               region_id=self.region)
        ret = self.session.get(url=url, headers=self.console_sc_headers, verify=False)
        logger.info(ret.status_code)
        res = json.loads(ret.text)
        return ret.status_code, res

    def online_product(self, product_id):
        """租户侧对CSBS产品进行上线操作

        :return:
        """
        url = "https://{console_url}/moproductwebsite/goku/" \
              "rest/product/v3.0/products/{product_id}/action".format(console_url=self.console_host,
                                                                      product_id=product_id)
        body = {"action_type": "online", "process_ids": []}
        body1 = json.dumps(body)
        self._put(url, data=body1, verify=False)

    def get_product_info(self):
        """

        :return:
        """
        url = "https://{console_url}/moproductwebsite/goku/rest/" \
              "product/v3.0/products?service_type=csbs&" \
              "limit=20&start=1&region_id={region_id}".format(console_url=self.console_host,
                                                              region_id=self.region)
        return self._get(url)

    def Check_product_status(self):
        """租户侧检查CSBS产品是否上线，如果没有上线的话，进行上线操作

        :return:
        """
        try:
            logger.info("Get product information")
            res = self.get_product_info()
            logger.info("Succeed getting product information")
        except RuntimeError as e:
            logger.debug(e)
            errMsg = 'Failed getting product information, errMsg: {}'.format(e)
            return Message(500, errMsg)
        product_id = res["products"][0]["product_id"]
        logger.info("Check that CSBS products are on line or not, if not, to carry out on-line operation")
        if res["products"][0]["online_status"] == "offline":
            try:
                logger.info("The product is not on-line, carry out on-line operation")
                self.online_product(product_id)
                logger.info("The on-line operation was successful")
                time.sleep(15)
            except RuntimeError as e:
                logger.debug(e)
                return Message(500, "Failed carrying out on-line operation, errMsg:{}".format(e))
        else:
            logger.info("The product is online")

    def get_plan_info(self):
        """获取租户下所有策略信息

        :return:
}, {u'status': u'suspended', u'provider_id': u'1dc03359-1301-467b-9972-1a7599ae791d', u'name': u'backup_policy_1004', u'parameters': {u'common': {u'az': u'VRMDC1'}}, u'created_at': u'2018-07-11T09:55:21.305619', u'scheduled_operations': [{u'description': None, u'enabled': True, u'trigger_id': u'ebb28e2e-4076-4a7a-bd22-8d306d78b5ce', u'trigger': {u'type': u'time', u'properties': {u'pattern': u'BEGIN:VEVENT\nRRULE:FREQ=DAILY;INTERVAL=2;BYHOUR=02;BYMINUTE=30\nEND:VEVENT', u'start_time': u'2018-07-11 09:55:21', u'format': u'ical'}, u'name': u'default', u'id': u'ebb28e2e-4076-4a7a-bd22-8d306d78b5ce'}, u'operation_definition': {u'max_backups': -1, u'retention_duration_days': 30, u'permanent': False, u'plan_id': u'1408da0a-2441-4f15-a99f-239ef145f1fe', u'provider_id': u'1dc03359-1301-467b-9972-1a7599ae791d'}, u'operation_type': u'backup', u'id': u'c8c5b8ab-db68-4a45-b01f-76ded97a464a', u'name': u'default'}], u'project_id': u'438078f34856492188b47f8a9b9e90b7', u'id': u'1408da0a-2441-4f15-a99f-239ef145f1fe', u'resources': [], u'description': None}]}

        """
        url = "https://{console_url}/cbs/rest/karbor/v1/" \
              "{project_id}/policies?provider_id={provider_id}" \
              "&sort=created_at+%3A+desc".format(console_url=self.console_host,
                                                 project_id=self.project_id,
                                                 provider_id=self.provider_id)
        return self._get(url, verify=False)

    def get_region_id_from_name(self, region_name):
        url = "https://{console_url}/movdcwebsite/rest/silvan/rest" \
              "/v1.0/regions?start=0&limit=0".format(console_url=self.console_host)
        ret = self._get(url=url, verify=False, allow_redirects=False, headers=self.console_sc_headers)
        region_id = ''
        region_list = ret['regions']
        logger.info('Start to get region id from region name')
        for region_info in region_list:
            if region_info['name'] == region_name:
                region_id = region_info['id']
                logger.info('Succeed to get region_id:%s' % region_id)
                return region_id
        logger.error('Can not find region id from region name, please check')
        return region_id

    def Query_quota(self):
        """查询配额信息
        :return:

        """
        url = "https://{console_url}/cbs/rest/karbor/v1/{project_id}/quotas".format(console_url=self.console_host,
                                                                                    project_id=self.project_id)
        return self._get(url, verify=False)

    def Add_quota_CSBS(self):
        """固定增加50000B的CSBS备份空间

        :return:
        走的订单，如果是审批模式则需要审批
        """
        url = "https://{console_url}/cbs/rest/karbor/v1/{project_id}/products/{product_id}/action".format(
            console_url=self.console_host, project_id=self.project_id, product_id=self.product_id)
        body = {"apply": {"service_type": "csbs", "region_id": self.region,
                          "params": "{\"count\":1,\"display\":[{\"label\":\"备份\",\"type\":\"string\","
                                    "\"value\":\"100GB\"}],\"param_items\":"
                                    "[{\"type\":\"backup_capacity\",\"quota\":50000}]}"}}
        body1 = json.dumps(body)
        return self._post(url, data=body1, verify=False)

    def Delete_plan(self, plan_id):
        """删除策略

        :return:
        """
        url = "https://{console_url}/cbs/rest/karbor/v1/" \
              "{project_id}/policies/{plan_id}".format(console_url=self.console_host,
                                                       project_id=self.project_id,
                                                       plan_id=plan_id)
        self._delete(url)

    def Check_quota_CSBS(self):
        """检查CSBS的备份和复制容量，是否为0，如果为0则申请100GB的容量

        :return:
        """
        try:
            logger.info("Start getting CSBS quota information...")
            quota_info = self.Query_quota()
            logger.info("Get successfully")
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting CSBS quota information, errMsg: {}".format(e))
        backup = None
        copy = None
        for quota in quota_info["quotas"]["resources"]:
            if quota["type"] == "backup_capacity":
                if quota["quota"] == 0:
                    logger.info("The CSBS backup capacity is 0 and cannot be used normally")
                    backup = 1
                else:
                    logger.info("the CSBS backup capacity is {quota}M, "
                                "and can be used normally".format(quota=quota["quota"]))
                    pass
            if quota["type"] == "copy_capacity":
                if quota["quota"] == 0:
                    logger.info("The CSBS replication capacity is 0 and cannot be used normally")
                    copy = 1
                else:
                    logger.info("the CSBS replication capacity is {quota}M, "
                                "and can be used normally".format(quota=quota["quota"]))
                    pass
        if backup is not None:
            m = CSBSBaseAWAdapter(self.project_id)
            m.login_iam()

    def Querying_policy_information(self):
        """获取某租户下所有CSBS策略信息

        :return:
        """
        try:
            logger.info("Start getting all CSBS policy information under this tenant")
            info = self.get_plan_info()
            logger.info("Succeed getting all CSBS policy information:{i}".format(i=info))
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting CSBS policy information, errMsg:{}".format(e))
        return str(info)

    def Delete_plan_CSBS(self, plan_id):
        """删除策略

        :return:
        """
        i = 1
        while i < 7:
            try:
                logger.info("Delete policy {i} times".format(i=i))
                self.Delete_plan(plan_id)
                logger.info("Succeed deleting policy")
                time.sleep(5)
                i = 8
            except:
                logger.info("Failed deleting policy")
                time.sleep(40)
                i = i + 1

    def Check_plan(self):
        """检查租户下是否有策略存在，如果存在的话，返回策略的ID列表

        :return:
        """
        try:
            logger.info("Get information of all policies under the tenant")
            re = self.get_plan_info()
            logger.info("Succeed getting information of all policies:{i}".format(i=re))
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting information of all policies "
                            "under the tenant, errMsg:{}".format(e))
        plan_id_list = []
        if not re["policies"]:
            logger.info("After checking, there is no policy under the tenant")
            plan_id_list = None
        else:
            logger.info("After checking, there are some policies under the tenant")
            for policies in re["policies"]:
                plan_id_list.append(policies["id"])
            logger.info("All policies ids under tenant: %s" % plan_id_list)
        return plan_id_list

    def Check_setup(self):
        """任务开始前的检查
        1. 检查租户下是否有策略存在，如果有，则提前删除
        2. 检查租户下备份复制空间是否有容量，如果没有，则自动申请
        :return:
        """
        logger.info("Start checking for policies under the tenant and delete them ahead of time")
        plan_info = self.Check_plan()
        if plan_info is None:
            pass
        else:
            logger.info("Start deleting policy...")
            for plan_id in plan_info:
                self.Delete_plan_CSBS(plan_id)
        logger.info("Check if there is capacity in the backup space under the tenant,"
                    " and if not, automatically request")
        self.Check_quota_CSBS()

    def get_product_id_from_name(self, evs_product_name):
        """根据product名字获取product_id

        :param region_name:
        :return:
        """
        code, res = self.show_product_list()
        total = res["total"]
        product_list = res["products"]
        for n in range(0, total):
            if evs_product_name in product_list[n]["name"]:
                logger.info(product_list[n]["product_id"])
                return product_list[n]["product_id"]
        else:
            logger.info("Product not exists")
            raise Exception("Product not exists")

    def get_service_list(self):
        """获取该租户下所有虚拟机信息

        :return:
        """
        url = "https://{console_url}/ecm/rest/v1.0/servers?" \
              "offset=1&not-tags=__type_baremetal&detail=3".format(console_url=self.console_host)
        return self._get(url)

    def get_service_volume_info(self, servers_id):
        """获取虚拟机的卷信息

        :param service_id:
        :return:
        """
        url = "https://{console_url}/ecm/rest/v2/{project_id}/" \
              "servers/{servers_id}/os-volume_attachments".format(console_url=self.console_host,
                                                                  project_id=self.project_id,
                                                                  servers_id=servers_id)
        return self._get(url)

    def get_id_through_name_service(self, server_name):
        """根据虚拟机名字，获取虚拟机ID

        :return: server_id
        """
        server_id = None
        try:
            logger.info("Start getting all vms list information")
            service_list_info = self.get_service_list()
            logger.info("Succeed getting all vms list information")
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting all vms list information, errMsg:{}".format(e))
        for server_info in service_list_info["servers"]:
            if server_info["name"] == server_name:
                server_id = server_info["id"]
        return server_id

    def get_sysvol_info_us_serverid(self, server_id):
        """根据虚拟机ID获取虚拟机系统盘信息id

        :param server_id
        :return: sys_vol_id
        """
        sys_vol_id = None
        sys_vol_name = None
        try:
            logger.info("Start getting the vm's system volume information")
            volattach_info = self.get_service_volume_info(server_id)
            logger.info("Succeed get the vm's system volume information")
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed get the vm's system volume information, errMsg:{}".format(e))
        for vol_info in volattach_info["volumeAttachments"]:
            if vol_info["device"] in ["/dev/vda", "/dev/sda"]:
                sys_vol_id = vol_info["volumeId"]
        logger.info("The system volume ID of the vm {i} is:{j}".format(i=server_id, j=sys_vol_id))
        try:
            logger.info("Start getting information of the volume: {i}".format(i=sys_vol_id))
            vol_info = self.get_vol_info_use_volid(sys_vol_id)
            logger.info("Succeed getting information of the volume")
        except RuntimeError as e:
            logger.info(e)
            raise Exception("Failed getting information of the volume, errMsg:{}")
        sys_vol_name = vol_info["volume"]["name"]
        logger.info("The system volume name of the vm {i} is {j}".format(i=server_id, j=sys_vol_name))
        return sys_vol_id, sys_vol_name

    def get_vol_info_use_volid(self, volume_id):
        """通过volume_id获取vol_info

        :param volume_id:
        :return:
        """
        url = "https://{console_url}/ecm/rest/v2/{project_id}" \
              "/get/volumes/{volume_id}".format(console_url=self.console_host,
                                                project_id=self.project_id,
                                                volume_id=volume_id)
        return self._get(url)

    # 获取虚拟机存储标识id
    def get_vm_storage_tag(self):
        tag_id = ''
        vms_info = self.get_service_list()
        logger.info('Start to get vm storage tag')
        if vms_info['servers']:
            tag_id = vms_info['servers'][0]['tags'][0]
        logger.info('Current storage tag id :{}'.format(tag_id))
        return tag_id

    # 给虚拟机添加存储标识
    def add_storage_tag_to_vm(self, vm_name):
        try:
            tag_id = self.get_vm_storage_tag()
            if not tag_id:
                raise Exception('Failed to get storage tag id')
            vms_info = self.get_service_list()
            vm_id = ''
            if not vms_info['servers']:
                raise Exception('There is no vm in the tenant')
            for vm_info in vms_info['servers']:
                vmName = vm_info['name']
                if vmName == vm_name:
                    if len(vm_info['tags']) == 2 and vm_info['tags'][1] == '__single_storage':
                        logger.info('The vm already has the storage tag')
                        return tag_id
                    vm_id = vm_info['id']
                    break
            if not vm_id:
                raise Exception('Can not find vm from vm_name:{}'.format(vm_name))
            logger.info('Current vm id:{}, vm name:{}'.format(vm_id, vm_name))
            url = 'https://{}/ecm/rest/v1/{}/apicom/{}/tags'.format(self.console_host, self.project_id, vm_id)
            req_data = {"tags": [tag_id, "__single_storage"],
                        "vmName": vm_name,
                        "bindTags": [],
                        "unbindTags": [],
                        "region": self.region}
            result = self._put(url, data=json.dumps(req_data))
            return result.get('tags', [None])[0]
        except Exception as e:
            logger.error('Failed to add storage tag to vm:{},ErrMsg:{}'.format(vm_name, str(e)))
            raise e


# 虚拟机创建及资源清理
class CheckOrderError(Exception):
    pass


class TenantApi(object):
    def __init__(self, project_id):
        self.pro_id = project_id
        region_info = get_region_info(project_id)
        self.region_type = region_info.region_type
        self.region_id = region_info.region_id
        self.region_name = region_info.region_name
        self.az_id = get_az_id(project_id, region_info)
        self.project_id = None
        params = ParamUtil()
        self.user_name = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                           '%s_vdc1_vsm_user1' % self.region_type,
                                                           self.region_id)
        self.password = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                          '%s_sc_user_new_password' % self.region_type,
                                                          self.region_id)
        self.console_host = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                              '%s_console_host' % self.region_type,
                                                              self.region_id)
        self.auth_host = params.get_value_from_cloud_param(project_id, 'Basic_MO_Test',
                                                           '%s_iam_host' % self.region_type,
                                                           self.region_id)
        self.image_name = params.get_value_from_cloud_param(project_id, 'Basic_FS_Test',
                                                            '%s_vm_template_kvm' % self.region_type,
                                                            self.region_id)
        self.flavor_name = params.get_value_from_cloud_param(project_id, 'Basic_FS_Test',
                                                             '%s_kvm_flavor_name1' % self.region_type,
                                                             self.region_id)
        self.vpc_name = VPC_NAME
        self.subnet_name = SUBNET_NAME
        self.keypair_name = KEYPAIR_NAME
        self.server1_name = SERVER1_NAME
        self.session = requests.session()
        self.volume_type_list = self.get_volume_type_list_from_az(self.az_id)

    def set_vm_info(self, vol_type=None):
        return dict(name=self.server1_name, az_id=self.az_id, vol_type=vol_type)

    def init_request_param(self):
        try:
            logger.info('start init request params')
            logion_obj = Login_SC(self.pro_id, self.region_id, self.region_type)
            self.user_id, self.project_id, self.region_id, self.bak_global_headers, self.global_headers = \
                logion_obj.login_sc_on_web(self.console_host, self.auth_host, self.user_name, self.password,
                                           self.region_name)
        except Exception as e:
            logger.error('fail to init request params,errMsg:{}'.format(str(e)))
            raise

    # 根据az_id获取卷类型
    def get_random_volume_type_from_az(self, az_id):
        try:
            # 初始化请求参数
            self.init_request_param()
            url = 'https://{}/ecm/rest/v1/resource-tags/volume_type' \
                  '?availability_zone-any={}&not-hybrid_filter=false'.format(self.console_host, az_id)
            ret = self.session.get(url=url, headers=self.global_headers, verify=False)
            logger.info('The current request status:[{}]'.format(str(ret.status_code)))
            if ret.status_code != 200:
                raise Exception('Fail to send request, errMsg:[{}]'.format(str(ret.text)))
            for vol_type_info in json.loads(ret.text)['resources']:
                vol_type_name_list = vol_type_info['tags']['name']
                if vol_type_name_list:
                    vol_type_name = vol_type_name_list[0]
                    return vol_type_name
            raise Exception('Fail to get volume type from az, detail info:[{}]'.format(str(ret.text)))
        except Exception as e:
            raise e

    # 根据az_id获取卷类型列表
    def get_volume_type_list_from_az(self, az_id):
        simple_vol_type_list = []
        try:
            # 初始化请求参数
            self.init_request_param()
            url = 'https://{}/ecm/rest/v1/resource-tags/volume_type' \
                  '?availability_zone-any={}&not-hybrid_filter=false'.format(self.console_host, az_id)
            ret = self.session.get(url=url, headers=self.global_headers, verify=False)
            logger.info('The current request status:[{}]'.format(str(ret.status_code)))
            if ret.status_code != 200:
                raise Exception('Fail to send request, errMsg:[{}]'.format(str(ret.text)))
            for vol_type_info in json.loads(ret.text)['resources']:
                vol_type_name_list = vol_type_info['tags']['name']
                if vol_type_name_list:
                    vol_type_name = vol_type_name_list[0]
                    simple_vol_type_list.append(vol_type_name)
            if not simple_vol_type_list:
                raise Exception('Fail to get volume type from az, detail info:[{}]'.format(str(ret.text)))
        except Exception as e:
            raise e
        return simple_vol_type_list

    # 根据镜像名获取镜像信息
    def get_image_info_by_name(self, image_name):
        logger.info("start get image info by image name: %s" % image_name)
        url = "https://%s/ecm/rest/v2/ims/combine/images?__support_kvm=true&__imagetype=gold" \
              "&status=active&sort_key=name&sort_dir=asc"
        result = self.session.get(url=url % self.console_host, headers=self.global_headers, verify=False)
        if result.status_code != 200:
            raise Exception(
                "Failed to get image info, status code[%s], response[%s]" % (result.status_code, str(result.text)))
        for image in json.loads(result.text)["images"]:
            logger.info("check image... [%s]" % image["name"])
            if image_name == image["name"]:
                logger.info("find the image, id=%s" % image["id"])
                return image
        logger.error("can not find the image, get image info failed.")
        raise Exception("Failed to get image info, image name is %s" % image_name)

    # 根据规格名获取规格信息
    def get_flavor_info_by_name(self, flavor_name):
        logger.info("start get flavor info by flavor name, name is %s" % flavor_name)
        url = "https://%s/ecm/rest/v1/%s/cloudservers/flavors" % (self.console_host, self.project_id)
        result = self.session.get(url=url, headers=self.global_headers, verify=False)
        if result.status_code != 200:
            raise Exception(
                "Failed to get flavor info, status code[%s], response[%s]" % (result.status_code, str(result.text)))
        for flavor in json.loads(result.text)["flavors"]:
            logger.info("check flavor... [%s]" % flavor["name"])
            if flavor_name == flavor["name"]:
                logger.info("find the flavor, id=%s" % flavor["id"])
                return flavor
        logger.error("can not find the flavor, get flavor info failed.")
        raise Exception("Failed to get flavor info, flavor name is %s" % flavor_name)

    # 根据vpc名获取vpc信息
    def get_vpc_info_by_name(self, vpc_name):
        logger.info("start get vpc info by vpc name, name is %s" % vpc_name)
        url = "https://%s/vpc/rest/v1/%s/vpcs?regionId=%s" % (self.console_host, self.project_id, self.region_id)
        result = self.session.get(url=url, headers=self.global_headers, verify=False)
        if result.status_code != 200:
            raise Exception(
                "Failed to get vpc info, status code[%s], response[%s]" % (result.status_code, str(result.text)))
        for vpc in json.loads(result.text)["vpcs"]:
            logger.info("check vpc... [%s]" % vpc["name"])
            if vpc_name == vpc["name"]:
                logger.info("find the vpc. id=%s" % vpc["id"])
                return vpc
        logger.error("can not find the vpc, get vpc info failed.")
        return None

    # 获取外部网络信息
    def get_external_network(self):
        logger.info("start get external network.")
        url = "https://%s/vpc/rest/v2/%s/vpc_external_networks" % (self.console_host, self.project_id)
        result = self.session.get(url=url, headers=self.global_headers, verify=False)
        if result.status_code != 200:
            logger.error("Failed to get external network, status code[%s], response[%s]" % (
                result.status_code, str(result.text)))
            raise Exception("Failed to get external network, status code[%s], response[%s]" % (
                result.status_code, str(result.text)))
        vpc_networks = json.loads(result.text)["vpc_external_networks"]
        has_dummy_vpc = get_dummy_vpc(vpc_networks)
        for net in vpc_networks:
            if has_dummy_vpc and net.get("name") != "dummy_external_network":
                continue
            logger.info("return external network , id=%s" % net["id"])
            return net
        logger.error("can not find the external_network, get external_network info failed.")
        raise Exception("Failed to get external network, external network not exists")

    # 根据vpc的子网名获取子网信息
    def get_subnet_info_by_name(self, subnet_name, vpc_id):
        logger.info("start get subnet info by name, name is %s" % subnet_name)
        url = "https://%s/vpc/rest/v1.5/subnets?regionId=%s&vpcId=%s" % (self.console_host, self.region_id, vpc_id)
        # type3类型url不同
        if self.region_type.lower() == 'type3':
            url = "https://%s/vpc/rest/v1.5/subnets?regionId=%s&vpcId=%s&routed=false" % (
                self.console_host, self.region_id, vpc_id)
        result = self.session.get(url=url, headers=self.global_headers, verify=False)
        if result.status_code != 200:
            raise Exception(
                "Failed to get subnet info, status code[%s], response[%s]" % (result.status_code, str(result.text)))
        for subnet in json.loads(result.text)["subnets"]:
            logger.info("check subnet... [%s]" % subnet["name"])
            if subnet_name == subnet["name"]:
                logger.info("find the subnet,subnet_id=[%s]" % subnet['id'])
                return subnet
        logger.info("can not find the subnet, get subnet info failed.")
        return None

    # 根据产品类型获取产品id
    def get_product_id_by_service_type(self, product_type):
        logger.info("start get product id, type is %s" % product_type)
        url = "https://%s/moproductwebsite/goku/rest/product/v3.0/products?" \
              "service_type=%s&limit=8&start=1&region_id=%s&project_id=%s" \
              % (self.console_host, product_type.lower(), self.region_id, self.project_id)
        ret = self.session.get(url=url, headers=self.global_headers, verify=False)
        if ret.status_code != 200:
            logger.error("Failed to get product info, "
                         "status code[%s], response[%s]" % (ret.status_code, str(ret.text)))
            raise Exception("Failed to get product info, "
                            "status code[%s], response[%s]" % (ret.status_code, str(ret.text)))
        logger.info("response product info is %s" % ret.text)
        res = json.loads(ret.text)

        total = res["total"]
        product_list = res["products"]
        for n in range(0, total):
            region_id_value = product_list[n]["region_id"]
            is_need_approval = product_list[n]["approval"]
            is_online = product_list[n]["online_status"]

            # 选择不需要审批的产品
            if region_id_value == self.region_id and \
                    is_need_approval is False and is_online == "online":
                logger.info("get product id successfully, product "
                            "name=%s, id=%s" % (product_list[n]["name"],
                                                product_list[n]["product_id"]))
                return product_list[n]["product_id"]
        else:
            logger.info("%s Product not exist, product_list: %s",
                        (product_type, product_list))
            raise Exception("%s Product not exist", product_type)

    # 获取安全组id
    def get_security_group_id(self):
        logger.info("start get security group id.")
        url = "https://%s/ecm/rest/network/v1/securitygroup" % self.console_host
        result = self.session.get(url=url, headers=self.global_headers, verify=False)
        if result.status_code != 200:
            raise Exception(
                "Failed to get security, status code[%s], response[%s]" % (result.status_code, str(result.text)))
        for group in json.loads(result.text)["security_groups"]:
            logger.info("find a random security, id=%s" % group["id"])
            return group["id"]
        logger.error("can not find the security group.")
        raise Exception("Failed to get security group info")

    # 根据密钥对名获取密钥对信息
    def get_keypair_info_by_name(self, keypair_name):
        """
        Request URL:https://console.t1solution.com/ecm/rest/v2/0db91460cec547e9922140274aa80723/os-keypairs

        """
        url = 'https://{}/ecm/rest/v2/{}/os-keypairs'.format(self.console_host, self.project_id)
        result = self.session.get(url=url, headers=self.global_headers, verify=False)
        if result.status_code != 200:
            raise Exception("Failed to get keypairs, status code[%s], response[%s]"
                            % (result.status_code, str(result.text)))
        for keypair_info in json.loads(result.text)['keypairs']:
            if keypair_info['keypair']['name'] == keypair_name:
                logger.info('get current keypair:%s' % keypair_name)
                return keypair_info['keypair']
        logger.info('can not find the keypair')
        return None

    # 创建密钥对
    def create_keypair(self, keypair_name):
        logger.info('Start to create a keypair')
        keypair_info = self.get_keypair_info_by_name(keypair_name)
        if keypair_info:
            logger.info('the keypair:%s has been exsiting' % keypair_name)
            return keypair_info
        url = 'https://{}/ecm/rest/v2/{}/os-keypairs'.format(self.console_host, self.project_id)
        req_data = {"keypair": {"name": keypair_name, "tenantId": self.project_id}}
        result = self.session.post(url=url, data=json.dumps(req_data), headers=self.global_headers, verify=False)
        if result.status_code != 200:
            raise Exception("Failed to create keypair, status code[%s], response[%s]"
                            % (result.status_code, str(result.text)))
        logger.info('succeed to create the keypair:%s' % keypair_name)
        return json.loads(result.text)['keypair']

    # 删除秘钥对
    def delete_keypair(self, keypair_name):
        logger.info('Start to delete a keypair')
        keypair_info = self.get_keypair_info_by_name(keypair_name)
        if not keypair_info:
            logger.info('the keypair:%s is not exsiting' % keypair_name)
            return True
        url = 'https://{}/ecm/rest/v2/{}/os-keypairs/{}'.format(self.console_host, self.project_id, keypair_name)
        result = self.session.delete(url=url, headers=self.global_headers, verify=False)
        if result.status_code not in [200, 202]:
            raise Exception("Failed to delete keypair, status code[%s], response[%s]"
                            % (result.status_code, str(result.text)))
        logger.info('succeed to delete the keypair:%s' % keypair_name)
        return True

    # 创建vpc
    def create_vpc(self):
        logger.info("start apply vpc, vpc name is %s" % self.vpc_name)
        current_vpc_id = self.get_vpc_info_by_name(self.vpc_name)
        if current_vpc_id is not None:
            logger.info("vpc already exists, no need create, vpc id=%s" % current_vpc_id)
            return True
        product_id = self.get_product_id_by_service_type(VPC_PRODUCT_TYPE)
        if self.region_type.lower() == 'type3':
            params = {"regionId": self.region_id,
                      "tenant_id": self.project_id,
                      "vpc": {"name": self.vpc_name,
                              "ntp": [],
                              "ntp_v6": []
                              },
                      "subnet": {"regionId": self.region_id, "tenantId": self.project_id,
                                 "availableZoneId": "", "name": self.subnet_name, "cidr": '192.168.111.0/24',
                                 "gatewayIp": '192.168.111.1', "dhcpEnable": True,
                                 "physicalNetwork": "physnet1",  # 物理网络
                                 "segmentationId": "", "routed": False,
                                 "dnsList": []
                                 },
                      "display": []
                      }
        else:
            params = {"regionId": self.region_id,
                      "tenant_id": self.project_id,
                      "vpc": {"name": self.vpc_name,
                              "external_gateway_info": {"network_id": self.get_external_network()["id"]}
                              },
                      "subnet": {"regionId": self.region_id, "tenantId": self.project_id,
                                 "availableZoneId": "", "name": self.subnet_name, "cidr": '192.168.111.0/24',
                                 "gatewayIp": '192.168.111.1', "dhcpEnable": True,
                                 "dnsList": []
                                 },
                      "display": []
                      }

        request_body = {"subscriptions": [
            {"operate_type": "apply",
             "product_id": product_id,  # c81cd91207e64ef48663f4cac334f6e2
             "region_id": self.region_id,
             "project_id": self.project_id,
             "service_type": "vpc",
             "tenancy": 0,
             "params": json.dumps(params)}]
        }
        data = json.dumps(request_body)

        length = None
        if 'Content-Length' in self.global_headers:
            length = self.global_headers["Content-Length"]
        self.global_headers.update({'Content-Length': '1145'})
        response = self.session.request(method="post",
                                        url="https://%s/vpc/rest/subscription/v3.0/subscriptions" % self.console_host,
                                        headers=self.global_headers, data=data, verify=False)
        if length is None:
            self.global_headers.pop("Content-Length")
        else:
            self.global_headers.update({"Content-Length": length})
        if response.status_code != 200:
            logger.error(
                "send apply vpc cmd failed, status_code=%s, response=%s" % (
                    response.status_code, str(response.text)))
            raise Exception(
                "Failed to apply vpc, status_code[%s], response[%s]" % (response.status_code, str(response.text)))

        resp = json.loads(response.text)
        sub_id = resp['purchases'][0]['subscription_id']
        logger.info("send command successfully, order id=%s" % sub_id)
        self.monitor_order_status(sub_id)

    # 删除vpc
    def delete_vpc(self):
        logger.info("start delete vpc, vpc name is %s" % self.vpc_name)
        current_vpc = self.get_vpc_info_by_name(self.vpc_name)
        if current_vpc is None:
            logger.info("vpc not exists, no need delete")
            return True

        subnet = self.get_subnet_info_by_name(self.subnet_name, current_vpc["id"])
        if subnet is not None:
            logger.info("start delete subnet. id=%s" % subnet["id"])
            url = "https://%s/vpc/rest/v2/%s/subnets/%s?regionId=%s" % \
                  (self.console_host, self.project_id, subnet["id"], self.region_id)
            response = self.session.delete(url=url, headers=self.global_headers, verify=False)
            if response.status_code >= 400:
                logger.error(
                    "delete subnet failed. status code[%s], response[%s]" % (
                        response.status_code, str(response.text)))
                raise Exception(
                    "delete subnet failed. status code[%s], response[%s]" % (
                        response.status_code, str(response.text)))
        retry = 1
        while True:
            if retry == 3:
                raise Exception('Failed to delete the subnet,please check!')
            time.sleep(3)
            subnet = self.get_subnet_info_by_name(self.subnet_name, current_vpc["id"])
            if subnet is None:
                logger.info('succeed to delete the subnet!')
                break
            retry += 1
            logger.info('wait to delete the subnet,retry num:%s' % retry)

        logger.info("start delete vpc. id=%s" % current_vpc["id"])
        params = {"vpcId": current_vpc["id"],
                  "tenant_id": self.project_id,
                  "regionId": self.region_id,
                  "display": [],
                  "ids": [{"id": current_vpc["id"], "service_type": "vpc"}]}

        request_body = {"subscriptions": [
            {"operate_type": "delete",
             "project_id": self.project_id,
             "region_id": self.region_id,
             "service_type": "vpc",
             "params": json.dumps(params)}
        ]}
        response = self.session.request(method="post",
                                        url="https://%s/vpc/rest/subscription/v3.0/subscriptions" % self.console_host,
                                        headers=self.global_headers, data=json.dumps(request_body), verify=False)
        if response.status_code != 200:
            logger.error(
                "send delete vpc cmd failed, status_code=%s, response=%s" % (
                    response.status_code, str(response.text)))
            raise Exception(
                "Failed to delte vpc, status_code[%s], response[%s]" % (response.status_code, str(response.text)))

        resp = json.loads(response.text)
        sub_id = resp['purchases'][0]['subscription_id']
        logger.info("send command successfully, order id=%s" % sub_id)
        self.monitor_order_status(sub_id)

    # 监控申请与删除订单的状态
    def monitor_order_status(self, order_id):
        logger.info("start monitor order status, order_id: %s" % order_id)
        url = "https://%s/motenantconsolehomewebsite/goku/rest/order/v3.0/orders/%s" % (self.console_host, order_id)
        # 申请服务实例时必须设置该headers, 但检查状态必须删除该headers
        if 'Content-Length' in self.global_headers:
            del self.global_headers['Content-Length']
        retry_times = 0
        while True:
            response = self.session.get(url=url, headers=self.global_headers, verify=False)
            if response.status_code != 200:
                # 如果请求失败, 连续重试6次
                logger.error(
                    "query sub status failed, times[%s], retry after 10s, id[%s], status_code=[%s], resp=[%s]"
                    % (retry_times, order_id, response.status_code, response.text))
                if retry_times == 5:
                    logger.error("retry times out.")
                    raise Exception("Failed to get order status information, order id[%s], status code[%s], "
                                    "response[%s]" % (order_id, response.status_code, response.text))
                time.sleep(5)
                retry_times = retry_times + 1
                continue
            resp = json.loads(response.text)
            if resp['status'] == "successed":
                logger.info("the order status is successed. operation successfylly.")
                return True
            if resp['status'] == "failed":
                detail = self.query_order_detail(order_id)
                logger.error("the order status is failed, detail_info=%s" % detail)
                raise CheckOrderError(
                    "The order status is failed. order id[%s], detail[%s]" % (order_id, detail))

            logger.info("order status is %s, retry check after 20s" % resp['status'])
            time.sleep(20)

    # 查询订单资源详情信息
    def query_order_detail(self, order_id):
        try:
            url = "https://%s/motenantconsolehomewebsite/goku" \
                  "/rest/order/v3.0/orders/%s/resources" % (self.console_host, order_id)
            response = self.session.get(url=url, headers=self.global_headers, verify=False)
            if response.status_code != 200:
                logger.error(
                    "query failed information failed, status_code=%s, resp[%s]" % (
                        response.status_code, response.text))
                raise Exception("Failed to query order detail info, status code: %s, response: %s" % (
                    response.status_code, response.text))
            resp = json.loads(response.text)
            return resp[0]
        except Exception as e:
            logger.error("query order detail failed, errMsg=%s" % str(e))
        return ""

    # 根据虚拟机名称获取虚拟机信息
    def get_vm_info(self, vm_name):
        try:
            logger.info("start get vm info, vm_name=%s" % vm_name)
            url = "https://{console_url}/ecm/rest/v1.0/servers?limit=10" \
                  "&offset=1&not-tags=__type_baremetal&detail=3".format(console_url=self.console_host)
            res = self.session.get(url=url, headers=self.global_headers, verify=False)
            if res.status_code != 200:
                logger.error("failed to get vm list, status_code=%s, content=%s" % (res.status_code, res.text))
                raise Exception("Query vm info by name failed, vm name is %s, status_code[%s], response[%s]"
                                % (vm_name, res.status_code, res.text))
            service_list_info = json.loads(res.text)
            logger.info("get all vm info successfully")
        except Exception as e:
            logger.error("failed to get vm list, errMsg=%s" % str(e))
            raise Exception("Failed to get vm info, errMsg=%s" % str(e))
        for server_info in service_list_info["servers"]:
            if server_info["name"] == vm_name:
                logger.info("get vm info successfully, vm name is %s, id is %s" % (vm_name, server_info["id"]))
                return server_info
        logger.info("can not find the vm, vm name is [%s]" % vm_name)
        return None

    # 构造创建虚拟机请求体
    def get_create_vm_data(self, az_id, vm_name, vol_type):
        vpc_info = self.get_vpc_info_by_name(self.vpc_name)
        if vpc_info is None:
            raise Exception("Failed to get vpc info by name, "
                            "the name is %s" % self.vpc_name)
        subnet_id = self.get_subnet_info_by_name(self.subnet_name,
                                                 vpc_info["id"])["id"]
        if subnet_id is None:
            raise Exception("Failed to get subnet info, "
                            "name is %s" % self.vpc_name)
        keypair_info = self.get_keypair_info_by_name(self.keypair_name)
        if keypair_info is None:
            raise Exception("Failed to get keypair info, "
                            "name is %s" % self.keypair_name)
        image_id = self.get_image_info_by_name(self.image_name)["id"]
        flavor_id = self.get_flavor_info_by_name(self.flavor_name)["id"]
        security_group_id = self.get_security_group_id()
        product_id = self.get_product_id_by_service_type("ecs")
        nics = [{"subnet_id": subnet_id,
                 "ip_address": "",
                 "ip_address_v6": "",
                 "nictype": "",
                 "physical_network": "",
                 "extra_dhcp_opts": [],
                 "binding:profile": {"disable_security_groups": False,
                                     "availability_zone": ""}
                 }]
        params = {"tenantId": self.project_id,
                  "availability_zone": az_id,
                  "name": vm_name,
                  "imageRef": image_id,
                  "flavorRef": flavor_id,
                  "root_volume": {
                      "volumetype": vol_type,
                      "size": 20,
                      "extendparam": {"resourceSpecCode": "",
                                      "resourceType": ""}
                  },
                  "data_volumes": [],
                  "vpcid": vpc_info["id"],
                  "nics": nics,
                  "security_groups": [{"id": security_group_id}],
                  "personality": [],
                  "count": 1,
                  "extendparam": {"chargingMode": 0,
                                  "regionID": self.region_id},
                  "metadata": {"op_svc_userid": "",
                               "__instance_vwatchdog": False,
                               "_ha_policy_type": "remote_rebuild"
                               },
                  "tags": ["__single_storage"],
                  "display": {},
                  "extra": {"devices": [{"device_type": "cdrom"}]}}
        secret_params = {"metadata": {}, "key_name": keypair_info['name']}
        # 构造ecs预计费参数
        rate_params = [{
            "params": [{
                "region_id": self.region_id,
                "az_id": az_id,
                "cloud_service_type_code": "hws.service.type.ebs",
                "resource_type_code": "hws.resource.type.volume",
                "extend_params": 0
            }],
            "unit": {
                "zh_CN": "小时",
                "en_US": "Hour"
            },
            "price": 0.00
        }]
        send_data = {
            "subscriptions": [{"operate_type": "apply",
                               "project_id": self.project_id,
                               "product_id": product_id,
                               "region_id": self.region_id,
                               "service_type": "ecs", "tenancy": "0",
                               "params": json.dumps(params),
                               "rate_params": json.dumps(rate_params),
                               "time_zone": "GMT+08:00",
                               "secret_params": json.dumps(secret_params)}]}
        return send_data

    # 对指定的云服务器上电或下电
    def operate_cloud_server(self, opt_type, vm_name):
        opt_url = "https://%s/ecm/rest/v1/%s/cloudservers/action" % (self.console_host, self.project_id)
        logger.info(" %s cloud server is start, vm name is %s, url is %s" % (opt_type, vm_name, opt_url))
        vm_state = self.get_vm_info(vm_name)
        vm_id = vm_state['id']
        if opt_type == OPT_TYPE_OFF:
            if vm_state['status'] == "SHUTOFF":
                logger.info("the server status is already SHUTOFF, no need stop again.")
                return True
            opt_body = {"os-stop": {"type": "HARD", "servers": [{"id": vm_id}]}}
        elif opt_type == OPT_TYPE_ON:
            if vm_state['status'] == "ACTIVE":
                logger.info("the server status is already ACTIVE, no need start again.")
                return True
            opt_body = {"os-start": {"servers": [{"id": vm_id}]}}
        else:
            raise Exception("Failed to operate cloud server, operation type is error[%s]" % opt_type)

        response = self.session.request(method="post", url=opt_url, headers=self.global_headers,
                                        data=json.dumps(opt_body))
        if response.status_code != 200:
            logger.error("send %s cloud server cmd failed, status_code=%s, response=%s" % (
                opt_type, response.status_code, str(response.text)))
            raise Exception("Failed to %s cloud server, status_code[%s], response[%s]" % (
                opt_type, response.status_code, str(response.text)))
        logger.info("send %s cloud server cmd successfully. start check server status after 10s" % opt_type)
        while True:
            time.sleep(10)
            vmInfo = self.get_vm_info(vm_name)
            if opt_type == OPT_TYPE_OFF:
                if vmInfo['status'] == "SHUTOFF":
                    logger.info("%s cloud server successfully." % opt_type)
                    return True
                if vmInfo['OS-EXT-STS:task_state'] == "powering-off":
                    logger.info("the cloud server is powering-off, retry check after 10s")
                    continue
                logger.error("%s cloud server failed., vm status is %s, task_state is %s" % (
                    opt_type, vmInfo['status'], vmInfo['OS-EXT-STS:task_state']))
                raise Exception("Failed to %s cloud server, server status is %s, task state is %s" % (
                    opt_type, vmInfo['status'], vmInfo['OS-EXT-STS:task_state']))
            if opt_type == OPT_TYPE_ON:
                if vmInfo['status'] == "ACTIVE":
                    logger.info("%s cloud server successfully." % opt_type)
                    return True
                if vmInfo['OS-EXT-STS:task_state'] == "powering-on":
                    logger.info("the cloud server is powering-on, retry check after 10s")
                    continue
                logger.error("%s cloud server failed., vm status is %s, task_state is %s" % (
                    opt_type, vmInfo['status'], vmInfo['OS-EXT-STS:task_state']))
                raise Exception("Failed to %s cloud server, server status is %s, task state is %s" % (
                    opt_type, vmInfo['status'], vmInfo['OS-EXT-STS:task_state']))

    # 创建虚拟机
    def create_vm(self, region_name, vm_list):
        logger.info("start create vm in region[%s]" % region_name)
        order_id_list = []
        created_vm_list = []
        for vm_info in vm_list:
            logger.info("start create vm which named %s" % vm_info["name"])
            vminfo = self.get_vm_info(vm_info["name"])
            if vminfo is not None:
                logger.info("the vm already exists, no need create it.")
                if vminfo["status"].upper() == "SHUTOFF":
                    self.operate_cloud_server(OPT_TYPE_ON, vm_info["name"])
                continue
            body_data = self.get_create_vm_data(vm_info["az_id"],
                                                vm_info["name"],
                                                vm_info["vol_type"])
            ecs_url = "https://%s/ecm/rest/subscription/v3.0/" \
                      "subscriptions" % self.console_host
            response = self.session.request(method="post", url=ecs_url,
                                            headers=self.global_headers,
                                            data=json.dumps(body_data),
                                            verify=False)
            if response.status_code != 200:
                logger.error("send apply vm cmd failed, status_code=%s, "
                             "response=%s" % (response.status_code,
                                              str(response.text)))
                raise Exception(
                    "Failed to create vm, status_code[%s], response[%s]" % (
                        response.status_code, str(response.text)))

            resp = json.loads(response.text)
            sub_id = resp['purchases'][0]['subscription_id']
            logger.info("send command successfully, order id=%s" % sub_id)
            order_id_list.append(sub_id)
            created_vm_list.append(vm_info["name"])
        # 监控订单状态
        for id_ in order_id_list:
            self.monitor_order_status(id_)
        # 监控订单创建成功, 但虚拟机状态异常情况
        for vm_name in created_vm_list:
            logger.info("Start checking created vm's status")
            vminfo = self.get_vm_info(vm_name)
            vm_status = vminfo["status"].upper()
            logger.info("Current vm name:{}, vm status: {}".format(vm_name,
                                                                   vm_status))
            if vm_status != "ACTIVE":
                err_info = 'The order status is successful,' \
                           ' but vm status is {}'.format(vm_status)
                logger.error(err_info)
                raise Exception(err_info)

    # 删除虚拟机
    @retry(retry_nums=3)
    def delete_vm(self, region_name, vm_list):
        logger.info("start delete vm in region[%s], vm list[%s]" % (region_name, vm_list))
        order_list = []
        for vm_info in vm_list:
            logger.info("check vm which named %s" % vm_info["name"])
            vminfo = self.get_vm_info(vm_info["name"])
            if vminfo is None:
                logger.info('the vm:%s is not existing' % vm_info['name'])
                continue
            logger.info('start to delete the vm:%s' % vm_info['name'])

            params = {"count": 1, "display": [], "ids": [{"id": vminfo["id"]}], "delete_publicip": True,
                      "delete_volume": True, "delete_snapshot": True}
            send_data = {"subscriptions": [{"operate_type": "delete", "project_id": self.project_id,
                                            "region_id": self.region_id, "service_type": "ecs", "secret_params": "",
                                            "params": json.dumps(params)}]}
            response = self.session.request(method="post",
                                            url="https://%s/ecm/rest/subscription"
                                                "/v3.0/subscriptions" % self.console_host,
                                            headers=self.global_headers,
                                            data=json.dumps(send_data), verify=False)
            if response.status_code != 200:
                logger.error("send delete vm cmd failed, status_code=%s, response=%s" % (
                    response.status_code, str(response.text)))
                raise Exception(
                    "Failed to delte vm, status_code[%s], response[%s]" % (
                        response.status_code, str(response.text)))

            resp = json.loads(response.text)
            sub_id = resp['purchases'][0]['subscription_id']
            logger.info("send command successfully, order id=%s" % sub_id)
            order_list.append(sub_id)

        for id_ in order_list:
            self.monitor_order_status(id_)
        # 有环境出现订单发送成功 ,虚拟机未删除情况
        for vm_info in vm_list:
            logger.info("check vm which named %s" % vm_info["name"])
            vminfo = self.get_vm_info(vm_info["name"])
            if vminfo is None:
                logger.info('the vm:%s is not existing' % vm_info['name'])
            else:
                raise Exception("The order status is 'successed', "
                                "but the cloud server is not deleted! ")
