import os
import utils.common.log as logger
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts.cbs_params import getVersion
from plugins.CSBS_VBS.common.scripts.cbs_params import is_primary_region
from plugins.CSBS_VBS.common.scripts.cbs_params import is_global_dr_scene
from plugins.CSBS_VBS.common.scripts.cbs_params import is_last_region
from plugins.CSBS_VBS.common.scripts.cbs_apiadapter import API
from plugins.CSBS_VBS.common.scripts.cbs_configfiles import ConfigFiles
from utils.common.exception import FCUException
from platforms.upgradecheck.check_result import CheckResult

logger.init("CSBS-VBS")


class PreCheckConsole(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS-VBS"
        params = Params(project_id, pod_id, self.service_name, self.regionid_list[0])
        self.db_param_dict = params.get_params()
        self.dmk_config_content = {}

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            check_results = []

            if not is_primary_region(project_id) and is_global_dr_scene(project_id):
                # 备Region
                pkg_path, file_name = API.find_file("OceanStor BCManager",
                                                    "_CBS_Console.zip")
                pkg = os.path.join(pkg_path, file_name)
                if not file_name:
                    raise FCUException("645000")

                self.db_param_dict['console_version'] = getVersion(pkg)
                config = ConfigFiles(project_id, pod_id,
                                     self.regionid_list, self.db_param_dict)
                self.dmk_config_content = config.get_console_config_files()
                host_content = self.dmk_config_content["host"]
                vars_content = self.dmk_config_content["config"]

                API.login_dmk(self.db_param_dict['DMK_dmk_floatIp'],
                              self.db_param_dict['dmk_account'],
                              self.db_param_dict['dmk_account_pwd'])
                upgrade_user_id = API.get_dmk_account_id(
                    self.db_param_dict['dmk_account'], "cbs_admin")

                logger.info("login dmk successfully, "
                            "begin to call DMK to check console.")

                # Console服务检查
                ret = API.executedmkdeployment("CBS-Console", 'true',
                                               self.db_param_dict['console_version'],
                                               "8 Service Check",
                                               host_content, vars_content,
                                               upgrade_user_id)
                if ret:
                    logger.info("Check Current Console service successfully.")
                    result = CheckResult(itemname_ch="当前RegionConsole服务检查",
                                         itemname_en="CurrentRegionConsoleServiceCheck",
                                         status="success")
                else:
                    logger.error("Failed to Check Current Console Service.")
                    result = CheckResult(itemname_ch="当前RegionConsole服务检查",
                                         itemname_en="CurrentRegionConsoleServiceCheck",
                                         status="failure",
                                         error_msg_cn="Console服务异常。",
                                         error_msg_en="Console service exception.",
                                         suggestion_cn="请登录Console节点，执行命令sh /etc/console/control.sh status，检查Console服务。",
                                         suggestion_en="Login Console nodes, execute cmd "
                                                       "sh /etc/console/control.sh status to check Console service.")
                check_results.append(result)

            if is_last_region(project_id, regionid_list[0]):
                # last Region
                pkg_path, file_name = API.find_file("OceanStor BCManager",
                                                    "_CBS_Console.zip")
                pkg = os.path.join(pkg_path, file_name)
                if not file_name:
                    raise FCUException("645000")

                self.db_param_dict['console_version'] = getVersion(pkg)
                config = ConfigFiles(project_id, pod_id,
                                     self.regionid_list, self.db_param_dict)
                self.dmk_config_content = config.get_console_config_files()
                host_content = self.dmk_config_content["primary_region_host"]
                vars_content = self.dmk_config_content["primary_region_config"]

                if is_primary_region(project_id):
                    API.login_dmk(self.db_param_dict['DMK_dmk_floatIp'],
                                  self.db_param_dict['dmk_account'],
                                  self.db_param_dict['dmk_account_pwd'])
                else:
                    API.login_dmk(self.db_param_dict['DMK_Global-DMK-Float-IP'],
                                  self.db_param_dict['dmk_account'],
                                  self.db_param_dict['dmk_account_pwd'])
                upgrade_user_id = API.get_dmk_account_id(
                    self.db_param_dict['dmk_account'], "cbs_admin")

                logger.info("login dmk successfully, "
                            "begin to call DMK to check console.")

                # Console服务检查
                ret = API.executedmkdeployment("CBS-Console", 'true',
                                               self.db_param_dict['console_version'],
                                               "8 Service Check",
                                               host_content, vars_content,
                                               upgrade_user_id)
                if ret:
                    logger.info("Check Primary region Console service successfully.")
                    result = CheckResult(itemname_ch="主RegionConsole服务查",
                                         itemname_en="PrimaryRegionConsoleServiceCheck",
                                         status="success")
                else:
                    logger.error("Failed to Check Current Console Service.")
                    result = CheckResult(itemname_ch="主RegionConsole服务检查",
                                         itemname_en="PrimaryRegionConsoleServiceCheck",
                                         status="failure",
                                         error_msg_cn="Console服务异常。",
                                         error_msg_en="Console service exception.",
                                         suggestion_cn="请登录Console节点，执行命令sh /etc/console/control.sh status，检查Console服务。",
                                         suggestion_en="Login Console nodes, execute cmd "
                                                       "sh /etc/console/control.sh status to check Console service.")
                check_results.append(result)

            if not check_results:
                logger.info("not last region and not standby region.")
                ret = CheckResult(itemname_ch="Console服务检查",
                                  itemname_en="ConsoleServiceCheck",
                                  status="success")
                check_results.append(ret)

            return Message(200, check_results=check_results)
        except Exception as e:
            logger.error("Check Console service failed, reason: %s", str(e))
            return Message(500, e, e)

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)
