import os
from configparser import ConfigParser

from utils.business.param_util import ParamUtil
from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.message import Message
from  platforms.param.param_service import ParamService
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client
from utils.common.exception import FCUException
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil

logger.init("CSBS-VBS")
conf_path = os.path.realpath(__file__ + '../../../conf/upgrade.ini')


class RegisterCMDB(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS"
        self.params = Params(project_id, pod_id, self.service_name, self.regionid_list[0])
        vm_names = self.params.get_vmnames()
        vm_ips = []
        for vm_name in vm_names:
            vm_info = ParamService().get_vm_info_by_vm_name(project_id, vm_name)
            vm_ips.append(vm_info[0].get('ip'))
        self.zip_vm_name_ip = list(zip(vm_names, vm_ips))
        self.float_ip = self.get_float_ip()
        conf = ConfigParser()
        conf.read(conf_path)
        self.original_version = conf.get('VERSION', 'original_version')
        self.target_version = conf.get('VERSION', 'target_version')
        self.current_region_dmk_float_ip = ParamUtil().get_param_value(self.pod_id,
                                                                       "DMK",
                                                                       "DMK_dmk_floatIp")
        self.cmdb_util = ManageOneCmdbUtil(project_id)

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            deploy_node_infos = [{"name": vm_name, "ipAddresses": [{"ip": vm_ip}]}
                                 for vm_name, vm_ip in self.zip_vm_name_ip]
            # 要求注册浮动ip信息同部署工程导出lld表中vm信息一致
            for deploy_node in deploy_node_infos:
                if deploy_node['name'].startswith('Service'):
                    float_ip_info = \
                        {'floatIpAddresses': [{'floatIp': self.float_ip,
                                               'fixedIp': deploy_node['ipAddresses'][0]['ip']}]}
                    deploy_node.update(float_ip_info)

            csbs_cloud_service = {"indexName": "CSBS-VBS",
                                  "version": self.target_version,
                                  "originalVersion": self.original_version,
                                  "extendInfos": [{"key": "dmk_floatIp",
                                                   "value": self.current_region_dmk_float_ip}],
                                  "deploy_node_infos": deploy_node_infos
                                  }
            logger.info('Current csbs_cloud_service: {}'.format(csbs_cloud_service))
            logger.info('Start calling cmdb_util.set_cloud_service_info '
                        'to execute registerCMDB ')
            self.cmdb_util.set_cloud_service_info(self.regionid_list[0],
                                                  csbs_cloud_service)

            result = self.cmdb_util.get_cloud_service_info(self.regionid_list[0],
                                                           'CSBS-VBS',
                                                           self.target_version)
            logger.info('Calling get_cloud_service_info to '
                        'get csbs info ,result:{}'.format(result))
            if not result:
                raise Exception('Failed Calling get_cloud_service_info'
                                ' to get csbs info')
        except FCUException as e:
            logger.error("Failed executing registerCMDB, ErrMsg: {}".format(str(e)))
            return Message(500, e)
        except Exception as e:
            logger.error("Failed executing registerCMDB, ErrMsg: {}".format(e))
            return Message(500, e)
        logger.info('Succeed executing registerCMDB')
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def get_float_ip(self):
        karbor1_ip = self.params.get_vm_ips()['karbor_host0']
        params_dict = self.params.get_params()
        djmanager_pwd = params_dict['karbor_djmanager_pwd']
        root_pwd = params_dict['karbor_root_pwd']
        logger.info('Ssh to karbor node: {}'.format(karbor1_ip))
        node = karbor_client.CBSNode(karbor1_ip, 'djmanager',
                                     djmanager_pwd, root_pwd)
        ssh_client = karbor_client.get_sshclient_user_su_root(node)
        cmds = 'get_info.py --tenant_float_ip'
        logger.info('Cmd: {}'.format(cmds))
        result = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        if not karbor_client.is_ssh_cmd_executed(result):
            msg = "Failed to execute cmd: {}".format(cmds)
            logger.error(msg)
            raise Exception(msg)
        logger.info('Get karbor float ip info: {}'.format(result))
        karbor_float_ip = result[0].strip()
        return karbor_float_ip
