import os
from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.message import Message
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts.cbs_params import getVersion
from plugins.CSBS_VBS.common.scripts.cbs_apiadapter import API
from plugins.CSBS_VBS.common.scripts.cbs_configfiles import ConfigFiles
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client
from utils.common.exception import FCUException
from plugins.CSBS_VBS.common.scripts.cbs_params import is_csha_scene


logger.init("CSBS-VBS")


class UpgradeKarbor(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS-VBS"
        params = Params(project_id, pod_id, self.service_name,
                        self.regionid_list[0])
        self.db_param_dict = params.get_params()
        self.dmk_config_content = {}

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            pkg_path, file_name = API.find_file("OceanStor BCManager",
                                                "_CBS_Karbor_Upgrade_X86.zip")
            pkg = os.path.join(pkg_path, file_name)
            if not file_name:
                msg = "can not find Karbor pkg."
                raise Exception(msg)
        except Exception as ex:
            logger.error(ex)
            return Message(500, FCUException(645000))

        try:
            self.db_param_dict['karbor_version'] = getVersion(pkg)
            self.db_param_dict['check_karbor'] = 'false'

            config = ConfigFiles(project_id, pod_id,
                                 self.regionid_list, self.db_param_dict)
            self.dmk_config_content = config.get_karbor_config_files()
            host_content = self.dmk_config_content["host"]
            vars_content = self.dmk_config_content["config"]

            API.login_dmk(self.db_param_dict['DMK_dmk_floatIp'],
                          self.db_param_dict['dmk_account'],
                          self.db_param_dict['dmk_account_pwd'])
            upgrade_user_id = API.get_dmk_account_id(
                self.db_param_dict['dmk_account'], "djmanager")

            logger.info("login dmk successfully, "
                        "begin to call DMK to upgrade Karbor.")

            # csha场景，升级karbor前先卸载仲裁节点的zookeeper
            if is_csha_scene(self.project_id):
                logger.info("This is csha scene, uninstall zookeeper on "
                            "arbitration node brfore upgrade.")
                try:
                    arb_node = karbor_client.ARBNode(
                        self.db_param_dict.get("arb_ip"),
                        self.db_param_dict.get("arb_username"),
                        self.db_param_dict.get("arb_passwd"),
                        self.db_param_dict.get("arb_root_passwd"))
                    ssh_client = karbor_client.get_sshclient_user_su_root(
                        arb_node)
                    cmds = "cd /opt/huawei/dj/inst/services/" \
                           "zookeeper;sh uninstall.sh"
                    result = karbor_client.ssh_exec_command_return(
                        ssh_client, cmds)
                    karbor_client.is_ssh_cmd_executed(result)
                except Exception as err:
                    logger.error("Failed to uninstall zookeeper on "
                                 "arbitration node successfully, "
                                 "reason: %s", str(err))
                    return Message(500, err, err)
                logger.info("Uninstall zookeeper on arbitration "
                            "node successfully.")
                karbor_client.ssh_close(ssh_client)
            # 升级 Karbor
            result = API.executedmkdeployment(
                "Karbor", 'true',
                self.db_param_dict['karbor_version'],
                "[Upgrade] CSBS-VBS Karbor",
                host_content, vars_content,
                upgrade_user_id)
            if result:
                logger.info("Upgrade Karbor successfully.")
            else:
                logger.error("Failed to upgrade Karbor.")
                return Message(500, "升级Karbor失败",
                               "Failed to upgrade Karbor")
        except Exception as e:
            logger.error("Failed to upgrade Karbor, reason: %s", str(e))
            return Message(500, e, e)

        if not self.cert_patch_tool():
            # 即使失败了不阻塞工步只打印日志，真正执行证书替换的时候还有手动解决指导
            logger.error("Patch cert failed, please check manual.")
        # stop karbor operation service
        try:
            self.stop_operation()
        except Exception as e:
            logger.error("Failed to stop karbor-operation service, "
                         "reason: %s", str(e))
            return Message(500, e, e)

        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def stop_operation(self):
        cbs_node = karbor_client.CBSNode(
            self.db_param_dict.get("karbor_host0"), "djmanager",
            self.db_param_dict.get("karbor_djmanager_pwd"),
            self.db_param_dict.get("karbor_root_pwd"))
        ssh_client = karbor_client.get_sshclient_user_su_root(cbs_node)

        cmds = "stop_service --service karbor-operation"
        result = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        ret = karbor_client.is_ssh_cmd_executed(result)
        if not ret:
            msg = "Failed to execute cmd stop_service " \
                  "--service karbor-operation"
            karbor_client.ssh_close(ssh_client)
            logger.error(msg)
            raise Exception(msg)

        logger.info("Stop karbor operation service successfully.")
        karbor_client.ssh_close(ssh_client)

    def cert_patch_tool(self):
        """
        修复karbor统一证书替换csha场景错误。
        只有在FCU6.5.1补丁版本才有这段代码
        :return:
        """
        node0 = karbor_client.CBSNode(
            self.db_param_dict["karbor_host0"], "djmanager",
            self.db_param_dict["karbor_djmanager_pwd"],
            self.db_param_dict["karbor_root_pwd"])
        node1 = karbor_client.CBSNode(
            self.db_param_dict["karbor_host1"], "djmanager",
            self.db_param_dict["karbor_djmanager_pwd"],
            self.db_param_dict["karbor_root_pwd"])
        node_list = [node0, node1]
        old_line = "return karbor_nodes.split(',')"
        new_line = "return karbor_nodes.split(',')[:2] if " \
                   "commands.getoutput('get_info.py --profile') " \
                   "== 'private_recover' else karbor_nodes.split(',')"
        cert_path = "/usr/lib/python2.7/site-packages/kangaroo/dfxextend/" \
                    "common/dfx_utils.py"
        fix_cmd = "docker exec -u root karborapi bash " \
                  "-c \"sed -i \\\"s/%s/%s/g\\\" %s\"" \
                  % (old_line, new_line, cert_path)
        restart_cmd = "karbor-apiControl -A RESTART"
        copy_cmd = "/usr/bin/cp -rf /karbor/CSBS/services/haproxy/" \
                   "cli_client/command_agent.xml " \
                   "/opt/huawei/dj/tools/services/haproxy/command_agent.xml"
        try:
            for node in node_list:
                logger.info('Patch cert on node:%s.' % node.ip)
                ssh_client = karbor_client.get_sshclient(node)
                if not ssh_client:
                    return False
                result = karbor_client.ssh_exec_command_return(ssh_client,
                                                               copy_cmd)
                if not karbor_client.is_ssh_cmd_executed(result):
                    logger.error('Failed Executing: {}'.format(copy_cmd))
                    karbor_client.ssh_close(ssh_client)
                    return False
                result = karbor_client.ssh_exec_command_return(ssh_client,
                                                               fix_cmd)
                if not karbor_client.is_ssh_cmd_executed(result):
                    logger.error('Failed Executing: {}'.format(fix_cmd))
                    karbor_client.ssh_close(ssh_client)
                    return False
                result = karbor_client.ssh_exec_command_return(ssh_client,
                                                               restart_cmd)
                if not karbor_client.is_ssh_cmd_executed(result):
                    logger.error('Failed Executing: {}'.format(restart_cmd))
                    karbor_client.ssh_close(ssh_client)
                    return False
                logger.info('Patch cert on node:%s successfully.' % node.ip)
        except Exception as e:
            logger.error('Execute error: {}'.format(e))
            return False
        return True
