import os
from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.message import Message
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts.cbs_params import getVersion
from plugins.CSBS_VBS.common.scripts.cbs_params import is_csha_scene
from plugins.CSBS_VBS.common.scripts.cbs_apiadapter import API
from plugins.CSBS_VBS.common.scripts.cbs_configfiles import ConfigFiles
from utils.common.exception import FCUException

logger.init("CSBS-VBS")


class UpgradeZK(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS-VBS"
        params = Params(project_id, pod_id, self.service_name,
                        self.regionid_list[0])
        self.db_param_dict = params.get_params()
        self.dmk_config_content = {}

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            # 判断CSHA场景
            if not is_csha_scene(self.project_id):
                logger.info("it not csha scene, no need to upgrade zookeeper.")
                return Message(200)

            pkg_path, file_name = API.find_file("OceanStor BCManager",
                                                "_CBS_Karbor_Upgrade_X86.zip")
            pkg = os.path.join(pkg_path, file_name)
            if not file_name:
                raise FCUException("645000")

            self.db_param_dict['karbor_version'] = getVersion(pkg)
            self.db_param_dict['check_karbor'] = 'false'

            config = ConfigFiles(project_id, pod_id,
                                 self.regionid_list, self.db_param_dict)
            self.dmk_config_content = config.get_karbor_config_files()
            host_content = self.dmk_config_content["host"]
            vars_content = self.dmk_config_content["config"]

            API.login_dmk(self.db_param_dict['DMK_dmk_floatIp'],
                          self.db_param_dict['dmk_account'],
                          self.db_param_dict['dmk_account_pwd'])
            upgrade_user_id = API.get_dmk_account_id(
                self.db_param_dict['dmk_account'], "djmanager")

            logger.info("login dmk successfully, "
                        "begin to call DMK to upgrade Zookeeper.")

            # 升级 Zookeeper
            result = API.executedmkdeployment("Karbor", 'true',
                                              self.db_param_dict['karbor_version'],
                                              "[Upgrade] Zookeeper_arb",
                                              host_content, vars_content,
                                              upgrade_user_id)
            if result:
                logger.info("Upgrade Zookeeper on third-party arbitration node successfully.")
            else:
                logger.error("Failed to upgrade Zookeeper on third-party arbitration node.")
                return Message(500, "升级Karbor失败",
                               "Failed to upgrade Karbor on third-party arbitration node")
        except Exception as e:
            logger.error("Failed to upgrade Zookeeper on third-party arbitration node, reason: %s", str(e))
            return Message(500, e, e)

        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)
