# -*- coding:utf-8 -*-
from utils.business.vm_util import can_vm_pinged_to
from utils.common.ssh_util import Ssh
from utils.common.code2Msg import code2Msg
from utils.common.exception import FCUException
from utils.common.param_check import check_param_ip
from utils.business.param_util import ParamUtil
from utils.security.crypt import decrypt
from platforms.param.param_service import ParamService
from platforms.project.ProjectUtils import get_project_conditions
from platforms.utils.manageone.mo_util import MOOperationPlaneClient
from platforms.upgradecheck.check_result import CheckResult
from .CommonDefine import PATH_VALUE
import os
import zipfile
import utils.common.log as logger
import utils.common.software_package_util as FileUtil

def check_value_null(value):
    if value is None \
            or type(value) is str and value.replace(' ', '').__len__() == 0:
        return True
    return False


def get_data_path():
    local_path = os.path.realpath(__file__ + "/../../data")
    if not os.path.exists(local_path):
        os.makedirs(local_path)
    if not os.path.exists(local_path):
        raise Exception("create data path failed.")
    return local_path + os.path.sep


def close_ssh_client(ssh_client):
    if ssh_client is not None:
        try:
            Ssh.ssh_close(ssh_client)
        except Exception as e:
            logger.error("close ssh client failed, errMsg=%s" % str(e))


def check_network_connection(ip_list):
    check_result_list = []
    try:
        logger.info("start check all host is connected[%s]" % str(ip_list))
        __msg__ = {}
        for ip in ip_list:
            if can_vm_pinged_to(ip) is False:
                logger.error("ping [%s] failed." % ip)
                __msg__.__setitem__("IP", ip)
        if __msg__.__len__() != 0:
            logger.error("check host connected failed[%s]" % __msg__)
            ex = FCUException('665001', __msg__)
            check_result_list.append(CheckResult(
                itemname_ch="检查网络连接",
                itemname_en="Check network connection",
                status="failure",
                error_msg_cn=ex))
    except Exception as e:
        logger.error("check host connected failed, errMsg=%s" % str(e))
        ex = FCUException('665001', str(e))
        check_result_list.append(CheckResult(
            itemname_ch="检查网络连接",
            itemname_en="Check network connection",
            status="failure",
            error_msg_cn=ex))
    return check_result_list


def upload_os_patch_pkg(ip, user, pwd, super_pwd, project_id):
    """
    upload patch package.
    :return:
    """
    logger.info("start upload patch package to %s" % ip)
    ssh_client = None
    try:
        ssh_client = Ssh.ssh_create_client(ip, user, pwd, port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
        Ssh.ssh_send_command(ssh_client, super_pwd, '#', 20)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

        __path__, __fileName__ = FileUtil.find_software_package_by_name(
            "OceanStor BCManager", "_Patch.tar.gz", None, project_id)
        if __path__ and __fileName__:
            file_path = os.path.join(__path__, __fileName__)
        else:
            raise Exception(
                "Failed to upload package to %s, "
                "find_software_package_by_name return false" % ip)

        # 先检查文件是否存在，且是否有上传成功的标签, 都满足则不再重复上传
        tmp_file_name = "is_ok"
        cmds = "[ -f '%s/%s' ] && [ -f '%s/%s' ];echo CMD_RESULT=$?" % (
            PATH_VALUE.PATCH_PACKAGE_PATH, __fileName__,
            PATH_VALUE.PATCH_PACKAGE_PATH, tmp_file_name)
        result = Ssh.ssh_exec_command_return(ssh_client, cmds)
        if str(result).__contains__('CMD_RESULT=0'):
            logger.info("find the file, no need upload.")
            return True

        Ssh.ssh_exec_command_return(
            ssh_client,
            "rm -fr {package_path};mkdir -p {package_path}".format(
                package_path=PATH_VALUE.PATCH_PACKAGE_PATH))
        Ssh.ssh_exec_command_return(ssh_client, "chown %s %s" % (
            user, PATH_VALUE.PATCH_PACKAGE_PATH))

        result = Ssh.put_file(ip, user, pwd, file_path,
                              PATH_VALUE.PATCH_PACKAGE_PATH)
        if result is False:
            logger.error(
                "Failed to upload package to %s, put_file return false" % ip)
            raise Exception(
                "Failed to upload package to %s, put_file return false" % ip)

        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "[ -f '%s/%s' ];echo CMD_RESULT=$?" % (
                PATH_VALUE.PATCH_PACKAGE_PATH,
                __fileName__))
        if not str(result).__contains__("CMD_RESULT=0"):
            logger.error("can not find the package in os[%s]" % ip)
            raise Exception("Can not find the OS patch package in %s" % ip)

        # 如果上传成功了， 则打个标签, 用于标识已上传, 避免重复上传
        Ssh.ssh_exec_command_return(ssh_client, "touch %s/%s" % (
            PATH_VALUE.PATCH_PACKAGE_PATH, tmp_file_name))
    except FCUException as e:
        logger.error("upload pkg failed[%s]" % str(e))
        raise e
    except Exception as e:
        logger.error("Failed to upload pkg, [%s]" % str(e))
        raise e
    finally:
        if ssh_client is not None:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error("close ssh client failed[%s]" % str(e))


def install_os_patch(ip, user, pwd, super_pwd):
    """
    安装OS系统补丁
    :return:
    """
    logger.info("start install patch package on %s" % ip)
    ssh_client = None
    tmp_file = PATH_VALUE.INSTALLED_PATCH_TMP_FILE
    try:
        ssh_client = Ssh.ssh_create_client(ip, user, pwd, port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
        Ssh.ssh_send_command(ssh_client, super_pwd, '#', 20)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

        dest_path = PATH_VALUE.PATCH_PACKAGE_PATH
        package_name = PATH_VALUE.PATCH_PACKAGE_PATH + "/OceanStor*_Patch." \
                                                       "tar.gz"
        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "ls %s;echo CMD_RESULT=$?" % package_name)
        if not str(result).__contains__('CMD_RESULT=0'):
            # 此处找不到补丁包可能是后面已经安装成功并删除了,
            # 找是否存在标识文件，如果存在则返回成功
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "ls '%s';echo CMD_RESULT=$?" % tmp_file)
            if str(result).__contains__('CMD_RESULT=0'):
                logger.info("already installed os patch. "
                            "no need install again.")
                return True
            logger.error("can not find the os patch package.[%s]" % str(result))
            raise Exception("Can not find the OS patch package on %s[%s]" % (
                ip, str(result)))

        tmp_path = "/tmp/os_patch_tmp"
        Ssh.ssh_exec_command_return(ssh_client, "rm -fr %s; mkdir -p %s" % (
            tmp_path, tmp_path))
        Ssh.ssh_exec_command_return(
            ssh_client,
            "cd %s;tar xf OceanStor*_Patch.tar.gz -C %s &>/dev/null" % (
                dest_path, tmp_path))
        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "cd $(dirname `find %s -name 'patch.sh'`);"
            " [ -f 'patch.sh' ];echo CMD_RESULT=$?" % tmp_path)
        if not str(result).__contains__('CMD_RESULT=0'):
            logger.error("can not find the patch.sh script: [%s]" % str(result))
            raise Exception(
                "Can not find the patch.sh on %s [%s]" % (ip, str(result)))
        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "sh patch.sh patch;echo CMD_RESULT=$?")
        if not str(result).__contains__('CMD_RESULT=0'):
            logger.error(
                "Failed to install patch on %s[%s]" % (ip, str(result)))
            raise Exception(
                "Failed to install os patch on %s[%s]" % (ip, str(result)))
        Ssh.ssh_exec_command_return(ssh_client, "rm -fr %s" % dest_path)
        Ssh.ssh_exec_command_return(ssh_client, "rm -fr %s" % tmp_path)
        # 创建一个临时文件标识已成功安装过补丁
        Ssh.ssh_exec_command_return(ssh_client, "touch %s" % tmp_file)
        logger.info("install os patch successfully")
    except FCUException as e:
        logger.error("Failed to install patch[%s]" % str(e))
        raise e
    except Exception as e:
        logger.error("Failed to install patch pkg, [%s]" % str(e))
        raise e
    finally:
        if ssh_client is not None:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error("close ssh client failed[%s]" % str(e))


def get_all_server_nodes(ip, business_user, business_pwd, root_pwd):
    """
    获取所有eReplication Server节点的IP
    :return:
    """
    ssh_client = None
    ip_list = []
    try:
        logger.info("start get all server nodes info in %s" % ip)
        ssh_client = Ssh.ssh_create_client(ip, business_user, business_pwd,
                                           port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", "Password", 10)
        Ssh.ssh_send_command(ssh_client, root_pwd, '#', 10)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

        # 检查是否是单机的命令,从目标文件中搜索级联模式是否是disable,
        # 以及ha模式是single, 单机场景必须同时满足这两种情况
        # 如果搜索结果是iNodeCount=2, 则表示是单机模式, 直接返回连接的节点IP地址
        check_single_cmd = "echo iNodeCount=`cat /opt/BCManager" \
                           "/Runtime/ha/module/hacom/conf/hacom.xml  | " \
                           "grep -E 'cascade.*disable|hamode.*single' | wc -l`"
        check_result = Ssh.ssh_exec_command_return(ssh_client, check_single_cmd)
        if str(check_result).__contains__("iNodeCount=2"):
            logger.info("current server is single mode.")
            return [ip]

        # 非单机模式, 从当前节点查询出对端节点信息
        config_file = "/opt/BCManager/Runtime/ha/module/hasync/conf/hasync.xml"
        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "[ -f '%s' ];echo CMD_RESULT=$?" % config_file)
        logger.info("check exists result [%s]" % str(result))
        if not str(result).__contains__('CMD_RESULT=0'):
            logger.error("can not find the hasync.xml file in %s" % ip)
            raise Exception("Can not find the hasync.xml file in %s" % ip)
        cmd = "echo IP_LIST=`cat %s | grep -E -o '(([0-9]|[1-9][0-9]|" \
              "1[0-9]{2}|2[0-4][0-9]|25[0-5])\.){3}([0-9]|[1-9][0-9]|" \
              "1[0-9]{2}|2[0-4][0-9]|25[0-5])'`" % config_file
        results = Ssh.ssh_exec_command_return(ssh_client, cmd)
        logger.info("get all ip result: %s" % str(results))
        for res in results:
            if res.__contains__('IP_LIST=') is False or res.__contains__(
                    "echo IP_LIST="):
                continue
            logger.info("find the all dest str: %s" % res)
            ips = res.replace('\n', '').split('=')[1].split(' ')
            for ip in ips:
                if check_value_null(ip) is False and ip_list.__contains__(
                        ip) is False:
                    ip_list.append(ip)
            break
        logger.info("get ip successfully, ip lis: %s" % str(ip_list))
    except Exception as e:
        logger.error("get ip list failed, errmsg=%s, ip=%s" % (str(e), ip))
    finally:
        try:
            Ssh.ssh_close(ssh_client)
        except Exception as e:
            logger.error("close ssh client failed, errmsg=%s" % str(e))
    return ip_list


def get_pkg_version(project_id):
    """
    获取当前升级包版本
    :param project_id: 工程id
    :return:
    """
    file_fs = None
    version = None
    try:
        project_dict = get_project_conditions(project_id)
        package_suffix = "eReplication_for_FusionCloudDeploy_X86.zip"
        if project_dict.get("is_x86_scene"):
            logger.info("x86 scene upload server package")
            package_suffix = "eReplication_for_FusionCloudDeploy_X86.zip"
        if project_dict.get("is_arm_scene"):
            logger.info("arm scene upload server package")
            package_suffix = "eReplication_for_FusionCloudDeploy_ARM.zip"
        __path__, __fileName__ = FileUtil.find_software_package_by_name(
            "OceanStor", package_suffix, None, project_id)
        if __path__ and __fileName__:
            __filepath__ = os.path.join(__path__, __fileName__)
            file_fs = zipfile.ZipFile(r'%s' % __filepath__, 'r')
            for f in file_fs.namelist():
                logger.debug("check file name[%s]" % f)
                if f.endswith(".json"):
                    version = f.split("-").__getitem__(1)
                    break
        if version is None:
            logger.error("Failed to get pkg version.")
            raise Exception("Failed to get eReplication package version.")
        logger.info("get package version is %s" % version)
        return version
    except Exception as e:
        logger.error("getVersion failed, errMsg=%s" % str(e))
        raise Exception("Failed to get package version[%s]" % str(e))
    finally:
        if file_fs is not None:
            try:
                file_fs.close()
            except Exception as e:
                logger.error("close fs failed, errMsg=%s" % str(e))


def get_console_pkg_version(project_id):
    """
    :param project_id: 工程id
    :return: console包的版本号
    """
    file_fs = None
    try:

        project_dict = get_project_conditions(project_id)
        package_suffix = "eReplication_Console_X86.zip"
        if project_dict.get("is_x86_scene"):
            logger.info("x86 scene upload server package")
            package_suffix = "eReplication_Console_X86.zip"
        if project_dict.get("is_arm_scene"):
            logger.info("arm scene upload server package")
            package_suffix = "eReplication_Console_ARM.zip"
        __path__, __fileName__ = FileUtil.find_software_package_by_name(
            "OceanStor", package_suffix, None, project_id)
        if __path__ and __fileName__:
            __filepath__ = os.path.join(__path__, __fileName__)
            file_fs = zipfile.ZipFile(r'%s' % __filepath__, 'r')
            for f in file_fs.namelist():
                logger.debug("check file name[%s]" % f)
                if f.endswith(".json"):
                    return f.split("-").__getitem__(1)
    except Exception as e:
        logger.error("getVersion failed, errMsg=%s" % str(e))
        raise Exception("Failed to get package version[%s]" % str(e))
    finally:
        if file_fs is not None:
            try:
                file_fs.close()
            except Exception as e:
                logger.error("close fs failed, errMsg=%s" % str(e))
    return None


def get_current_version(ip, user, pwd, sudo_pwd):
    """
    获取当前待升级的eReplication的版本号信息
    :return:
    """
    ssh_client = None
    version = None
    try:
        ssh_client = Ssh.ssh_create_client(ip, user, pwd, port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
        Ssh.ssh_send_command(ssh_client, sudo_pwd, '#', 20)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)
        results = Ssh.ssh_exec_command_return(
            ssh_client,
            "echo version=`cat /opt/BCManager/Runtime/LegoRuntime/"
            "conf/legoDeploy.conf | grep version= | awk '{print $NF}'`")
        logger.info("get verson result on %s: %s" % (ip, str(results)))
        if not str(results).__contains__("version"):
            logger.error("can not find the version[%s]" % str(results))
            raise Exception("results: %s" % str(results))
        for res in results:
            if res.__contains__('version=') is False or res.__contains__(
                    "echo version"):
                continue
            version = res.replace('\n', '').replace(' ', '').split('=')[1]
            break
        if version is None:
            logger.error("Failed to get eReplication version.")
            raise Exception(
                "Failed to get eReplication version information on %s." % ip)
        return version
    except Exception as e:
        logger.error("get current version failed[%s]" % str(e))
        raise Exception(
            "Failed to get eReplication version information on %s.[%s]" % (
                ip, str(e)))
    finally:
        if ssh_client is not None:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error("close ssh client failed: %s" % str(e))


def get_version_info(ip, user, pwd, sudo_pwd, project_id):
    """
    获取升级前和升级后的eReplication的版本号信息
    :return:
    """
    ssh_client = None
    oldversion = None
    try:
        ssh_client = Ssh.ssh_create_client(ip, user, pwd, port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
        Ssh.ssh_send_command(ssh_client, sudo_pwd, '#', 20)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

        # 获取升级前eReplication版本号
        results = Ssh.ssh_exec_command_return(
            ssh_client,
            "echo version=`cat /opt/BCManager/Runtime/LegoRuntime/conf/"
            "legoDeploy.conf | grep version= | awk '{print $NF}'`")
        logger.info("get old version result on %s: %s" % (ip, str(results)))
        if not str(results).__contains__("version"):
            logger.error("can not find old version[%s]" % str(results))
            raise Exception("results: %s" % str(results))
        for res in results:
            if res.__contains__('version=') is False or res.__contains__(
                    "echo version"):
                continue
            oldversion = res.replace('\n', '').replace(' ', '').split('=')[1]
            break
        if oldversion is None:
            logger.error("Failed to get old version.")
            raise Exception("Failed to get old version information on %s." % ip)

        # 获取当前版本号，也就是升级后eReplication版本号
        newversion = get_pkg_version(project_id)

        info = {"Originalversion": oldversion, "Targetversion": newversion}
        return info
    except Exception as e:
        logger.error("get old or new version failed[%s]" % str(e))
        raise Exception(
            "Failed to get old or new version information on %s.[%s]" % (
                ip, str(e)))
    finally:
        if ssh_client is not None:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error("close ssh client failed: %s" % str(e))


def get_server_node(rd_ip, os_user, os_passwd, root_pwd):
    """
    获取正在提供服务的节点,返回正在提供服务的节点IP,以及服务的监听IP
    :return:
    """
    all_ips = get_all_server_nodes(rd_ip, os_user, os_passwd, root_pwd)
    for ip in all_ips:
        ssh_client = None
        try:
            logger.info("start query service listening ip in %s" % ip)
            ssh_client = Ssh.ssh_create_client(ip, os_user, os_passwd, port=22,
                                               timeout=10)
            Ssh.ssh_send_command(ssh_client, "su - root", "Password", 10)
            Ssh.ssh_send_command(ssh_client, root_pwd, '#', 10)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "ps -ef | grep -w ^Tomcat | "
                "grep BCManager &>/dev/null;echo CMD_RESULT=$? "
                "LISTEN_IP=`cat /opt/BCManager/Runtime/Tomcat6/conf/server.xml"
                " | grep keystorePass | grep -w address | "
                "awk -F 'address=\\\\\"' '{print $2}'  | "
                "awk -F '\\\\\"' '{print $1}'`=")
            # 从CMD_RESULT=x LISTEN_IP=x.x.x.x=中解析服务监听IP
            for res in result:
                if not res.__contains__("CMD_RESULT=0"):
                    continue
                listen_ip = res.split("=")[2]
                logger.info("get listening ip successfully.[%s]" % listen_ip)
                return ip, listen_ip
        except Exception as e:
            logger.error("get primary server node failed.: %s" % str(e))
            raise Exception("Failed to query service ip.[%s]" % str(e))
        finally:
            try:
                if ssh_client is not None:
                    Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error("close ssh client failed: %s" % str(e))
    raise Exception("Please check if eReplication service is running.")


def check_system_pwd(ip_addr, business_user, business_pwd_key, root_pwd_key,
                     check_result, params, true_keys):
    """
    检查系统用户密码
    :param ip_key:
    :param business_user
    :param business_pwd_key:
    :param root_pwd_key:
    :param check_result:
    :param params:
    :param true_keys
    :param ip
    :return:
    """
    if true_keys.__contains__(
            business_pwd_key) and true_keys.__contains__(root_pwd_key):
        # 检查连通性

        if check_param_ip(ip_addr) is False:
            logger.error("the ip address format is error[%s]" % ip_addr)
            check_result.set_check_result(param_keys=[ip_addr], status=500,
                                          error_msg=code2Msg(665007))
        elif can_vm_pinged_to(ip_addr) is False:
            logger.error("can not connect to %s" % ip_addr)
            check_result.set_check_result(param_keys=[ip_addr], status=500,
                                          error_msg=code2Msg(665008))
        else:
            ssh_client = None
            try:

                try:
                    ssh_client = Ssh.ssh_create_client(
                        ip_addr, business_user,
                        decryp_pwd(business_pwd_key, params),
                        port=22, timeout=60)
                    Ssh.ssh_send_command(ssh_client, "su - root", "Password",
                                         20)
                    Ssh.ssh_send_command(ssh_client,
                                         decryp_pwd(root_pwd_key, params), '#',
                                         20)
                    check_result.set_check_result(
                        param_keys=[business_pwd_key, root_pwd_key],
                        status=200, error_msg="")
                    return True
                except Exception as e:
                    logger.error("root password is error, ip=%s, errMsg=%s" % (
                        ip_addr, str(e)))
                    check_result.set_check_result(
                        param_keys=[root_pwd_key], status=500,
                        error_msg=code2Msg(665009))
            except Exception as e:
                logger.error("business pwd is error, errmsg=%s, ip=%s" % (
                    str(e), ip_addr))
                check_result.set_check_result(
                    param_keys=[ip_addr, business_pwd_key], status=500,
                    error_msg=code2Msg(665009))
            finally:
                try:
                    if ssh_client is not None:
                        Ssh.ssh_close(ssh_client)
                except Exception as e:
                    logger.error("close ssh client failed, errmsg=%s" % str(e))
    return False


def decryp_pwd(key, params):
    value = params.get(key)

    # 如果解密成功则直接返回解密后的值
    try:
        decrypted_value = decrypt(value)
        return decrypted_value
    except Exception:
        logger.error("decryp pwd failed, key=%s" % key)

    # 解密失败, 直接返回原value值
    return value


def open_file_upgrade(file_name, mode='r', encoding=None, **kwargs):
    if mode in ['r', 'rt', 'tr'] and encoding is None:
        with open(file_name, 'rb') as f:
            context = f.read()
            for encoding_item in ['UTF-8', 'GBK', 'ISO-8859-1']:
                try:
                    context.decode(encoding=encoding_item)
                    encoding = encoding_item
                    break
                except UnicodeDecodeError:
                    pass
    return open(file_name, mode=mode, encoding=encoding, **kwargs)

def get_server_params(project_id, region_id, service_name):
    params = ParamUtil()
    server_params = {}
    server_params["server_business_user"] = params.get_value_from_cloud_param(
        project_id, service_name, "server_business_user", region_id)
    server_params["server_business_user_pwd"] = \
        params.get_value_from_cloud_param(
        project_id, service_name, "server_business_user_pwd", region_id)
    server_params["server_root_pwd"] = params.get_value_from_cloud_param(
        project_id, service_name, "server_root_pwd", region_id)


    if service_name == PATH_VALUE.CSDR_SERVICE_TYPE:
        vm_info = ParamService().get_vm_info_by_vm_name(project_id,
                                                        "Service-eReplication")
        if not vm_info:
            raise Exception("Failed to get vm info by vm name:{}"
                            .format("Service-eReplication"))
    else:
        vm_info = ParamService().get_vm_info_by_vm_name(project_id,
                                                        "Service-eReplication01")
        if not vm_info:
            raise Exception("Failed to get vm info by vm name:{}"
                            .format("Service-eReplication01"))

    server_params["server_ip1"] = vm_info[0].get('ip')
    all_ips = get_all_server_nodes(server_params['server_ip1'],
                                   server_params["server_business_user"],
                                   server_params["server_business_user_pwd"],
                                   server_params["server_root_pwd"])
    if all_ips.__len__() < 1 or all_ips.__len__() > 2 \
            or all_ips.__contains__(server_params['server_ip1']) is False:
        logger.error(
            "query peer ip from %s failed" % server_params['server_ip1'])

        raise Exception("Failed to get peer ip from server[%s]"
                        % server_params['server_ip1'])
    for ip in all_ips:
        if ip == server_params['server_ip1']:
            continue
        else:
            server_params['server_ip2'] = ip
    return server_params

def get_console_params(project_id, region_id, service_name):
    params = ParamUtil()
    console_params = {}
    vm_info = ParamService().get_vm_info_by_vm_name(project_id,
                                                    "Console-eReplication01")
    if not vm_info:
        raise Exception("Failed to get vm info by vm name:{}"
                        .format("Console-eReplication01"))
    console_params["console_ip1"] = vm_info[0].get('ip')

    vm_info = ParamService().get_vm_info_by_vm_name(project_id,
                                                    "Console-eReplication02")
    if not vm_info:
        raise Exception("Failed to get vm info by vm name:{}"
                        .format("Console-eReplication02"))
    console_params["console_ip2"] = vm_info[0].get('ip')

    console_params["console_business_user"] = \
        params.get_value_from_cloud_param(
        project_id, service_name, "console_business_user", region_id)
    console_params["console_business_user_pwd"] = \
        params.get_value_from_cloud_param(
        project_id, service_name, "console_business_user_pwd", region_id)
    console_params["console_root_pwd"] = params.get_value_from_cloud_param(
        project_id, service_name, "console_root_pwd", region_id)

    return console_params

def get_install_components(project_id, region_id):
    csdr_existed = "False"
    csha_existed = "False"
    vha_existed = "False"
    params_dict = ParamUtil().get_service_cloud_param(project_id,
                                                      'OC',
                                                      region_id)
    iam_ip = params_dict.get("oc_node_ip")
    admin = params_dict.get("oc_common_account")
    common_pwd = params_dict.get("oc_common_pwd")

    mo_client = MOOperationPlaneClient()
    iam_token = mo_client.get_iam_token(iam_ip, admin, common_pwd)
    endpoints_list_cn = mo_client.get_endpoints_com_list(iam_token, iam_ip,
                                                         region_id, "zh-cn")
    endpoints_list_en = mo_client.get_endpoints_com_list(iam_token, iam_ip,
                                                        region_id, "en-us")
    extend_infos = []
    if ("CSDR" in endpoints_list_cn) or ("CSDR" in endpoints_list_en):
        csdr_existed = "True"
    extend_infos.append({"key": "CSDR", "value": csdr_existed})

    if ("CSHA" in endpoints_list_cn) or ("CSHA" in endpoints_list_en):
        csha_existed = "True"
    extend_infos.append({"key": "CSHA", "value": csha_existed})

    if ("VHA" in endpoints_list_cn) or ("VHA" in endpoints_list_en):
        vha_existed = "True"
    extend_infos.append({"key": "VHA", "value": vha_existed})

    return extend_infos