# -*- coding:utf-8 -*-
from utils.common.ssh_util import Ssh
from utils.common.exception import FCUException
from .CommonDefine import PATH_VALUE
from utils.business.dmk_util import DmkApi
from platforms.project.ProjectUtils import get_project_conditions
from platforms.upgradecheck.check_result import CheckResult
import utils.common.software_package_util as FileUtil
import utils.common.log as logger
from . import CommonUtil
import os

logger.init("CSDR")


class ConsoleProcessor:
    """
    处理console升级流程
    """

    def __init__(self, ip, business_user, business_pwd, root_pwd):
        self.ip = ip
        self.business_user = business_user
        self.business_pwd = business_pwd
        self.root_pwd = root_pwd
        # 当前由各个方法中创建ssh_client并释放链接,而不在init方法中直接创建,
        # 因为部分操作执行完成过后可能穿插其它操作导致超时退出，且设置了TMOUT=0也不生效

    def query_console_config(self):
        """
        从Console环境中的/etc/console/confd.conf查询指定的配置信息,
        只能查询key=value形式的配置
        :return:
        """
        logger.info("start query console config in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "echo CONFIG_START;cat /etc/console/confd.conf;echo CONFIG_END")
            config_dit = {}
            for res in result:
                format_str = res.replace('\n', '')
                if format_str.__contains__("=") is False \
                        or format_str.__contains__('CONFIG_START'):
                    continue
                if format_str.__contains__('CONFIG_END'):
                    break
                config_dit.update({format_str.split("=")[0]: format_str[
                                                             format_str.index(
                                                                 "=") + 1:]})
            logger.info("query config successfully: %s" % str(config_dit))
            return config_dit
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def check_console_connection(self):
        """
        检查console环境能否正常访问
        :return:
        """
        return CommonUtil.check_network_connection([self.ip])

    def check_console_env(self):
        """
        检查console环境, 主要是升级前预检查
        :return:
        """
        result_check_list = []
        logger.info("start check env in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "[ -d '%s/huawei' ];echo CMD_RESULT=$?" %
                PATH_VALUE.CONSOLE_BACK_PATH)
            if str(result).__contains__('CMD_RESULT=0'):
                logger.error("back path is already exists, need user confirm.")
                ex = FCUException('665003', PATH_VALUE.CONSOLE_BACK_PATH,
                                  self.ip)
                result_check_list.append(
                    CheckResult(
                        itemname_ch="检查备份文件[%s]" % self.ip,
                        itemname_en="Check the backup file[%s]" % self.ip,
                        status="failure",
                        error_msg_cn=ex))
            else:
                result_check_list.append(
                    CheckResult(
                        itemname_ch="检查备份文件[%s]" % self.ip,
                        itemname_en="Check the backup file[%s]" % self.ip,
                        status="success"))

            # 检查磁盘空间是否足够
            result_list = Ssh.ssh_exec_command_return(
                ssh_client,
                "echo checkresult=`df -m /opt | sed -n '2p' | "
                "awk '{print $4,$6}' | sed 's/ /:/g'`=")

            for result in result_list:
                if str(result).__contains__("checkresult=") and str(
                        result).__contains__("df -m") is False:
                    info = str(result).split("=")[1].split(":")
                    ava_size = info[0]
                    disk = info[1]
                    if int(ava_size) < 2096:
                        logger.error(
                            "check disk space result: available size: %s, "
                            "disk: %s" % (ava_size, disk))
                        exception = FCUException('675010', self.ip, disk, "2")
                        result_check_list.append(
                            CheckResult(itemname_ch="检查磁盘空间",
                                        itemname_en="Check disk space",
                                        status="failure",
                                        error_msg_cn=exception))
                    else:
                        result_check_list.append(
                            CheckResult(
                                itemname_ch="检查磁盘空间[%s]" % self.ip,
                                itemname_en="Check disk space[%s]" % self.ip,
                                status="success"))
                    break
        except FCUException as fe:
            logger.error(
                "check runtime environment failed, errMsg=%s" % str(fe))
            result_check_list.append(
                CheckResult(
                    itemname_ch="检查运行环境[%s]" % self.ip,
                    itemname_en="Check runtime environment[%s]" % self.ip,
                    status="failure",
                    error_msg_cn=fe))
        except Exception as e:
            logger.error("check runtime environment failed, errMsg=%s" % str(e))
            ex = FCUException('665002', str(e))
            result_check_list.append(
                CheckResult(
                    itemname_ch="检查运行环境[%s]" % self.ip,
                    itemname_en="Check runtime environment[%s]" % self.ip,
                    status="failure",
                    error_msg_cn=ex))
        finally:
            CommonUtil.close_ssh_client(ssh_client)
        return result_check_list

    def backup_console_runtime(self):
        """
        备份Console的运行目录/opt/huawei
        :return:
        """
        logger.info("start backup console in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /etc/console/;sh control.sh stop")
            if str(result).__contains__("is not running") is False and str(
                    result).__contains__('stopped successfully') is False:
                logger.error(
                    "shutdown service failed, result: %s" % str(result))
                raise Exception(
                    "Failed to shutdown console service[%s]" % str(result))

            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -fr %s;mkdir -p %s;cp -frp %s %s;echo CMD_RESULT=$?"
                % (
                    PATH_VALUE.CONSOLE_BACK_PATH,
                    PATH_VALUE.CONSOLE_BACK_PATH,
                    PATH_VALUE.CONSOLE_SRC_PATH,
                    PATH_VALUE.CONSOLE_BACK_PATH))
            if not str(result).__contains__("CMD_RESULT=0"):
                logger.error("backup runtime file failed[%s]" % str(result))
                raise Exception(str(result))

            # 备份jre
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cp -frp %s %s;echo CMD_RESULT=$?" % (
                    PATH_VALUE.CONSOLE_JRE_PATH,
                    PATH_VALUE.CONSOLE_BACK_PATH))
            if not str(result).__contains__("CMD_RESULT=0"):
                logger.error("backup jre failed[%s]" % str(result))
                raise Exception(str(result))

            logger.info("backup runtime path successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def restore_console_runtime(self):
        """
        恢复console的运行目录, 适用于回滚
        :return:
        """
        logger.info("start rollback console in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /etc/console/;sh control.sh stop")
            if str(result).__contains__("is not running") is False and str(
                    result).__contains__('stopped successfully') is False:
                logger.error(
                    "shutdown service failed, result: %s" % str(result))
                raise Exception(
                    "Failed to shutdown console service[%s]" % str(result))

            back_path = PATH_VALUE.CONSOLE_BACK_PATH + '/huawei'
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "[ -d '%s' ];echo CMD_RESULT=$?" % back_path)
            if not str(result).__contains__('CMD_RESULT=0'):
                logger.info("back path is not exists, rollback failed.")
                raise Exception("Can not find the backup data")

            # 对于Console，回滚时,将备份目录回拷贝
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -fr %s;cp -frp %s %s;echo CMD_RESULT=$?" %
                (PATH_VALUE.CONSOLE_SRC_PATH,
                 back_path,
                 PATH_VALUE.CONSOLE_SRC_PATH))
            if not str(result).__contains__("CMD_RESULT=0"):
                logger.error("restore runtime file failed[%s]" % str(result))
                raise Exception(str(result))

            # 恢复jre
            jre_back_path = PATH_VALUE.CONSOLE_BACK_PATH + '/jre'
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -fr %s;cp -frp %s %s;echo CMD_RESULT=$?" %
                (PATH_VALUE.CONSOLE_JRE_PATH,
                 jre_back_path,
                 PATH_VALUE.CONSOLE_JRE_PATH))
            if not str(result).__contains__("CMD_RESULT=0"):
                logger.error("restore jre file failed[%s]" % str(result))
                raise Exception(str(result))

            logger.info("rollback console successfully.")

            # 回滚成功后,删除备份文件
            logger.info("start delete console backup data")
            res = Ssh.ssh_exec_command_return(ssh_client,
                                              "rm -fr %s;echo EXEC_RESULT=$?"
                                              % PATH_VALUE.CONSOLE_BACK_PATH)
            if not str(res).__contains__("EXEC_RESULT=0"):
                logger.error(
                    "delete console backup data failed, output: %s" % str(res))
                raise Exception(str(res))
            logger.info("delete console backup data successfully")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def start_console_service(self):
        """
        启动console服务
        :return:
        """
        logger.info("start console in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(
               self.ip, self.business_user, self.business_pwd, port=22,
               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /etc/console/;sh control.sh start")
            if str(result).__contains__("service is running") or str(
                    result).__contains__('successfully'):
                logger.info("start successfully")
                return True
            raise Exception(str(result))
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def stop_console_service(self):
        """
        停止console服务
        :return:
        """
        logger.info("stop console service in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /etc/console;sh control.sh stop")
            if str(result).__contains__("is not running") is False and str(
                    result).__contains__('stopped successfully') is False:
                logger.error("Failed to stop console service: %s" % str(result))
                raise Exception(str(result))
            logger.info("stop eReplication console service successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def uninstall_console(self):
        """
        卸载console服务
        :return:
        """
        logger.info("start uninstall console in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "[ -d '/opt/huawei/console/tomcat' ];echo CMD_RESULT=$?")
            if not str(result).__contains__('CMD_RESULT=0'):
                logger.info("can not find the software, no need uninstall.")
                return True

            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /etc/console/;sh uninstall.sh")
            if not str(result).__contains__("unintalled successfully"):
                logger.error(
                    "uninstall console failed, result: %s" % str(result))
                raise Exception(
                    "Failed to uninstall console service in %s[%s]" % (
                        self.ip, str(result)))
            logger.info("uninstall console successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def recopy_iam_cert(self):
        """
        Console升级是重装方式, 因此升级完成后需要恢复IAM证书, IAM证书会在升级前进行备份
        :return:
        """
        logger.info("start recopy iam cert in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

            back_cert_path = PATH_VALUE.CONSOLE_BACK_PATH + \
                             '/huawei/console/tomcat/webapps'
            tmp_dest_path = "WEB-INF/classes/config/iamKeyStore.jks"
            dest_path = "/opt/huawei/console/tomcat/webapps"
            cmds = "cd %s;for name in `ls | grep -v rootkey`; " \
                   "do echo y|cp -frp $name/%s %s/$name/%s || " \
                   " (echo 'restore $name cert failed';exit 1);done" % (
                       back_cert_path, tmp_dest_path, dest_path, tmp_dest_path)
            result = Ssh.ssh_exec_command_return(ssh_client, cmds)
            if str(result).__contains__('cert failed'):
                logger.error("restore iam cert failed[%s]" % str(result))
                raise Exception(str(result))
            logger.info("recopy iam cert successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def upgrade_console_commit(self):
        """
        升级后console提交
        :return:
        """
        logger.info("upgrade submit console in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(
                self.ip, self.business_user, self.business_pwd, port=22,
                timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, "TMOUT=0", '#', 20)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -fr %s;echo EXEC_RESULT=$?" % PATH_VALUE.CONSOLE_BACK_PATH)
            if not str(result).__contains__("EXEC_RESULT=0"):
                logger.error(
                    "delete console backup data failed, output: %s" % str(
                        result))
                raise Exception(str(result))
            logger.info("upgrade submit console successfully")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)


def upload_console_package(dmk_float_ip, dmk_deploy_user, dmk_deploy_pwd,
                           dmk_os_user, dmk_os_pwd, pod_id, project_id):
    """
    上传console的安装包到pkg
    :return:
    """
    logger.info("start upload console package.")

    project_dict = get_project_conditions(project_id)
    package_suffix = "eReplication_Console_X86.zip"
    if project_dict.get("is_x86_scene"):
        logger.info("x86 scene upload server package")
        package_suffix = "eReplication_Console_X86.zip"
    if project_dict.get("is_arm_scene"):
        logger.info("arm scene upload server package")
        package_suffix = "eReplication_Console_ARM.zip"

    __path__, __fileName__ = FileUtil.find_software_package_by_name(
        "OceanStor", package_suffix, None, project_id)
    if __path__ and __fileName__:
        __filepath__ = os.path.join(__path__, __fileName__)
        # 登陆dmk
        dmk_obj = DmkApi()
        dmk_obj.login_dmk(dmk_float_ip, dmk_deploy_user, dmk_deploy_pwd)

        # 获取上传包结果
        __result__ = dmk_obj.upload_pkg_to_dmk(
            dmk_float_ip, dmk_os_user, dmk_os_pwd, __filepath__, pod_id)
        if __result__:
            logger.info("Upload eReplication console package success.")
            return True
        else:
            logger.error("upload_pkg_to_dmk return false")
            raise Exception("upload_pkg_to_dmk return false")
    else:
        logger.error("Find eReplication package failed.")
        raise Exception("find_software_package_by_name return false")
