# -*- coding:utf-8 -*-

from utils.common.message import Message
from utils.business.param_util import ParamUtil
from utils.common.exception import FCUException
from platforms.upgradecheck.check_result import CheckResult
from . import CommonUtil as commutil
from . import PatchUtil as patchutil

import utils.common.log as logger

logger.init("PatchProcessor")


class PatchProcessor:
    """
    处理server升级
    """

    def __init__(self, project_id, regionid_list, service_name):
        self.project_id = project_id
        self.regionid_list = regionid_list
        self.service_name = service_name

    def init_params(self):
        server_prams = commutil.get_server_params(self.project_id,
                                                  self.regionid_list[0],
                                                  self.service_name)
        self.region = self.regionid_list[0]
        self.server_ip1 = server_prams["server_ip1"]
        self.server_business_user = server_prams["server_business_user"]
        self.server_business_user_pwd = server_prams["server_business_user_pwd"]
        self.server_root_pwd = server_prams["server_root_pwd"]

    def install_patch(self):
        try:
            logger.info("start install eReplication patch.")
            # 初始化参数信息
            self.init_params()

            # 安装Server eReplication补丁
            all_ips = commutil.get_all_server_nodes(
                self.server_ip1,
                self.server_business_user,
                self.server_business_user_pwd,
                self.server_root_pwd)
            if all_ips.__len__() < 1 or all_ips.__len__() > 2 \
                    or all_ips.__contains__(self.server_ip1) is False:
                logger.error(
                    "query eReplication ip from %s failed" % self.server_ip1)
                raise Exception(
                    "Querying the peer eReplication node failed from %s"
                    % self.server_ip1)
            for ip in all_ips:
                patchutil.install_server_patch(ip, self.server_business_user,
                                              self.server_business_user_pwd,
                                              self.server_root_pwd,
                                              self.project_id,
                                              self.service_name)

            logger.info("install eReplication patch successfully.")
        except FCUException as fe:
            logger.error(
                "install eReplication patch failed, errMsg=%s" % str(fe))
            return Message(500, fe)
        except Exception as e:
            logger.error(
                "install eReplication patch failed, errMsg=%s" % str(e))
            return Message(500, FCUException('665002', str(e)))
        return Message(200)

    def upload_patch(self):
        try:
            logger.info("start upload patch package.")
            # 初始化参数信息
            self.init_params()

            # 上传到server节点
            all_ips = commutil.get_all_server_nodes(
                self.server_ip1,
                self.server_business_user,
                self.server_business_user_pwd,
                self.server_root_pwd)
            if all_ips.__len__() < 1 or all_ips.__len__() > 2 \
                    or all_ips.__contains__(self.server_ip1) is False:
                logger.error("query eReplication ip from %s failed"
                             % self.server_ip1)
                raise Exception(
                    "Querying the peer eReplication node failed from %s"
                    % self.server_ip1)
            logger.info("start upload patch package")
            for ip in all_ips:
                patchutil.upload_service_patch_pkg(
                    ip, self.server_business_user,
                    self.server_business_user_pwd, self.server_root_pwd,
                    self.project_id)

            logger.info("upload patch package successfully.")
        except FCUException as fe:
            logger.error("upload package failed, errMsg=%s" % str(fe))
            return Message(500, fe)
        except Exception as e:
            logger.error("upload package failed, errMsg=%s" % str(e))
            return Message(500, FCUException('665002', str(e)))
        return Message(200)

    def check_patch_version(self):
        logger.info("start check process.")
        results = []
        # 初始化参数信息
        try:
            self.init_params()

            system_current_version = commutil.get_current_version(
                self.server_ip1,
                self.server_business_user,
                self.server_business_user_pwd,
                self.server_root_pwd)

            patch_info = patchutil.get_patch_info(self.project_id,
                                                  self.service_name)
            patch_version = patch_info[0]['UpdateInfo_en']['Version']["value"]
            if not patch_version.__contains__(system_current_version):
                err_msg_cn = "当前补丁版本%s不匹配系统版本%s" \
                             % (patch_version, system_current_version)
                err_msg_en = "Current patch version %s does not match system version %s." \
                             % (patch_version, system_current_version)
                check = CheckResult(itemname_ch="检查补丁版本",
                                    itemname_en="Check the patch version",
                                    status="failure",
                                    error_msg_cn=err_msg_cn,
                                    error_msg_en=err_msg_en,
                                    suggestion_cn="请使用匹配的补丁包",
                                    suggestion_en="Use the matching patch package.")
                results.append(check)
                return Message(200, check_results=results)
            else:
                check = CheckResult(itemname_ch="检查补丁版本",
                            itemname_en="Check the patch version",
                            status="success")
                results.append(check)

            logger.info("check patch version end.")
            for check_data in results:
                if check_data.get_status() == "failure":
                    # 如果有失败,则返回检查失败
                    return Message(200, check_results=results)
            return Message(200, check_results=results)
        except Exception:
            check = CheckResult(itemname_ch="检查补丁版本",
                                itemname_en="check patch version",
                                status="failure",
                                error_msg_cn="执行异常",
                                error_msg_en="execute failed",
                                suggestion_cn="",
                                suggestion_en="")
            results.append(check)
            return Message(500, check_results=results)


    def rollback_patch(self):
        try:
            logger.info("start install eReplication patch.")
            # 初始化参数信息
            self.init_params()

            # 安装Server eReplication补丁
            all_ips = commutil.get_all_server_nodes(
                self.server_ip1,
                self.server_business_user,
                self.server_business_user_pwd,
                self.server_root_pwd)
            if all_ips.__len__() < 1 or all_ips.__len__() > 2 \
                    or all_ips.__contains__(self.server_ip1) is False:
                logger.error(
                    "query eReplication ip from %s failed" % self.server_ip1)
                raise Exception(
                    "Querying the peer eReplication node failed from %s"
                    % self.server_ip1)
            for ip in all_ips:
                patchutil.rollback_server_patch(ip,
                                                self.server_business_user,
                                                self.server_business_user_pwd,
                                                self.server_root_pwd,
                                                self.project_id,
                                                self.service_name)

            logger.info("install eReplication patch successfully.")
        except FCUException as fe:
            logger.error(
                "install eReplication patch failed, errMsg=%s" % str(fe))
            return Message(500, fe)
        except Exception as e:
            logger.error(
                "install eReplication patch failed, errMsg=%s" % str(e))
            return Message(500, FCUException('665002', str(e)))
        return Message(200)
