# -*- coding:utf-8 -*-
import os
import utils.common.log as logger
import utils.common.software_package_util as FileUtil
import shutil
import zipfile

from utils.common.component_upgrade_common import read_xml
from utils.common.software_package_util import find_software_package
from utils.common.ssh_util import Ssh
from utils.common.exception import FCUException
from .CommonDefine import PATH_VALUE

def get_patch_info(project_id, service_name):
    pkg_name = ""
    pkg_re = "^OceanStor BCManager (.*)_eReplication_for_Euler.zip$"
    # 此处的三个硬编码暂未处理，请开发人员注意
    try:
        # 根据正则找产品包
        return_result = list()
        logger.info("Start looking for the {} product package.".
                    format(service_name))
        pkg_dict, num = find_software_package(pkg_re, project_id)
        if num > 1:
            raise FCUException(675014, pkg_re)
        if not pkg_dict:
            raise FCUException(675015, service_name)
        pkg_name = list(pkg_dict.keys())[0]
        pkg_path = list(pkg_dict.values())[0]
        # 解压产品包中的xml配置文件到pkg临时目录下
        logger.info("Start extracting the xml "
                    "configuration file in the package.")
        xml_file_path, temp_dir_path = unzip_product_pkg(pkg_path,
                                                         pkg_name,
                                                         service_name)
        # 调用解析XML方法
        logger.info("Start getting xml configuration file content.")
        xml_content = read_xml(xml_file_path)
        # 删除临时文件夹
        logger.info("Start cleaning up temporary resource files.")
        shutil.rmtree(temp_dir_path)
        return_result.append(xml_content)
        return return_result
    except Exception as err_msg:
        logger.error('Failed to get the patch upgrade '
                     'information of the component({})'.format(err_msg))
        if isinstance(err_msg, FCUException):
            raise
        raise FCUException(675016, pkg_name, err_msg)


def unzip_product_pkg(pkg_path, pkg_name, service_name):
    """
    解压产品包中的xml配置文件到pkg临时目录下
    :param pkg_path: 工具pkg绝对路径
    :param pkg_name: 产品包名
    :param service_name: 服务名
    :return:
    """
    try:
        xml_file_path = None
        data_file_name = "UpdatePackageManifest.xml"
        # 此处的这个硬编码暂未处理，请开发人员注意
        temp_dir_path = os.path.join(pkg_path, r"{}_temporary".
                                     format(service_name))
        if os.path.exists(temp_dir_path):
            shutil.rmtree(temp_dir_path)
        os.makedirs(temp_dir_path)
        with zipfile.ZipFile(os.path.join(pkg_path, pkg_name), "r") as zip_file:
            for each in zip_file.namelist():
                if each == data_file_name:
                    xml_file_path = zip_file.extract(each, temp_dir_path)
                    break
        if xml_file_path is None:
            raise FCUException(675017, pkg_name)
        return xml_file_path, temp_dir_path
    except Exception as ex:
        logger.error('Unpacking the product package({}) failed. '
                     'Failure reason:{}'.format(service_name, ex))
        raise FCUException(675017, pkg_name)


def upload_service_patch_pkg(ip, user, pwd, super_pwd, project_id):
    """
    upload patch package.
    :return:
    """
    logger.info("start upload patch package to %s" % ip)
    ssh_client = None
    try:
        ssh_client = Ssh.ssh_create_client(ip, user, pwd, port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
        Ssh.ssh_send_command(ssh_client, super_pwd, '#', 20)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

        __path__, __fileName__ = FileUtil.find_software_package_by_name(
            "OceanStor BCManager", "eReplication_for_Euler.zip", None, project_id)
        if __path__ and __fileName__:
            file_path = os.path.join(__path__, __fileName__)
        else:
            raise Exception(
                "Failed to upload to %s, "
                "find_software_package_by_name return false" % (ip))

        # 先检查文件是否存在，且是否有上传成功的标签, 都满足则不再重复上传
        tmp_file_name = "is_ok"
        cmds = "[ -f '%s/%s' ] && [ -f '%s/%s' ];echo CMD_RESULT=$?" % (
            PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH, __fileName__,
            PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH, tmp_file_name)
        result = Ssh.ssh_exec_command_return(ssh_client, cmds)
        if str(result).__contains__('CMD_RESULT=0'):
            logger.info("find the file, no need upload.")
            return True

        Ssh.ssh_exec_command_return(
            ssh_client,
            "rm -fr {package_path};mkdir -p {package_path}".format(
                package_path=PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH))
        Ssh.ssh_exec_command_return(ssh_client, "chown %s %s" % (
            user, PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH))

        result = Ssh.put_file(ip, user, pwd, file_path,
                              PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH)
        if result is False:
            logger.error(
                "Failed to upload package to %s, put_file return false" % ip)
            raise Exception(
                "Failed to upload package to %s, put_file return false" % ip)

        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "[ -f '%s/%s' ];echo CMD_RESULT=$?" % (
                PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH,
                __fileName__))
        if not str(result).__contains__("CMD_RESULT=0"):
            logger.error("can not find the package in server[%s], path[%s]" % (
            ip, PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH))
            raise Exception("Can not find the service patch package in %s" % ip)

        # 如果上传成功了，则打个标签, 用于标识已上传, 避免重复上传
        Ssh.ssh_exec_command_return(ssh_client, "touch %s/%s" % (
            PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH, tmp_file_name))
    except FCUException as e:
        logger.error("upload pkg failed[%s]" % str(e))
        raise e
    except Exception as e:
        logger.error("Failed to upload patch package, [%s]" % str(e))
        raise Exception("Failed to upload patch package to server[%s]" % ip)
    finally:
        if ssh_client is not None:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error("close ssh client failed[%s]" % str(e))


def install_server_patch(ip, user, pwd, super_pwd, project_id, service_name):
    """
    安装eReplication补丁
    :return:
    """
    logger.info("start install eReplication on %s" % ip)
    ssh_client = None

    patch_info = get_patch_info(project_id, service_name)
    patch_version = patch_info[0]['UpdateInfo_en']['Version']["value"]
    tmp_file = PATH_VALUE.INSTALLED_PATCH_TMP_FILE + patch_version
    try:
        ssh_client = Ssh.ssh_create_client(ip, user, pwd, port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
        Ssh.ssh_send_command(ssh_client, super_pwd, '#', 20)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

        dest_path = PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH
        package_name = PATH_VALUE.SERVICE_PATCH_PACKAGE_PATH + "/OceanStor*eReplication_for_Euler.zip"
        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "ls %s;echo CMD_RESULT=$?" % package_name)
        if not str(result).__contains__('CMD_RESULT=0'):
            # 此处找不到补丁包可能是后面已经安装成功并删除了,
            # 找是否存在标识文件，如果存在则返回成功
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "ls '%s';echo CMD_RESULT=$?" % tmp_file)
            if str(result).__contains__('CMD_RESULT=0'):
                logger.info("already installed patch. "
                            "no need install again.")
                return True
            logger.error("can not find the patch package.[%s]" % str(result))
            raise Exception("Can not find the patch package on %s[%s]" % (
                ip, str(result)))

        tmp_path = "/tmp/service_patch_tmp"
        Ssh.ssh_exec_command_return(ssh_client, "rm -fr %s; mkdir -p %s" % (
            tmp_path, tmp_path))
        Ssh.ssh_exec_command_return(
            ssh_client,
            "cd %s;unzip -o OceanStor*eReplication_for_Euler.zip -d %s &>/dev/null" % (
                dest_path, tmp_path))
        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "cd $(dirname `find %s -name 'cliPatch.sh'`);"
            " [ -f 'cliPatch.sh' ];echo CMD_RESULT=$?" % tmp_path)
        if not str(result).__contains__('CMD_RESULT=0'):
            logger.error("can not find the cliPatch.sh script: [%s]" % str(result))
            raise Exception(
                "Can not find the cliPatch.sh on %s [%s]" % (ip, str(result)))

        ret = Ssh.ssh_send_command(ssh_client, "sh cliPatch.sh", "Shut it down now? (y/n):", 20)
        logger.info("After execute sh cliPatch.sh, the response is %s" % ret)

        ret = Ssh.ssh_send_command(ssh_client, "y", "exported and saved? (y/n):", 20)
        logger.info("after shutdown service, the response is %s" % ret)

        ret = Ssh.ssh_send_command(ssh_client, "y", "install the new patch? (y/n):", 100)
        logger.info("After exported configuration, the response is %s" % ret)

        ret = Ssh.ssh_send_command(ssh_client, "y", "immediately? (y/n)", 100)
        logger.info("After ensure install patch, the response is %s" % ret)

        result = Ssh.ssh_send_command(ssh_client, "n", "", 20)
        logger.info("after dont start immediately, result is %s" % result)

        result = Ssh.ssh_exec_command_return(ssh_client, "echo CMD_RESULT=$?")

        if not str(result).__contains__('CMD_RESULT=0'):
            logger.error(
                "Failed to install patch on %s[%s]" % (ip, str(result)))
            raise Exception(
                "Failed to install patch on %s[%s]" % (ip, str(result)))
        Ssh.ssh_exec_command_return(ssh_client, "rm -fr %s" % dest_path)
        Ssh.ssh_exec_command_return(ssh_client, "rm -fr %s" % tmp_path)
        # 创建一个临时文件标识已成功安装过补丁
        Ssh.ssh_exec_command_return(ssh_client, "touch %s" % tmp_file)
        logger.info("install service patch successfully")
    except FCUException as e:
        logger.error("Failed to install patch[%s]" % str(e))
        raise e
    except Exception as e:
        logger.error("Failed to install patch pkg, [%s]" % str(e))
        raise e
    finally:
        if ssh_client is not None:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error("close ssh client failed[%s]" % str(e))


def rollback_server_patch(ip, user, pwd, super_pwd, project_id, service_name):
    """
    卸载eReplication补丁
    :return:
    """
    logger.info("start rollback eReplication patch on %s" % ip)
    patch_info = get_patch_info(project_id, service_name)
    patch_version = patch_info[0]['UpdateInfo_en']['Version']["value"]
    tmp_file = PATH_VALUE.INSTALLED_PATCH_TMP_FILE + patch_version
    ssh_client = None
    try:
        ssh_client = Ssh.ssh_create_client(ip, user, pwd, port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
        Ssh.ssh_send_command(ssh_client, super_pwd, '#', 20)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

        result = Ssh.ssh_exec_command_return(
            ssh_client,
            "cd $(dirname `find %s -name 'rollBackPatch.sh'`);"
            " [ -f 'rollBackPatch.sh' ];echo CMD_RESULT=$?" % "/opt/BCManager/Runtime/bin")
        if not str(result).__contains__('CMD_RESULT=0'):
            logger.error("can not find the rollBackPatch.sh script: [%s]" % str(result))
            raise Exception(
                "Can not find the rollBackPatch.sh on %s [%s]" % (ip, str(result)))

        ret = Ssh.ssh_send_command(ssh_client, "sh rollBackPatch.sh", "want to continue? (y/n):", 20)
        logger.info("After execute sh cliPatch.sh, the response is %s" % ret)

        ret = Ssh.ssh_send_command(ssh_client, "y", "succeeded.", 100)
        logger.info("after shutdown service, the response is %s" % ret)

        result = Ssh.ssh_exec_command_return(ssh_client, "echo CMD_RESULT=$?")

        if not str(result).__contains__('CMD_RESULT=0'):
            logger.error(
                "Failed to rollback patch on %s[%s]" % (ip, str(result)))
            raise Exception(
                "Failed to rollback patch on %s[%s]" % (ip, str(result)))
        Ssh.ssh_exec_command_return(ssh_client, "rm -rf %s" % tmp_file)
        logger.info("rollback service patch successfully")
    except FCUException as e:
        logger.error("Failed to rollback patch[%s]" % str(e))
        raise e
    except Exception as e:
        logger.error("Failed to rollback patch pkg, [%s]" % str(e))
        raise Exception("Failed to rollback patch package on server[%s]" % ip)
    finally:
        if ssh_client is not None:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error("close ssh client failed[%s]" % str(e))


