# -*- coding:utf-8 -*-
import utils.common.software_package_util as FileUtil
import utils.common.log as logger
import os

from utils.common.ssh_util import Ssh
from utils.common.exception import FCUException
from .CommonDefine import PATH_VALUE
from utils.business.dmk_util import DmkApi
from utils.business.param_util import ParamUtil
from platforms.upgradecheck.check_result import CheckResult
from platforms.project.ProjectUtils import get_project_conditions
from utils.DBAdapter.DBConnector import BaseOps
from .ConsoleProcessor import ConsoleProcessor
from . import CommonUtil


logger.init("ServerProcessor")


class ServerProcessor:
    """
    处理server升级
    """

    def __init__(self, ip, business_user, business_pwd, root_pwd):
        self.ip = ip
        self.business_user = business_user
        self.business_pwd = business_pwd
        self.root_pwd = root_pwd

    def check_server_connection(self):
        """
        检查console环境能否正常访问
        :return:
        """
        return CommonUtil.check_network_connection([self.ip])

    def backup_data(self, encrypt_pwd, project_id):
        """
        备份系统配置数据与运行环境
        :return:
        """
        logger.info("start backup dr data in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

            # 升级包版本号与当前环境版本号一致, 说明是新版本,不用升级, 也就不用再备份数据
            if CommonUtil.get_pkg_version(project_id) == \
                    CommonUtil.get_current_version(self.ip,
                                                   self.business_user,
                                                   self.business_pwd,
                                                   self.root_pwd):
                logger.info("The current version equal to upgrade package "
                            "version, no need backup data.")
                return True

            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /opt/BCManager/Runtime/bin/;echo y|sh shutdownSystem.sh")
            if not str(result).__contains__("successfully"):
                logger.error(
                    "shutdown service failed, result: %s" % str(result))
                raise Exception(
                    "Failed to shutdown DR service[%s]" % str(result))

            back_cmd = "cd /opt/BCManager/Runtime/bin;sh backupData.sh"
            Ssh.ssh_send_command(ssh_client, back_cmd,
                                 'Please enter the password', 30)
            dest_str = "Export system configuration data successfully"
            cmd_out = Ssh.ssh_send_command(ssh_client, encrypt_pwd, dest_str,
                                           180)
            data_back_file = ""
            for result in cmd_out:
                if result.__contains__(dest_str) and result.__contains__('/'):
                    result = result.replace('\n', '')
                    data_back_file = result[result.index('/'):]
            if CommonUtil.check_value_null(data_back_file):
                logger.error(
                    "get database config file failed, %s" % str(cmd_out))
                raise Exception(
                    "Failed to obtain DR config data file[%s]" % str(cmd_out))
            logger.info(
                "get DR config data file successfully[%s]" % data_back_file)

            Ssh.ssh_exec_command_return(ssh_client, "rm -fr %s;mkdir -p %s" % (
                PATH_VALUE.DATA_BACK_UP_PATH, PATH_VALUE.DATA_BACK_UP_PATH))

            # 备份系统配置数据
            dest_file = PATH_VALUE.DATA_BACK_UP_PATH + '/' + \
                        PATH_VALUE.BACKUP_DATA_FILE_NAME
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -fr %s;cp -frp %s %s;echo EXEC_RESULT=$?" % (
                    dest_file, data_back_file,
                    dest_file))
            if not str(result).__contains__("EXEC_RESULT=0"):
                logger.error(
                    "back system config data failed, [%s]" % str(result))
                raise Exception(str(result))

            # 备份安装文件RDInstalled.xml
            install_back_file = PATH_VALUE.DATA_BACK_UP_PATH + '/' + \
                                PATH_VALUE.INSTALL_FILE_NAME
            result = Ssh.ssh_exec_command_return(
                ssh_client, "cp -frp %s %s;echo EXEC_RESULT=$?" % (
                    PATH_VALUE.INSTALL_FILE_SRC, install_back_file))
            if not str(result).__contains__("EXEC_RESULT=0"):
                logger.error(
                    "back RDInstalled.xml file failed, [%s]" % str(result))
                raise Exception(str(result))

            # 备份运行环境
            runtime_path_back = PATH_VALUE.DATA_BACK_UP_PATH + '/' + \
                                PATH_VALUE.RUNTIME_BACK_PATH
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -fr %s;cp -frp /opt/BCManager/Runtime %s;"
                "echo EXEC_RESULT=$?" % (runtime_path_back, runtime_path_back))
            if not str(result).__contains__("EXEC_RESULT=0"):
                logger.error("back runtime path failed, [%s]" % str(result))
                raise Exception(str(result))

            logger.info("back eReplication data successfully")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def check_env(self, project_id):
        """
        检查环境信息
        :return:
        """
        result_check_list = []
        logger.info("start check env in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

            # 将升级前的Server版本号保存到临时文件
            results = Ssh.ssh_exec_command_return(
                ssh_client,
                "ls %s;echo checkresult=$?" %
                PATH_VALUE.ORIGINAL_VERSION_FILE)
            if not str(results).__contains__("checkresult=0"):
                logger.info("save old version.")
                Ssh.ssh_exec_command_return(
                    ssh_client,
                    "echo old_version=`cat /opt/BCManager/Runtime/"
                    "LegoRuntime/conf/legoDeploy.conf | grep version= | "
                    "awk '{print $NF}'` > %s" %
                    PATH_VALUE.ORIGINAL_VERSION_FILE)

            # 升级包版本号与当前环境版本号一致, 说明是新版本,不用升级,
            # 也就不用再检查备份数据
            if CommonUtil.get_pkg_version(project_id) == \
                    CommonUtil.get_current_version(self.ip,
                                                   self.business_user,
                                                   self.business_pwd,
                                                   self.root_pwd):
                logger.info(
                    "The current version is equals upgrade package version, "
                    "no need check backup data.")
                result_check_list.append(
                    CheckResult(itemname_ch="检查运行环境[%s]" % self.ip,
                                itemname_en="Check runtime environment[%s]" %
                                            self.ip, status="success"))
                return result_check_list

            # 如果备份文件已经存在,需要用户手工判断数据备份文件是否有用，此处情况较多，
            # 工具无法正常判断，但在一个正常升级流程结束后的数据清理动作会删除该目录
            # 此处如果检查到文件存在，说明上一次数据清理未做或操作失败
            dest_file = PATH_VALUE.DATA_BACK_UP_PATH + '/' + \
                        PATH_VALUE.BACKUP_DATA_FILE_NAME
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "[ -f '%s' ];echo CMD_RESULT=$?" % dest_file)
            if str(result).__contains__("CMD_RESULT=0"):
                logger.error(
                    "the backup data is already exists, please confirm...")
                exception = FCUException('665003', PATH_VALUE.DATA_BACK_UP_PATH,
                                         self.ip)
                result_check_list.append(
                    CheckResult(itemname_ch="检查备份文件[%s]" % self.ip,
                                itemname_en="Check the backup file[%s]" %
                                            self.ip,
                                status="failure",
                                error_msg_cn=exception))
            else:
                result_check_list.append(
                    CheckResult(itemname_ch="检查备份文件[%s]" % self.ip,
                                itemname_en="Check the backup file[%s]" %
                                            self.ip,
                                status="success"))

            # 检查运行环境备份文件是否存在
            runtime_path = PATH_VALUE.DATA_BACK_UP_PATH + '/' + \
                           PATH_VALUE.RUNTIME_BACK_PATH
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "[ -d '%s' ];echo CMD_RESULT=$?" % runtime_path)
            if str(result).__contains__("CMD_RESULT=0"):
                logger.error("the backup runtime path is already exists, "
                             "please confirm...")
                exception = FCUException('665003', PATH_VALUE.DATA_BACK_UP_PATH,
                                         self.ip)
                result_check_list.append(
                    CheckResult(itemname_ch="检查备份文件[%s]" % self.ip,
                                itemname_en="Check the backup file[%s]" %
                                            self.ip,
                                status="failure",
                                error_msg_cn=exception))
            else:
                result_check_list.append(
                    CheckResult(itemname_ch="检查备份文件[%s]" % self.ip,
                                itemname_en="Check the backup file[%s]" %
                                            self.ip,
                                status="success"))

            # 检查磁盘空间是否足够
            result_list = Ssh.ssh_exec_command_return(
                ssh_client,
                "echo checkresult=`df -m /opt/BCManager/ | "
                "sed -n '2p' | awk '{print $4,$6}' | sed 's/ /:/g'`=")
            for result in result_list:
                if str(result).__contains__("checkresult=") and str(
                        result).__contains__("df -m") is False:
                    info = str(result).split("=")[1].split(":")
                    ava_size = info[0]
                    disk = info[1]
                    if int(ava_size) < 4096:
                        logger.error(
                            "check disk space result: available size: %s, "
                            "disk: %s" % (ava_size, disk))
                        exception = FCUException('675010', self.ip, disk, "4")
                        result_check_list.append(
                            CheckResult(itemname_ch="检查磁盘空间",
                                        itemname_en="Check disk space",
                                        status="failure",
                                        error_msg_cn=exception))
                    else:
                        result_check_list.append(
                            CheckResult(itemname_ch="检查磁盘空间[%s]" % self.ip,
                                        itemname_en="Check disk space[%s]" %
                                                    self.ip,
                                        status="success"))

                    break

        except FCUException as fe:
            logger.error(
                "check runtime environment failed, errMsg=%s" % str(fe))
            result_check_list.append(
                CheckResult(itemname_ch="检查运行环境[%s]" % self.ip,
                            itemname_en="Check Runtime Environment[%s]" %
                                        self.ip,
                            status="failure",
                            error_msg_cn=fe))
        except Exception as e:
            logger.error("check runtime environment failed, errMsg=%s" % str(e))
            ex = FCUException('665002', str(e))
            result_check_list.append(
                CheckResult(itemname_ch="检查运行环境[%s]" % self.ip,
                            itemname_en="Check Runtime Environment[%s]" %
                                        self.ip,
                            status="failure",
                            error_msg_cn=ex))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

        return result_check_list

    def rollback_data(self, encrypt_pwd, project_id):
        """
        恢复环境到升级前版本
        :return:
        """
        logger.info("start rollback eReplication in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)

            # 获取系统当前版本号
            system_current_version = CommonUtil.get_current_version(
                self.ip,
                self.business_user,
                self.business_pwd,
                self.root_pwd)

            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /opt/BCManager/Runtime/bin/;echo y|sh shutdownSystem.sh")
            if not str(result).__contains__("successfully"):
                logger.error(
                    "shutdown service failed, result: %s" % str(result))
                raise Exception(
                    "Failed to shutdown DR service[%s]" % str(result))

            # 恢复运行环境
            logger.info("start recovery runtime.")
            package_version = CommonUtil.get_pkg_version(project_id)
            logger.info("package_version is %s, system_current_version is %s" %
                        (package_version, system_current_version))

            # 检查/opt/.FCU_AUTO_BACK/Runtime_Bak和
            # /opt/.FCU_AUTO_BACK/FCDConfigData.zip是否存在
            dest_file = PATH_VALUE.DATA_BACK_UP_PATH + '/' + \
                        PATH_VALUE.BACKUP_DATA_FILE_NAME
            runtime_path_back = PATH_VALUE.DATA_BACK_UP_PATH + '/' + \
                                PATH_VALUE.RUNTIME_BACK_PATH
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "[ -d '%s' ] && [ -f '%s' ];echo CMD_RESULT=$?" % (
                    runtime_path_back,
                    dest_file))
            if not str(result).__contains__('CMD_RESULT=0'):
                # 如果未找到备份数据，且版本号不同，说明此时是重试，
                # 是已经回退过了把备份的数据删除了(因为删除备份数据是最后一步，
                # 很有可能上次失败的原因是删除备份数据成功，但是向FCU返回结果的时候出现
                # 了网络异常，导致上次步骤执行失败，所以现在直接return，不抛出异常）
                # 否则则说明还未回退但是备份的数据不存在
                if package_version != system_current_version:
                    logger.info("can not find the backup data file")
                    return
                else:
                    logger.error(
                        "can not find the backup data file, rollback failed.")
                    raise Exception("Can not find the backup data.")


            # 如果版本号相同，则说明还未回退，则要先删除升级后的资源信息
            if package_version == system_current_version:
                logger.info("start clear resource")
                Ssh.ssh_exec_command_return(ssh_client,
                                            "chattr -iR /opt/BCManager/Runtime/*")
                Ssh.ssh_exec_command_return(ssh_client,
                                            "rm -fr /opt/BCManager/Runtime")
                Ssh.ssh_exec_command_return(ssh_client,
                                            "mkdir -p /opt/BCManager/Runtime")
                Ssh.ssh_exec_command_return(
                    ssh_client, "chown -R ICUser:LEGO /opt/BCManager")
                Ssh.ssh_exec_command_return(ssh_client,
                                            "chmod -R 750 /opt/BCManager")
                Ssh.ssh_exec_command_return(ssh_client,
                                            "rm -fr /opt/BCManager/Runtime/*")
                logger.info("clear resource success")

                result = Ssh.ssh_exec_command_return(
                    ssh_client,
                    "cp -frp %s/* /opt/BCManager/Runtime/;echo EXEC_RESULT=$?" % (
                        runtime_path_back))
                if not str(result).__contains__("EXEC_RESULT=0"):
                    logger.error("restore runtime path failed.")
                    raise Exception(str(result))
                logger.info("cp runtime_path_back success")

            # 检查/home/ICUser/RDInstalled.xml
            install_file = PATH_VALUE.DATA_BACK_UP_PATH + '/' + \
                           PATH_VALUE.INSTALL_FILE_NAME
            result1 = Ssh.ssh_exec_command_return(
                ssh_client,
                "[ -f '%s' ];echo CMD_RESULT=$?" % install_file)
            if not str(result1).__contains__('CMD_RESULT=0'):
                # 如果未找到备份的RDInstalled.xml，且版本号不同，说明此时是重试，
                # 是已经回退过了把备份的数据删除了，不抛出异常
                # 否则则说明还未回退但是备份的数据不存在
                if package_version != system_current_version:
                    logger.info("can not find the backup RDInstalled.xml")
                    return
                else:
                    logger.error("can not find the backup data file "
                                 "RDInstalled.xml, rollback failed.")
                    raise Exception("Can not find the backup data.")

            # 恢复安装文件
            logger.info("start recovery RDInstalled.xml.")
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -f %s;cp -frp %s %s;echo EXEC_RESULT=$?"
                % (PATH_VALUE.INSTALL_FILE_SRC,
                   install_file,
                   PATH_VALUE.INSTALL_FILE_SRC))
            if not str(result).__contains__("EXEC_RESULT=0"):
                logger.error("restore RDInstalled.xml failed.")
                raise Exception(str(result))

            # 恢复系统配置数据
            logger.info("start recovery config data")
            cmd = "cd /opt/BCManager/Runtime/bin;sh import.sh"
            Ssh.ssh_send_command(ssh_client, cmd, 'do you want to continue', 30)
            Ssh.ssh_send_command(ssh_client, "y",
                                 'Please input the restore file', 30)
            Ssh.ssh_send_command(ssh_client, dest_file, "encrypt password", 50)
            Ssh.ssh_send_command(ssh_client, encrypt_pwd,
                                 "Restore successfully", 300)

            logger.info("rollback eReplication successfully.")

            # 回滚成功后,删除备份文件
            logger.info("start delete server backup data")
            res = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -fr %s %s;echo EXEC_RESULT=$?" % (
                    PATH_VALUE.DATA_BACK_UP_PATH,
                    PATH_VALUE.ORIGINAL_VERSION_FILE))
            if not str(res).__contains__("EXEC_RESULT=0"):
                logger.error(
                    "delete server backup data failed, output: %s" % str(res))
                raise Exception(str(res))
            logger.info("delete server backup data successfully")
        except FCUException as fe:
            raise fe
        except Exception as e:
            logger.error(
                "Failed to rollback eReplication config data, ip=%s, "
                "errMsg=%s" % (self.ip, str(e)))
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def stop_service(self):
        """
        信息服务
        :return:
        """
        logger.info("start stop service in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /opt/BCManager/Runtime/bin;echo y|sh shutdownSystem.sh")
            if not str(result).__contains__("shut down successfully"):
                logger.error(
                    "stop eReplication service failed, output: %s" % str(
                        result))
                raise Exception(str(result))
            logger.info("stop eReplication service successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def upgrade_server_submit(self):
        """
        升级后server提交
        :return:
        """
        logger.info("upgrade submit server in %s" % self.ip)
        ssh_client = None
        try:
            ssh_client = Ssh.ssh_create_client(self.ip, self.business_user,
                                               self.business_pwd, port=22,
                                               timeout=60)
            Ssh.ssh_send_command(ssh_client, "su - root", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, self.root_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, "TMOUT=0", '#', 20)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "rm -fr %s %s;echo EXEC_RESULT=$?" % (
                    PATH_VALUE.DATA_BACK_UP_PATH,
                    PATH_VALUE.ORIGINAL_VERSION_FILE))
            if not str(result).__contains__("EXEC_RESULT=0"):
                logger.error(
                    "delete server backup data failed, output: %s" % str(
                        result))
                raise Exception(str(result))
            logger.info("upgrade commit server successfully")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception("IP[%s], ERROR[%s]" % (self.ip, str(e)))
        finally:
            CommonUtil.close_ssh_client(ssh_client)


class UpgradeProcess:
    """
    处理eReplication升级事务
    """

    def __init__(self, pod_id, project_id, region_id, service_type):
        params = ParamUtil()
        app_name = service_type.upper()
        self.pod_id = pod_id
        self.dmk_float_ip = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_floatIp", region_id)
        self.dmk_ui_user = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_ui_username", region_id)
        self.dmk_ui_user_pwd = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_ui_password", region_id)
        self.dmk_depoly_user = params.get_value_from_cloud_param(
            project_id, app_name, "dmk_server_deploy_user", region_id)
        self.dmk_depoly_group = params.get_value_from_cloud_param(
            project_id, app_name, "dmk_user_group_server", region_id)
        self.dmk_deploy_new_pwd = params.get_value_from_cloud_param(
            project_id, app_name, "dmk_deploy_new_pwd", region_id)
        self.dmk_os_user = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_os_business_username",
            region_id)
        self.dmk_os_pwd = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_os_business_user_password",
            region_id)

        server_params = CommonUtil.get_server_params(project_id,
                                                     region_id,
                                                     app_name)
        self.server_ip1 = server_params["server_ip1"]
        self.server_ip2 = server_params["server_ip2"]
        self.vm_server_ssh_user = server_params["server_business_user"]
        self.vm_server_ssh_user_pwd = server_params["server_business_user_pwd"]
        self.vm_server_ssh_user_sudo_pwd = server_params["server_root_pwd"]

        console_params = CommonUtil.get_console_params(project_id,
                                                       region_id,
                                                       app_name)
        self.console_ip1 = console_params["console_ip1"]
        self.console_ip2 = console_params["console_ip2"]
        self.console_business_user = console_params["console_business_user"]
        self.console_business_user_pwd = \
            console_params["console_business_user_pwd"]
        self.console_root_pwd = console_params["console_root_pwd"]

        self.project_id = project_id

    def get_config_content(self, file_path):
        file_object = CommonUtil.open_file_upgrade(file_path)
        content = ""
        try:
            for line in file_object:
                content = "{0}{1}".format(content, line)
        except Exception as e:
            logger.error("get_config_content failed, errMsg=%s" % str(e))
        finally:
            file_object.close()
        return content

    def start_service(self, server_ips, project_id):
        """
        启动服务
        :return:
        """
        logger.info("start service in %s" % str(server_ips))
        if server_ips.__len__() < 1 or server_ips.__len__() > 2:
            logger.error("upgrade service failed, server_ips is empty.")
            raise Exception(
                "Failed to upgrade service, can not find the eReplication ip.")
        pkg_version = CommonUtil.get_pkg_version(project_id)
        logger.info("start upgrade service[%s], dest version is %s" % (
            str(server_ips), pkg_version))
        __dmkObj__ = DmkApi()
        __dmkObj__.login_dmk(self.dmk_float_ip, self.dmk_depoly_user,
                             self.dmk_deploy_new_pwd, "8443")
        self.account_id = __dmkObj__.get_dmk_account_id(self.dmk_depoly_user,
                                                        self.vm_server_ssh_user)
        if CommonUtil.check_value_null(self.account_id):
            raise Exception("get_dmk_account_id return false")

        # 构造节点信息
        self.local_ip = server_ips[0]
        self.peer_ip = ""
        host_vars = self.local_ip
        if len(server_ips) == 2:
            self.peer_ip = server_ips[1]
            host_vars = host_vars + "," + self.peer_ip

        host_info = self.get_config_content(
            os.path.realpath(__file__ + '/../../conf/server_hosts'))
        host_info = host_info.format(host_vars)

        # 构造安装配置信息
        config_info = None
        try:
            config_info, hosts_config, app_version = \
                __dmkObj__.get_dmk_last_deploy_config("eReplication")
        except Exception as e:
            logger.info("get last deploy config failed, err=%s" % str(e))
        if config_info is None:
            config_info = self.get_config_content(
                os.path.realpath(__file__ + '/../../conf/server_config.yml'))
            config_info = config_info.format(
                pkg_version, self.vm_server_ssh_user,
                self.local_ip, self.peer_ip)
        logger.debug("host_info:[%s]" % host_info)
        logger.debug("config_info:[%s]" % config_info)
        __result__ = __dmkObj__.excute_dmk_deployment(
            "eReplication", True, pkg_version, "Start Service",
            host_info, config_info, self.account_id)
        if not __result__:
            logger.error("excute console install failed.")
            raise Exception("excute_dmk_deployment return false")

    def upgrade_service(self, server_ips, project_id):
        """
        升级服务
        :return:
        """
        if server_ips.__len__() < 1 or server_ips.__len__() > 2:
            logger.error("upgrade service failed, server_ips is empty.")
            raise Exception(
                "Failed to upgrade service, can not find the eReplication ip.")
        pkg_version = CommonUtil.get_pkg_version(project_id)
        logger.info("start upgrade service[%s], dest version is %s" % (
            str(server_ips), pkg_version))
        __dmkObj__ = DmkApi()
        __dmkObj__.login_dmk(self.dmk_float_ip, self.dmk_depoly_user,
                             self.dmk_deploy_new_pwd, "8443")
        self.account_id = __dmkObj__.get_dmk_account_id(self.dmk_depoly_user,
                                                        self.vm_server_ssh_user)
        if CommonUtil.check_value_null(self.account_id):
            raise Exception("get_dmk_account_id return false")

        # 构造节点信息
        self.local_ip = ""
        self.peer_ip = ""
        host_vars = None
        local_version = CommonUtil.get_current_version(
            server_ips[0],
            self.vm_server_ssh_user,
            self.vm_server_ssh_user_pwd,
            self.vm_server_ssh_user_sudo_pwd)
        logger.info(
            "get current version on %s is %s." % (server_ips[0], local_version))
        if pkg_version != local_version:
            # 判断环境上的版本号是否与升级包的版本号一致, 如果一致,就不再进行升级操作
            self.local_ip = server_ips[0]
            host_vars = self.local_ip
        if len(server_ips) == 2:
            peer_version = CommonUtil.get_current_version(
                server_ips[1],
                self.vm_server_ssh_user,
                self.vm_server_ssh_user_pwd,
                self.vm_server_ssh_user_sudo_pwd)
            logger.info("get current version on %s is %s." % (
                server_ips[1], peer_version))
            if pkg_version != peer_version:
                self.peer_ip = server_ips[1]
                if host_vars is None:
                    host_vars = self.peer_ip
                else:
                    host_vars = host_vars + "," + self.peer_ip

        # 如果检查结果未找到需要升级的版本, 则直接返回成功
        if host_vars is None:
            logger.info("there is no eReplication need upgrade, return.")
            return True

        # 从当前Region中的Console虚拟机中查询sc域名
        sc_address = None
        for ip in [self.console_ip1, self.console_ip2]:
            consle_proc = ConsoleProcessor(ip, self.console_business_user,
                                           self.console_business_user_pwd,
                                           self.console_root_pwd)
            sc_info = consle_proc.query_console_config()["SC_HOST"]
            # format: sc.type1.com:31943
            if CommonUtil.check_value_null(sc_info) is False \
                    and sc_info.__contains__(":"):
                sc_address = sc_info.split(":")[0]
                break
        if sc_address is None:
            raise Exception("Query configuration file failed")

        host_info = self.get_config_content(
            os.path.realpath(__file__ + '/../../conf/server_hosts'))
        host_info = host_info.format(host_vars)

        # 证书注册地址
        cert_url = sc_address + ":26335"

        # region_list
        region_list = BaseOps().get_regionId_by_projectId(self.project_id)

        # 构造安装配置信息
        config_info = self.get_config_content(
            os.path.realpath(__file__ + '/../../conf/server_config.yml'))
        config_info = config_info.format(
            pkg_version, self.vm_server_ssh_user, self.local_ip,
            self.peer_ip, cert_url, region_list[0])
        logger.info("host_info:[%s]" % host_info)
        logger.info("config_info:[%s]" % config_info)
        __result__ = __dmkObj__.excute_dmk_deployment(
            "eReplication", True, pkg_version, "Upgrade",
            host_info, config_info, self.account_id)
        if not __result__:
            logger.error("excute console install failed.")
            raise Exception("excute_dmk_deployment return false")

        # 升级成功后检查版本号是否是新版本
        v_dic = {}
        for ip in server_ips:
            new_version = CommonUtil.get_current_version(
                ip,
                self.vm_server_ssh_user,
                self.vm_server_ssh_user_pwd,
                self.vm_server_ssh_user_sudo_pwd)
            if new_version != pkg_version:
                v_dic.__setitem__(ip, "current version:%s, dest version: %s" % (
                    new_version, pkg_version))
        if v_dic.__len__() != 0:
            logger.error("upgrade failed,  %s" % str(v_dic))
            raise Exception(str(v_dic))
        logger.info(
            "upgrade eReplication server successfully in all node: %s" % str(
                server_ips))

    def upload_server_package(self, project_id):
        """
        上传Server的安装包到DMK
        :return:
        """
        logger.info("start upload server package.")
        project_dict = get_project_conditions(project_id)
        package_suffix = "eReplication_for_FusionCloudDeploy_X86.zip"
        if project_dict.get("is_x86_scene"):
            logger.info("x86 scene upload server package")
            package_suffix = "eReplication_for_FusionCloudDeploy_X86.zip"
        if project_dict.get("is_arm_scene"):
            logger.info("arm scene upload server package")
            package_suffix = "eReplication_for_FusionCloudDeploy_ARM.zip"

        __path__, __fileName__ = FileUtil.find_software_package_by_name(
            "OceanStor", package_suffix, None, project_id)
        if __path__ and __fileName__:
            __filepath__ = os.path.join(__path__, __fileName__)
            # 登陆dmk
            dmk_obj = DmkApi()
            dmk_obj.login_dmk(self.dmk_float_ip, self.dmk_depoly_user,
                              self.dmk_deploy_new_pwd)

            # 获取上传包结果
            __result__ = dmk_obj.upload_pkg_to_dmk(
                self.dmk_float_ip, self.dmk_os_user, self.dmk_os_pwd,
                __filepath__, self.pod_id)
            if __result__:
                logger.info("Upload eReplication server package success.")
                return True
            else:
                logger.error("upload_pkg_to_dmk return false")
                raise Exception("upload_pkg_to_dmk return false")
        else:
            logger.error("Find eReplication package failed.")
            raise Exception("find_software_package_by_name return false")
