from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.message import Message
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client
from platforms.upgradecheck.check_result import CheckResult

logger.init("CSBS-VBS")
SYS_CHECK_LIST = [
    {"item": "cpuStatus", "contain": "Cpu Idle:", "thresholds": 10,
     "itemname_ch": "CPU检查", "itemname_en": "CPU check",
     "error_msg_cn": "CPU使用率超过90%。",
     "error_msg_en": "The CPU usage exceeds 90%.",
     "suggestion_cn": "请登录Karbor节点，执行命令top，检查CPU使用率。",
     "suggestion_en": "Log in to the Karbor node and run the top "
                      "command to check the CPU usage."},
    {"item": "memory", "contain": "Free Memory:", "thresholds": 1024,
     "itemname_ch": "内存检查", "itemname_en": "Memory check",
     "error_msg_cn": "内存使用率超过80%。",
     "error_msg_en": "The memory usage exceeds 80%.",
     "suggestion_cn": "请登录Karbor节点，执行命令top，检查内存使用率。",
     "suggestion_en": "Log in to the Karbor node and run the top "
                      "command to check the memory usage."},
    {"item": "diskfree", "contain": "Disk Free:", "thresholds": 1024,
     "itemname_ch": "磁盘检查", "itemname_en": "Disk check",
     "error_msg_cn": "磁盘剩余空间不足。",
     "error_msg_en": "The available disk space is insufficient.",
     "suggestion_cn": "请登录Karbor节点，执行命令df -h，检查磁盘剩余空间。",
     "suggestion_en": "Log in to the Karbor node and run the df -h "
                      "command to check the available disk space."}]
CERTS_CHECK_LIST = [
    {"item": "IAM", "contain": "Certificates do not exist", "thresholds": 0,
     "itemname_ch": "IAM证书检查", "itemname_en": "IAM certificate check",
     "error_msg_cn": "IAM证书不存在。",
     "error_msg_en": "No IAM certificate exists.",
     "suggestion_cn": "请登录Karbor节点，执行命令check_karbor_connect，"
                      "查看IAM证书检查结果。",
     "suggestion_en": "Log in to the Karbor node and run the "
                      "check_karbor_connect command to view the "
                      "IAM certificate check result."},
    {"item": "IAM", "contain": "SSLError", "thresholds": 0,
     "itemname_ch": "IAM证书检查", "itemname_en": "IAM certificate check",
     "error_msg_cn": "IAM证书错误。",
     "error_msg_en": "IAM certificate error.",
     "suggestion_cn": "请登录Karbor节点，执行命令check_karbor_connect，"
                      "查看IAM证书检查结果。",
     "suggestion_en": "Log in to the Karbor node and run the "
                      "check_karbor_connect command to view "
                      "the IAM certificate check result."},
    {"item": "Nova", "contain": "Certificates do not exist", "thresholds": 0,
     "itemname_ch": "Nova证书检查", "itemname_en": "Nova certificate check",
     "error_msg_cn": "Nova证书不存在。",
     "error_msg_en": "No Nova certificate exists.",
     "suggestion_cn": "请登录Karbor节点，执行命令check_karbor_connect，"
                      "查看Nova证书检查结果。",
     "suggestion_en": "Log in to the Karbor node and run the "
                      "check_karbor_connect command to view "
                      "the Nova certificate check result."},
    {"item": "Nova", "contain": "SSLError", "thresholds": 0,
     "itemname_ch": "Nova证书检查", "itemname_en": "Nova certificate check",
     "error_msg_cn": "Nova证书错误。",
     "error_msg_en": "Nova certificate error.",
     "suggestion_cn": "请登录Karbor节点，执行命令check_karbor_connect，"
                      "查看Nova证书检查结果。",
     "suggestion_en": "Log in to the Karbor node and run the "
                      "check_karbor_connect command to view "
                      "the Nova certificate check result."}]
GAUSSDB_CHECK_DICT = {
    "itemname_ch": "Gaussdb数据库检查",
    "itemname_en": "Gaussdb database check",
    "error_msg_cn": "数据库状态异常。",
    "error_msg_en": "Database status is abnormal.",
    "suggestion_cn": "请登录Karbor节点，执行命令show_service，"
                     "查看gaussdb数据库检查结果。",
    "suggestion_en": "Log in to the Karbor node and run the "
                     "执行命令show_service command to view "
                     "the gaussdb check result."}


class PreCheckKarbor(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS-VBS"
        params = Params(project_id, pod_id, self.service_name,
                        self.regionid_list[0])
        self.db_param_dict = params.get_params()
        self.dmk_config_content = {}

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            # Karbor升级前检查
            check_results = list()
            node0 = karbor_client.CBSNode(
                self.db_param_dict["karbor_host0"], "djmanager",
                self.db_param_dict["karbor_djmanager_pwd"],
                self.db_param_dict["karbor_root_pwd"])
            node1 = karbor_client.CBSNode(
                self.db_param_dict["karbor_host1"], "djmanager",
                self.db_param_dict["karbor_djmanager_pwd"],
                self.db_param_dict["karbor_root_pwd"])
            node_list = [node0, node1]

            node_check_list = self.check_karbor_node(node_list)
            check_results.extend(node_check_list)

            # 检查各服务状态
            services = ['ascagent', 'backup', 'zookeeper', 'cps-monitor',
                        'cms', 'omm-ha', 'keepalivedinternal', 'rabbitmq',
                        'karbor-protection', 'karbor-api', 'keepalived',
                        'time-sync', 'scagent', 'haproxyinternal']
            ssh_client = karbor_client.get_sshclient_user_su_root(node0)
            for service in services:
                name_ch = service + "服务检查"
                name_en = service + "ServiceCheck"
                if not self.check_service(service, ssh_client):
                    result = CheckResult(
                        itemname_ch=name_ch,
                        itemname_en=name_en,
                        status="failure",
                        error_msg_cn="服务状态异常，"
                                     "预期状态为normal/avtive/standby。",
                        error_msg_en="Service exception, expected status "
                                     "is normal/avtive/standby.",
                        suggestion_cn="请登录Karbor节点，执行命令show_service "
                                      "--service xxx，检查该服务。",
                        suggestion_en="Login Karbor node, and execute cmd "
                                      "show_service --service xxx "
                                      "to check the service.")
                else:
                    result = CheckResult(itemname_ch=name_ch,
                                         itemname_en=name_en,
                                         status="success")
                check_results.append(result)
            karbor_client.ssh_close(ssh_client)
            logger.info("Check Karbor service success.")

            # 检查服务对接状态
            name_ch = "Karbor服务对接检查"
            name_en = "KarborServiceDockingCheck"
            if self.check_karbor_connect(self.db_param_dict['karbor_host0']) \
                    and self.check_karbor_connect(self.db_param_dict[
                                                      'karbor_host1']):
                result = CheckResult(itemname_ch=name_ch, itemname_en=name_en,
                                     status="success")
            else:
                result = CheckResult(
                    itemname_ch=name_ch, itemname_en=name_en, status="failure",
                    error_msg_cn="服务对接异常, 预期对接状态为OK。",
                    error_msg_en="Service docking exception, "
                                 "expected docking status is OK.",
                    suggestion_cn="请登录Karbor节点1和节点2，执行命令"
                                  "check_karbor_connect，检查服务对接情况。",
                    suggestion_en="Login Karbor first/second node, execute "
                                  "cmd check_karbor_connect to "
                                  "check service docking status.")
            check_results.append(result)
            logger.info("Check Karbor docking status success.")

            return Message(200, check_results=check_results)
        except Exception as e:
            logger.error("Check Karbor failed, reason: %s", str(e))
            return Message(500, e, e)

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    @staticmethod
    def check_service(service, ssh_client):
        cmds = "show_service --service " + service + " | sed 's/ //g'"
        ret = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        if not karbor_client.is_ssh_cmd_executed(ret):
            msg = "Failed to execute cmd show-service."
            logger.error(msg)
            raise Exception(msg)
        for item in ret:
            status = item.split("|")
            if len(status) == 6 and status[3] == "fault":
                logger.error("Check service %s fault." % service)
                return False
        return True

    def check_karbor_connect(self, karbor):
        cmds = "check_karbor_connect | sed 's/ //g'"
        node = karbor_client.CBSNode(karbor, "djmanager",
                                     self.db_param_dict["karbor_djmanager_pwd"],
                                     self.db_param_dict["karbor_root_pwd"])
        ssh_client = karbor_client.get_sshclient_user_su_root(node)
        result = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        if not karbor_client.is_ssh_cmd_executed(result):
            msg = "Failed to execute cmd check_karbor_connect."
            logger.error(msg)
            raise Exception(msg)
        if len(result) < 10:
            logger.error("check_karbor_connect is error on node %s" % karbor)
            return False

        karbor_client.ssh_close(ssh_client)

        for result in result:
            status = result.split("|")
            if len(status) == 7 and status[3] == "Error":
                logger.error("Result of check_karbor_connect is error "
                             "on node %s" % karbor)
                return False
        return True

    def check_karbor_node(self, karbor_list):
        """check karbor node cpu,memory,disk,certs and gaussdb

        :param karbor_list:karbor nodes
        :return: check result list
        """
        logger.info("Begin to check karbor nodes status.")
        result_list = list()
        for karbor_node in karbor_list:
            karbor_ip = karbor_node.ip
            ssh_client = karbor_client.get_sshclient_user_su_root(karbor_node)
            sys_check_list = self.check_sys_component(karbor_ip, ssh_client)
            result_list.extend(sys_check_list)
            cert_check_list = self.check_certs(karbor_ip, ssh_client)
            result_list.extend(cert_check_list)
            karbor_client.ssh_close(ssh_client)

        gauss_status = self.check_gaussdb_status(karbor_list)
        result_list.append(
            self.build_check_esult(gauss_status, GAUSSDB_CHECK_DICT))
        logger.info("Check karbor nodes status success.")
        return result_list

    def check_sys_component(self, karbor_ip, ssh_client):
        """check cpu, memory and diskfree

        :param karbor_ip: karbor node ip
        :param ssh_client: ssh client
        :return:
        """
        sys_list = list()
        for node_check in SYS_CHECK_LIST:
            check_cmd = "/usr/bin/sh /opt/huawei/dj/inspect/inspect.sh " \
                        "%s -c %s" % (karbor_ip, node_check.get("item"))
            logger.info("Check cmd: %s." % check_cmd)
            sys_result = self.get_sys_check_result(karbor_ip, check_cmd,
                                                   ssh_client, node_check)
            sys_list.append(self.build_check_esult(sys_result, node_check))
        logger.info("Check system component on node:%s success." % karbor_ip)
        return sys_list

    def get_sys_check_result(self, karbor_ip, check_cmd, ssh_client, node_):
        result = karbor_client.ssh_exec_command_return(ssh_client,
                                                       check_cmd)
        if not karbor_client.is_ssh_cmd_executed(result):
            msg = "Failed to execute cmd on node: %s." % karbor_ip
            logger.error(msg)
            raise Exception(msg)
        if not self.parse_sys_check(result,
                                    node_.get("contain"),
                                    node_.get("thresholds")):
            logger.error("Check %s error on node %s."
                         % (node_.get("item"), karbor_ip))
            return False
        return True

    @staticmethod
    def parse_sys_check(result, contain_str, check_limit):
        """parse check cmd result

        :param result: check cmd return
        :param contain_str: check item str
        :param check_limit: thresholds
        :return:
        """
        for line in result:
            if line.__contains__(contain_str):
                lr = line.replace("%", "").replace("MB", "").split(':')
                code = lr[1].strip()
                if int(code) < check_limit:
                    return False
                else:
                    return True

    def check_certs(self, karbor_ip, ssh_client):
        cert_list = list()
        for cert_ in CERTS_CHECK_LIST:
            cert_cmd = "check_karbor_connect | grep %s | grep \"%s\" | " \
                       "wc -l" % (cert_.get("item"), cert_.get("contain"))
            logger.info("Check cmd: %s." % cert_cmd)
            cert_result = self.get_cert_check_result(karbor_ip, cert_cmd,
                                                     ssh_client, cert_)
            cert_list.append(self.build_check_esult(cert_result, cert_))
        logger.info("Check certs on node:%s success." % karbor_ip)
        return cert_list

    @staticmethod
    def get_cert_check_result(karbor_ip, cert_cmd, ssh_client, cert_):
        result = karbor_client.ssh_exec_command_return(ssh_client,
                                                       cert_cmd)
        if not karbor_client.is_ssh_cmd_executed(result):
            msg = "Failed to execute cert cmd on node: %s." % karbor_ip
            logger.error(msg)
            raise Exception(msg)
        if int(result[0]) > cert_.get("thresholds"):
            logger.error("Check %s cert failed, error is:%s, "
                         "please check." % (cert_.get("item"),
                                            cert_.get("contain")))
            return False
        return True

    @staticmethod
    def check_gaussdb_status(karbor_list):
        gaussdb_cmd = "bash /usr/local/bin/ha/ha/config_script/" \
                      "sync_monitor.sh get_status"
        logger.info("Check cmd:%s." % gaussdb_cmd)
        gaussdb_status = list()
        for karbor_node in karbor_list:
            karbor_ip = karbor_node.ip
            ssh_client = karbor_client.get_sshclient_user_su_root(karbor_node)
            result = karbor_client.ssh_exec_command_return(ssh_client,
                                                           gaussdb_cmd)
            if not karbor_client.is_ssh_cmd_executed(result):
                msg = "Failed to execute gaussdb cmd on node: %s." % karbor_ip
                logger.error(msg)
                raise Exception(msg)
            for line in result:
                if line.__contains__("DB last online role :"):
                    lr = line.replace(" ", "").split(':')
                    status = lr[1].strip()
                    if status not in ["Primary", "Standby"]:
                        msg = "Gaussdb status %s on node %s is abnormal." \
                              % (status, karbor_ip)
                        logger.error(msg)
                        raise Exception(msg)
                    logger.info("Get gaussdb status %s on node:%s success."
                                % (status, karbor_ip))
                    gaussdb_status.append(status)
        if len(gaussdb_status) != 2:
            logger.error("Get all gaussdb status failed, please check.")
            return False
        if gaussdb_status[0] == gaussdb_status[1]:
            logger.error("Gaussdb status on karbor nodes is same: %s, "
                         "please check." % gaussdb_status)
            return False
        logger.info("Check gaussdb status success.")
        return True

    @staticmethod
    def build_check_esult(check_result, check_item):
        if check_result:
            result = CheckResult(itemname_ch=check_item.get("itemname_ch"),
                                 itemname_en=check_item.get("itemname_en"),
                                 status="success")
        else:
            result = CheckResult(itemname_ch=check_item.get("itemname_ch"),
                                 itemname_en=check_item.get("itemname_en"),
                                 status="failure",
                                 error_msg_cn=check_item.get("error_msg_cn"),
                                 error_msg_en=check_item.get("error_msg_en"),
                                 suggestion_cn=check_item.get("suggestion_cn"),
                                 suggestion_en=check_item.get("suggestion_en"))
        return result
