from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.message import Message
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client
from platforms.upgradecheck.check_result import CheckResult

logger.init("CSBS-VBS")


class PreCheckRunningTask(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS-VBS"
        params = Params(project_id, pod_id, self.service_name,
                        self.regionid_list[0])
        self.db_param_dict = params.get_params()
        self.dmk_config_content = {}

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        logger.info("Start to check whether there are running tasks.")

        check_results = []
        # stop karbor operation service
        name_ch = "检查当前是否有执行中的任务"
        name_en = "CheckRunningTasks"
        try:
            cbs_node = karbor_client.CBSNode(
                self.db_param_dict.get("karbor_host0"), "djmanager",
                self.db_param_dict.get("karbor_djmanager_pwd"),
                self.db_param_dict.get("karbor_root_pwd"))

            # 查询数据库，检查是否有正在执行的任务
            ssh_client = karbor_client.get_sshclient_user_su_root(cbs_node)
            cmds = "docker exec -it gaussdb bash"
            self.expect_cmd_on_client(ssh_client, cmds, "gaussdba@")
            cmds = 'gsql karbor -W %s -c "select count(*) from ' \
                   'checkpoint_item where status like \'%%ing%%\' ' \
                   'and deleted=\'false\';"' % self.db_param_dict["karbor_gaussdb_pwd"]
            if self.is_tasks_running(ssh_client, cmds):
                logger.info("There are tasks running.")
                check_ret = CheckResult(itemname_ch=name_ch, itemname_en=name_en, status="failure",
                                        error_msg_cn="CSBS-VBS服务有正在执行中的任务，如备份恢复等。",
                                        error_msg_en="CSBS-VBS service has tasks in execution, such as backup and recovery.",
                                        suggestion_cn="请等待任务执行完成再进行升级。",
                                        suggestion_en="Please wait until the task is completed before upgrading.")
            else:
                logger.info("There are no tasks running.")
                check_ret = CheckResult(itemname_ch=name_ch, itemname_en=name_en, status="success")
            karbor_client.ssh_close(ssh_client)

            # 清除历史记录
            ssh_client = karbor_client.get_sshclient_user_su_root(cbs_node)
            cmds = 'docker exec -it gaussdb bash -c "rm -f ~/.bash_history"'
            ret = karbor_client.ssh_exec_command_return(ssh_client, cmds)
            if not karbor_client.is_ssh_cmd_executed(ret):
                msg = "Failed to execute cmd %s." % cmds
                logger.error(msg)
                raise Exception(msg)

            check_results.append(check_ret)
            karbor_client.ssh_close(ssh_client)
            return Message(200, check_results=check_results)
        except Exception as e:
            logger.error("Failed to check running tasks, "
                         "reason: %s", str(e))
            return Message(500, e, e)

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    @staticmethod
    def expect_cmd_on_client(client, cmd, expect):
        expect_str = expect + ";$ " + ";# "
        result = karbor_client.ssh_send_command_expect(client, cmd, expect_str)
        if not karbor_client.is_expect_recv(expect.split(';'), '\n'.join(result)):
            msg = "Execute cmd(%s) in server(%s) failed" % (cmd, client['host_ip'])
            logger.error(msg)
            raise Exception(msg)

    @staticmethod
    def is_tasks_running(ssh_client, cmds):
        ret = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        if not karbor_client.is_ssh_cmd_executed(ret):
            msg = "Failed to execute cmd select checkpoint_item."
            logger.error(msg)
            raise Exception(msg)

        # 判断checkpoint_item count
        for item in ret:
            count = item.strip()
            if count == "0":
                return False
        return True
