import os
from configparser import ConfigParser
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client
from plugins.CSBS_VBS.common.scripts.cbs_params import is_primary_region
from plugins.CSBS_VBS.common.scripts.cbs_params import is_global_dr_scene
from plugins.CSBS_VBS.common.scripts.cbs_params import get_current_upgrade_path
from plugins.CSBS_VBS.common.scripts.cbs_apiadapter import API
from plugins.CSBS_VBS.common.scripts.cbs_params import getVersion
from plugins.CSBS_VBS.common.scripts.cbs_params import is_last_region
from plugins.CSBS_VBS.common.scripts.cbs_constant import UPGRADE_PATH, VERSION
from utils.business.project_util import ProjectApi
from platforms.upgradecheck.check_result import CheckResult
from utils.common.message import Message
from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.exception import FCUException
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.DBAdapter.DBConnector import BaseOps

logger.init("CSBS-VBS")
conf_path = os.path.realpath(__file__ + '../../../conf/upgrade.ini')


class PreCheckVersion(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS-VBS"
        params = Params(project_id, pod_id, self.service_name, self.regionid_list[0])
        self.db_param_dict = params.get_params()
        self.dmk_config_content = {}
        self.project_api = ProjectApi()
        self.current_upgrade_path = get_current_upgrade_path(project_id)
        self.original_version = None
        self.expect_version = None
        self.target_version = self.get_target_version()

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            # 升级前检查版本号,是否需要升级
            check_results = []

            # 检查kabror版本号
            name_ch = "Karbor版本号检查"
            name_en = "KarborVersionCheck"
            if self.check_karbor_version():
                result = CheckResult(itemname_ch=name_ch, itemname_en=name_en,
                                     status="success")
                check_results.append(result)
            else:
                result = CheckResult(itemname_ch=name_ch, itemname_en=name_en,
                                     status="failure",
                                     error_msg_cn="Karbor版本号错误，预期版本号为{}".format(self.expect_version),
                                     error_msg_en="Wrong Karbor Version, expected "
                                                  "version is {}".format(self.expect_version),
                                     suggestion_cn="请登录Karbor节点1，执行命令showsys，检查版本号是否正确。"
                                                   "如果该版本不需要升级，"
                                                   "请执行命令start_service --s karbor-operation，重新开启自动备份。",
                                     suggestion_en="Login Karbor first node, and execute cmd showsys to "
                                                   "check Karbor Version. If this version does not need to be upgraded, "
                                                   "execute the command start_service -- s karbor-operation to "
                                                   "restart the automatic backup")
                check_results.append(result)
                return Message(200, check_results=check_results)

            logger.info("Check Karbor version success.")

            # get cbs region
            self.get_cbs_regions()

            # 检查console版本号
            if not is_primary_region(project_id) and is_global_dr_scene(project_id):
                name_ch = "当前RegionConsole版本号检查"
                name_en = "CurrentRegionConsoleVersionCheck"
                if self.check_console_version(False):
                    result = CheckResult(itemname_ch=name_ch, itemname_en=name_en,
                                         status="success")
                else:
                    result = CheckResult(itemname_ch=name_ch, itemname_en=name_en,
                                         status="failure",
                                         error_msg_cn="Console版本号错误，预期版本号为{}".format(self.expect_version),
                                         error_msg_en="Wrong Console Version, expected "
                                                      "version is {}".format(self.expect_version),
                                         suggestion_cn="请登录Console节点，执行命令showsys，检查版本号是否正确",
                                         suggestion_en="Login Console node, and execute cmd showsys to "
                                                       "check Console Version")
                check_results.append(result)
                logger.info("Check current Console version success.")

            if is_last_region(project_id, regionid_list[0]):
                name_ch = "主RegionConsole版本号检查"
                name_en = "PrimaryRegionConsoleVersionCheck"
                if self.check_console_version(True):
                    result = CheckResult(itemname_ch=name_ch, itemname_en=name_en,
                                         status="success")
                else:
                    result = CheckResult(itemname_ch=name_ch, itemname_en=name_en,
                                         status="failure",
                                         error_msg_cn="Console版本号错误，预期版本号为{}".format(self.expect_version),
                                         error_msg_en="Wrong Console Version, expected "
                                                      "version is {}".format(self.expect_version),
                                         suggestion_cn="请登录Console节点，执行命令showsys，检查版本号是否正确",
                                         suggestion_en="Login Console node, and execute cmd showsys to "
                                                       "check Console Version")
                check_results.append(result)
                logger.info("Check Primary Region Console version success.")

            # 保存升级前和升级后版本号
            version_info = {"Originalversion": self.original_version,
                            "Targetversion": self.target_version}
            self.project_api.save_version_info(self.project_id,
                                               "CSBS-VBS",
                                               version_info)

            return Message(200, check_results=check_results)
        except Exception as e:
            logger.error("Check version failed, reason: %s", str(e))
            return Message(500, e, e)

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def check_karbor_version(self):
        node = karbor_client.CBSNode(self.db_param_dict["karbor_host0"],
                                     "djmanager",
                                     self.db_param_dict["karbor_djmanager_pwd"],
                                     self.db_param_dict["karbor_root_pwd"])
        ssh_client = karbor_client.get_sshclient_user_su_root(node)

        cmds = "cat /opt/huawei/dj/cfg/dj.version | cut -d ' ' -f 5"
        ret = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        karbor_client.ssh_close(ssh_client)
        if not karbor_client.is_ssh_cmd_executed(ret):
            msg = "Failed to execute cmd cat /opt/huawei/dj/cfg/dj.version."
            logger.error(msg)
            raise Exception(msg)
        if self.current_upgrade_path in [UPGRADE_PATH.FC6311_T0_651, UPGRADE_PATH.NFVI6311_TO_651]:
            versions_of_631 = [VERSION.v631, VERSION.v613]
            self.expect_version = '/'.join(versions_of_631)
            for version in versions_of_631:
                if version in ret:
                    self.original_version = version
            if not self.original_version:
                logger.error("Check Karbor Version failed, version {} is expect.".format(VERSION.v631))
                return False
        else:
            versions_of_650 = [VERSION.v650, VERSION.v650SPH1, VERSION.v650SPH2]
            self.expect_version = '/'.join(versions_of_650)
            for version in versions_of_650:
                if version in ret:
                    self.original_version = version
            if not self.original_version:
                logger.error("Check Karbor Version failed, version {} is expect.".format(self.expect_version))
                return False
        conf = ConfigParser()
        conf.read(conf_path)
        conf.set('VERSION', 'original_version', self.original_version)
        conf.set('VERSION', 'target_version', self.target_version)
        conf.write(open(conf_path, 'w'))
        logger.info('original_version: {}, target_version: {}'.format(self.original_version, self.target_version))
        return True

    def check_console_version(self, primary_region):
        cmds = "showsys |grep Version | cut -d '|' -f 2 |sed 's/ //g'"
        if primary_region:
            node = karbor_client.CBSNode(self.db_param_dict["primary_region_console_host0"], "cbs_admin",
                                         self.db_param_dict["console_cbs_admin_pwd"],
                                         self.db_param_dict["console_root_pwd"])
        else:
            node = karbor_client.CBSNode(self.db_param_dict["console_host0"], "cbs_admin",
                                         self.db_param_dict["console_cbs_admin_pwd"],
                                         self.db_param_dict["console_root_pwd"])

        ssh_client = karbor_client.get_sshclient_user_su_root_console(node)
        result = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        karbor_client.ssh_close(ssh_client)
        if not karbor_client.is_ssh_cmd_executed(result):
            msg = "Failed to execute cmd showsys."
            logger.error(msg)
            raise Exception(msg)
        if self.current_upgrade_path in [UPGRADE_PATH.FC6311_T0_651, UPGRADE_PATH.NFVI6311_TO_651]:
            if VERSION.v631 not in result:
                logger.error("Check Console Version failed, version {} is expect.".format(VERSION.v631))
                return False
        else:
            versions_of_650 = [VERSION.v650, VERSION.v650SPH1, VERSION.v650SPH2]
            judge_verison = [version not in result for version in versions_of_650]
            if all(judge_verison):
                logger.error("Check Console Version failed, "
                             "version {} is expect.".format('/'.join(versions_of_650)))
                return False
        return True

    def get_target_version(self):
        try:
            pkg_path, file_name = API.find_file("OceanStor BCManager",
                                                "_CBS_Karbor_Upgrade_X86.zip")
            pkg = os.path.join(pkg_path, file_name)
            if not file_name:
                msg = "can not find Karbor pkg."
                raise Exception(msg)
        except Exception as ex:
            logger.error(ex)
            return Message(500, FCUException(645000))
        target_verson = getVersion(pkg)
        if not target_verson:
            raise Exception("Failed getting version from karbor's packages")
        return str(target_verson)

    def get_cbs_regions(self):
        """get regions deploy csbs"""
        all_regions = BaseOps().get_all_regionid_by_projectid(self.project_id)
        mo_utils2 = ManageOneUtil2()
        param_dict = ParamUtil().get_service_cloud_param(self.project_id, "OC",
                                                         self.regionid_list[0])
        iam_ip = param_dict.get('oc_node_ip')
        iam_token = mo_utils2.getIamToken(iam_ip)
        cbs_region_list = []
        for region_id in all_regions:
            endpoint_info = mo_utils2.get_all_endpoints(iam_token, iam_ip,
                                                        self.regionid_list[0])
            for endpoint in endpoint_info['endpoints']:
                if endpoint['id'] == 'csbs':
                    cbs_region_list.append(region_id)
                    break

        conf = ConfigParser()
        conf.read(conf_path)
        conf.set('REGION', 'cbs_region_list', ','.join(cbs_region_list))
        conf.write(open(conf_path, 'w'))
