import os
import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.fic_base import StepBaseInterface
from utils.common.exception import FCUException
from utils.common.message import Message
from utils.business.apig_util import APIGOperateApi
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client
from plugins.CSBS_VBS.common.scripts.cbs_params import Params

logger.init("CSBS-VBS")


def get_config_content(file_path):
    file_object = open(file_path, encoding='utf-8')
    content = ""
    try:
        for line in file_object:
            content = "{0}{1}".format(content, line)
    except:
        logger.error('get file %s content failed' % file_path)
    finally:
        file_object.close()
    return content


class RegisterApiGW(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        super(RegisterApiGW, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.param_util = ParamUtil()
        self.apiGW = APIGOperateApi()
        self.service_name = "CSBS-VBS"
        params = Params(project_id, pod_id, self.service_name,
                        self.regionid_list[0])
        self.db_param_dict = params.get_params()

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            # karbor ssh client
            karbor_node = karbor_client.CBSNode(self.db_param_dict.get("karbor_host0"),
                                                "djmanager",
                                                self.db_param_dict.get("karbor_djmanager_pwd"),
                                                self.db_param_dict.get("karbor_root_pwd"))
            ssh_client = karbor_client.get_sshclient_user_su_root_console(karbor_node)

            # get sc global domain name
            get_cmd = "docker exec karborapi cat /etc/karbor/karbor.conf " \
                      "|grep sc_endpoint | awk -F ':' '{print $2}' | " \
                      "awk -F 'console-silvan.' '{print $2}'"
            result = karbor_client.ssh_exec_command_return(ssh_client, get_cmd)
            if not karbor_client.is_ssh_cmd_executed(result):
                karbor_client.ssh_close(ssh_client)
                msg = 'failed to get sc_url on karbor node.'
                raise Exception(msg)
            global_domain_name = result[0]

            service_domain_name = "csbs-vbs-api.%s.%s:8799" % (
                self.regionid_list[0], global_domain_name)
            api_gateway_domain_name = "csbs-vbs.%s.%s" % (
                self.regionid_list[0], global_domain_name)

            as_api_config = get_config_content(
                os.path.abspath(__file__ + '/../../conf/api_config.yaml'))
            as_api_config = as_api_config % (api_gateway_domain_name, service_domain_name)

            # 注册前先执行一次删除, 确保信息清空
            logger.info("start delete old api info.")
            try:
                self.apiGW.delete_apigateway(self.pod_id, as_api_config,
                                             self.regionid_list[0])
            except Exception as e:
                # 移除出错不抛异常
                logger.error("remove api failed, errMsg=%s" % str(e))

            # 开始注册API
            logger.info("start register api info.")
            self.apiGW.register_apigateway(self.pod_id, as_api_config,
                                           self.regionid_list[0])

            return Message(200)

        except Exception as e:
            logger.error(str(e))
            return Message(500, FCUException(645001))

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)
