import os
from configparser import ConfigParser
from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.message import Message
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts.cbs_params import is_primary_region
from plugins.CSBS_VBS.common.scripts.cbs_params import is_global_dr_scene
from plugins.CSBS_VBS.common.scripts.cbs_params import is_last_region
from plugins.CSBS_VBS.common.scripts.cbs_apiadapter import API
from plugins.CSBS_VBS.common.scripts.cbs_configfiles import ConfigFiles
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client

logger.init("CSBS-VBS")
conf_path = os.path.realpath(__file__ + '../../../conf/upgrade.ini')


class RollbackConsole(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS"
        params = Params(project_id, pod_id, self.service_name, self.regionid_list[0])
        self.db_param_dict = params.get_params()
        self.dmk_config_content = {}
        conf = ConfigParser()
        conf.read(conf_path)
        self.console_ex_version = conf.get('VERSION', 'original_version')

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        try:
            if not is_primary_region(project_id) and is_global_dr_scene(project_id):
                # standby region
                # backup iam cert
                standby_region = False
                self.backup_cert(standby_region)

                self.db_param_dict['console_version'] = self.console_ex_version
                config = ConfigFiles(project_id, pod_id,
                                     self.regionid_list, self.db_param_dict)
                self.dmk_config_content = config.get_console_config_files()
                host_content = self.dmk_config_content["host"]
                vars_content = self.dmk_config_content["config"]

                API.login_dmk(self.db_param_dict['DMK_dmk_floatIp'],
                              self.db_param_dict['dmk_account'],
                              self.db_param_dict['dmk_account_pwd'])
                upgrade_user_id = API.get_dmk_account_id(
                    self.db_param_dict['dmk_account'], "cbs_admin")

                logger.info("login dmk successfully, "
                            "begin to call DMK to rollback console.")

                # 回退 console
                result = API.executedmkdeployment("CBS-Console", 'true',
                                                  self.console_ex_version,
                                                  "7 Upgrade console",
                                                  host_content, vars_content,
                                                  upgrade_user_id)
                if result:
                    logger.info("Rollback current region Console successfully.")
                else:
                    logger.error("Failed to rollbackup current region console. The ex version "
                                 "of console pkg may not exist.")
                    return Message(500, "回退console失败",
                                   "Failed to rollback console")

                # replace iam cert
                self.replace_cert(standby_region)

            if is_last_region(project_id, regionid_list[0]):
                # primary region
                # backup iam cert
                primary_region = True
                self.backup_cert(primary_region)

                self.db_param_dict['console_version'] = self.console_ex_version
                config = ConfigFiles(project_id, pod_id,
                                     self.regionid_list, self.db_param_dict)
                self.dmk_config_content = config.get_console_config_files()
                host_content = self.dmk_config_content["primary_region_host"]
                vars_content = self.dmk_config_content["primary_region_config"]

                if is_primary_region(project_id):
                    API.login_dmk(self.db_param_dict['DMK_dmk_floatIp'],
                                  self.db_param_dict['dmk_account'],
                                  self.db_param_dict['dmk_account_pwd'])
                else:
                    API.login_dmk(self.db_param_dict['DMK_Global-DMK-Float-IP'],
                                  self.db_param_dict['dmk_account'],
                                  self.db_param_dict['dmk_account_pwd'])
                upgrade_user_id = API.get_dmk_account_id(
                    self.db_param_dict['dmk_account'], "cbs_admin")

                logger.info("login dmk successfully, "
                            "begin to call DMK to rollback console.")

                # 回退 console
                result = API.executedmkdeployment("CBS-Console", 'true',
                                                  self.console_ex_version,
                                                  "7 Upgrade console",
                                                  host_content, vars_content,
                                                  upgrade_user_id)
                if result:
                    logger.info("Rollback primary region Console successfully.")
                else:
                    logger.error("Failed to rollbackup primary region console. The ex version "
                                 "of console pkg may not exist.")
                    return Message(500, "回退console失败",
                                   "Failed to rollback console")

                # replace iam cert
                self.replace_cert(primary_region)

            return Message(200)

        except Exception as e:
            logger.error("Failed to rollback console, reason: %s", str(e))
            return Message(500, e, e)

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def backup_cert(self, primary_region):
        if primary_region:
            logger.info("primary region, backup_cert")
            ips = [self.db_param_dict.get("primary_region_console_host0"),
                   self.db_param_dict.get("primary_region_console_host1")]
            cbs_pwd = self.db_param_dict.get("console_cbs_admin_pwd")
            root_pwd = self.db_param_dict.get("console_root_pwd")
        else:
            logger.info("current region, backup_cert")
            ips = [self.db_param_dict.get("console_host0"),
                   self.db_param_dict.get("console_host1")]
            cbs_pwd = self.db_param_dict.get("console_cbs_admin_pwd")
            root_pwd = self.db_param_dict.get("console_root_pwd")

        for ip in ips:
            cbs_node = karbor_client.CBSNode(ip, "cbs_admin", cbs_pwd, root_pwd)
            ssh_client = karbor_client.get_sshclient_user_su_root_console(cbs_node)

            cmds = "ll /home/iamKeyStore.jks"
            result = karbor_client.ssh_exec_command_return(ssh_client, cmds)

            ret = karbor_client.is_ssh_cmd_executed(result)
            if not ret:
                logger.info("there is no iam cert in %s home pass." % ip)
                cmds = "cp -pf /opt/huawei/console/tomcat/webapps/cbs/WEB-INF/classes/config/iamKeyStore.jks /home"
                result = karbor_client.ssh_exec_command_return(ssh_client, cmds)
                karbor_client.ssh_close(ssh_client)

                ret = karbor_client.is_ssh_cmd_executed(result)
                if not ret:
                    msg = "Failed to backup console %s iam cert." % ip
                    logger.error(msg)
                    raise Exception(msg)

                logger.info("Backup console %s iam cert successfully." % ip)
                karbor_client.ssh_close(ssh_client)

    def replace_cert(self, primary_region):
        if primary_region:
            logger.info("primary region, backup_cert")
            ips = [self.db_param_dict.get("primary_region_console_host0"),
                   self.db_param_dict.get("primary_region_console_host1")]
            cbs_pwd = self.db_param_dict.get("console_cbs_admin_pwd")
            root_pwd = self.db_param_dict.get("console_root_pwd")
        else:
            logger.info("current region, backup_cert")
            ips = [self.db_param_dict.get("console_host0"),
                   self.db_param_dict.get("console_host1")]
            cbs_pwd = self.db_param_dict.get("console_cbs_admin_pwd")
            root_pwd = self.db_param_dict.get("console_root_pwd")

        for ip in ips:
            cbs_node = karbor_client.CBSNode(ip, "cbs_admin", cbs_pwd, root_pwd)
            ssh_client = karbor_client.get_sshclient_user_su_root_console(cbs_node)

            cmds = "rm -f /opt/huawei/console/tomcat/webapps/cbs/WEB-INF/classes/config/iamKeyStore.jks"
            result = karbor_client.ssh_exec_command_return(ssh_client, cmds)

            ret = karbor_client.is_ssh_cmd_executed(result)
            if not ret:
                msg = "Failed to del %s iam cert." % ip
                logger.error(msg)
                raise Exception(msg)

            cmds = "cp -pf /home/iamKeyStore.jks /opt/huawei/console/tomcat/webapps/cbs/WEB-INF/classes/config/"
            result = karbor_client.ssh_exec_command_return(ssh_client, cmds)
            karbor_client.ssh_close(ssh_client)

            ret = karbor_client.is_ssh_cmd_executed(result)
            if not ret:
                msg = "Failed to replace console %s iam cert." % ip
                logger.error(msg)
                raise Exception(msg)

            logger.info("Replace console %s iam cert successfully." % ip)
            karbor_client.ssh_close(ssh_client)
