import os
from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.message import Message
from plugins.CSBS_VBS.common.scripts.cbs_params import Params
from plugins.CSBS_VBS.common.scripts.cbs_params import getVersion
from plugins.CSBS_VBS.common.scripts.cbs_apiadapter import API
from plugins.CSBS_VBS.common.scripts.cbs_configfiles import ConfigFiles
from plugins.CSBS_VBS.common.scripts import cbs_karborclient as karbor_client
from utils.common.exception import FCUException
from plugins.CSBS_VBS.common.scripts.cbs_params import is_csha_scene

logger.init("CSBS-VBS")


class RollbackKarbor(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.service_name = "CSBS"
        params = Params(project_id, pod_id, self.service_name,
                        self.regionid_list[0])
        self.db_param_dict = params.get_params()
        self.dmk_config_content = {}

    def pre_check(self, project_id, pod_id, regionid_list):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            pkg_path, file_name = API.find_file("OceanStor BCManager",
                                                "_CBS_Karbor_Upgrade_X86.zip")
            pkg = os.path.join(pkg_path, file_name)
            if not file_name:
                msg = "can not find Karbor pkg."
                raise Exception(msg)
        except Exception as ex:
            logger.error(ex)
            return Message(500, FCUException(645000))

        try:
            self.db_param_dict['karbor_version'] = getVersion(pkg)
            self.db_param_dict['check_karbor'] = 'false'

            config = ConfigFiles(project_id, pod_id,
                                 self.regionid_list, self.db_param_dict)
            self.dmk_config_content = config.get_karbor_config_files()
            host_content = self.dmk_config_content["host"]
            vars_content = self.dmk_config_content["config"]

            API.login_dmk(self.db_param_dict['DMK_dmk_floatIp'],
                          self.db_param_dict['dmk_account'],
                          self.db_param_dict['dmk_account_pwd'])
            upgrade_user_id = API.get_dmk_account_id(
                self.db_param_dict['dmk_account'], "djmanager")

            logger.info("login dmk successfully, "
                        "begin to call DMK to rollback Karbor.")

            # csha场景，回滚karbor前先停止仲裁节点的zookeeper
            if is_csha_scene(self.project_id):
                logger.info("This is csha scene, stop zookeeper on "
                            "arbitration node brfore rollback.")
                try:
                    arb_node = karbor_client.ARBNode(
                        self.db_param_dict.get("arb_ip"),
                        self.db_param_dict.get("arb_username"),
                        self.db_param_dict.get("arb_passwd"),
                        self.db_param_dict.get("arb_root_passwd"))
                    ssh_client = karbor_client.get_sshclient_user_su_root(
                        arb_node)
                    cmds = "/usr/bin/zookeeperControl -S STOP"
                    result = karbor_client.ssh_exec_command_return(
                        ssh_client, cmds)
                    ret = karbor_client.is_ssh_cmd_executed(result)
                except Exception as err:
                    logger.error("Failed to stop zookeeper on "
                                 "arbitration node, reason: %s", str(err))
                    return Message(500, err, err)
                logger.info("Stop zookeeper on arbitration "
                            "node successfully.")
                karbor_client.ssh_close(ssh_client)

            # 回退 Karbor
            result = API.executedmkdeployment("Karbor", 'true',
                                              self.db_param_dict['karbor_version'],
                                              "[Rollback] CSBS-VBS Karbor",
                                              host_content, vars_content,
                                              upgrade_user_id)
            if result:
                logger.info("Rollback Karbor successfully.")
            else:
                logger.error("Failed to rollback Karbor.")
                return Message(500, "回退Karbor失败",
                               "Failed to rollback Karbor")
        except Exception as e:
            logger.error("Failed to rollback Karbor, reason: %s", str(e))
            return Message(500, e, e)

        # stop karbor operation service
        try:
            self.stop_operation()
        except Exception as e:
            logger.error("Failed to stop karbor-operation, reason: %s", str(e))
            return Message(500, e, e)

        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """
        标准调用接口：检查
        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        return Message(200)

    def stop_operation(self):
        cbs_node = karbor_client.CBSNode(
            self.db_param_dict.get("karbor_host0"), "djmanager",
            self.db_param_dict.get("karbor_djmanager_pwd"),
            self.db_param_dict.get("karbor_root_pwd"))
        ssh_client = karbor_client.get_sshclient_user_su_root(cbs_node)

        cmds = "stop_service --service karbor-operation"
        result = karbor_client.ssh_exec_command_return(ssh_client, cmds)
        ret = karbor_client.is_ssh_cmd_executed(result)
        if not ret:
            msg = "Failed to execute cmd stop_service --service karbor-operation"
            karbor_client.ssh_close(ssh_client)
            logger.error(msg)
            raise Exception(msg)

        logger.info("Stop karbor operation service successfully.")
        karbor_client.ssh_close(ssh_client)
