import utils.common.log as logger
from platforms.upgradecheck.check_result import CheckResult
from plugins.CSBS_VBS.common.ssh_client import SshClient

CMD_EXECUTE_COUNT = 5
DOUBLE_STATUS = ["Primary", "Standby"]


class KarborOperation(object):
    def __init__(self, params_store):
        self.params_store = params_store
        self.dmk_params = params_store.dmk_params
        self.karbor_params = params_store.karbor_params
        cbs_node0 = self.karbor_params.cbs_node0
        cbs_node1 = self.karbor_params.cbs_node1
        self.node_list = [cbs_node0, cbs_node1]
        self.ssh_client = SshClient()

    def reinforce_cut_over4arm(self):
        """This function is used to execute fix_architecture command in the

        karborprotection docker container to reinforce cut-over operation for
        node with arm os.
        """
        node = self.node_list[0]
        cmd = "docker exec karborprotection kangaroo-manage fix_architecture"
        logger.info("Start to execute fix_architecture command in the "
                    "karborprotection docker container to inforce cutover"
                    "operation on the node of {}, "
                    "with cmd: {}.".format(node.ip, cmd))
        ssh_client = self.ssh_client.get_sshclient_user_su_root(node)
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            raise Exception("Failed to execute the cmd {}, "
                            "please check.".format(cmd))
        logger.info("Fix_architecture command has been executed"
                    "on the node of {} successfully.".format(node.ip))

    def set_price_rate_switch(self, target="true"):
        """This function is used to modify the price rate switch on the Karbor

        node. If the switch is on, rate_params will be added into the
        request params when Karbor places an order, otherwise the rate_params
        will not be added.
        """
        if target not in ["true", "false"]:
            raise Exception("Value of target should be true or false.")
        node = self.node_list[0]
        cmd = "set_features --sc_price_rate_switch {}".format(target)
        logger.info("Start to set the sc_price_rate_witch to be {} "
                    "on the node of {}.".format(target, node.ip))
        ssh_client = self.ssh_client.get_sshclient_user_su_root(node)
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            raise Exception("Failed to execute the cmd {}, "
                            "please check.".format(cmd))
        logger.info("Set the sc_price_rate_witch to be {} "
                    "on the node of {} successfully.".format(target, node.ip))

    def set_max_sync_time_interval_allowed(self, time_interval):
        cmd = 'sed -i "s#allow_sync_time=.*#allow_sync_time={}#" ' \
              '/opt/huawei/dj/etc/gaussdb/db_sync.conf'.format(time_interval)
        for node in self.node_list:
            logger.info("Start to set the max sync time interval allowed to be"
                        " {} minutes for gaussdb on the "
                        "node {}.".format(time_interval, node.ip))
            ssh_client = self.ssh_client.get_sshclient_user_su_root(node)
            result = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
            logger.info(result)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                raise Exception("Failed to execute the cmd {}, "
                                "please check.".format(cmd))
            logger.info("Set the sync time to be {} minutes "
                        "successfully.".format(time_interval))

    def check_services(self, check_results, service_check_dict):
        logger.info("Start to check karbor service.")
        cbs_node0 = self.karbor_params.cbs_node0

        services = ['ascagent', 'backup', 'zookeeper', 'cps-monitor',
                    'cms', 'omm-ha', 'keepalivedinternal', 'rabbitmq',
                    'karbor-protection', 'karbor-api', 'keepalived',
                    'time-sync', 'scagent', 'haproxyinternal']

        services_check_success_result = \
            self._get_services_check_success_result(services,
                                                    service_check_dict)
        services_check_failed_result = \
            self._get_services_check_failed_result(services,
                                                   service_check_dict)

        ssh_client = self.ssh_client.get_sshclient_user_su_root(cbs_node0)
        for service in services:
            if not self.check_service(service, ssh_client):
                result = services_check_failed_result[service]
            else:
                result = services_check_success_result[service]
            check_results.append(result)
        self.ssh_client.ssh_close(ssh_client)
        logger.info("Finished checking karbor service.")
        return check_results

    @staticmethod
    def _get_services_check_success_result(services, service_check_dict):
        services_check_success_result = dict()
        for service in services:
            name_ch = "".join([service, service_check_dict["name_tail_ch"]])
            name_en = "".join([service, service_check_dict["name_tail_en"]])
            services_check_success_result[service] = CheckResult(
                itemname_ch=name_ch,
                itemname_en=name_en,
                status=service_check_dict["success_status"]
            )
        return services_check_success_result

    @staticmethod
    def _get_services_check_failed_result(services, service_check_dict):
        services_check_success_result = dict()
        for service in services:
            name_ch = "".join([service, service_check_dict["name_tail_ch"]])
            name_en = "".join([service, service_check_dict["name_tail_en"]])
            services_check_success_result[service] = CheckResult(
                itemname_ch=name_ch,
                itemname_en=name_en,
                status=service_check_dict["failure_status"],
                error_msg_cn=service_check_dict["error_msg_cn"],
                error_msg_en=service_check_dict["error_msg_en"],
                suggestion_cn=service_check_dict["suggestion_cn"],
                suggestion_en=service_check_dict["suggestion_en"]
            )
        return services_check_success_result

    def check_service(self, service, ssh_client):
        cmds = "show_service --service " + service + " | sed 's/ //g'"
        for i in range(CMD_EXECUTE_COUNT):
            ret = self.ssh_client.ssh_exec_command_return(ssh_client, cmds)
            try:
                if not self.ssh_client.is_ssh_cmd_executed(ret):
                    raise Exception("Failed to execute cmd show-service.")
            except Exception as e:
                logger.error("Failed to execute cmd show-service, "
                             "retry for {}, error: {}.".format(i, str(e)))
                continue
            for item in ret:
                status = item.split("|")
                if len(status) == 6 and status[3] == "fault":
                    logger.error("Check service {} fault.".format(service))
                    return False
            return True

    def check_connection_status(self, check_results, connect_check_dic):
        logger.info("Start to check karbor connection status.")
        if self.check_karbor_connect(self.node_list[0]) \
                and self.check_karbor_connect(self.node_list[1]):
            result = CheckResult(
                itemname_ch=connect_check_dic["itemname_ch"],
                itemname_en=connect_check_dic["itemname_en"],
                status=connect_check_dic["success_status"])
        else:
            result = CheckResult(
                itemname_ch=connect_check_dic["itemname_ch"],
                itemname_en=connect_check_dic["itemname_en"],
                status=connect_check_dic["failure_status"],
                error_msg_cn=connect_check_dic["error_msg_cn"],
                error_msg_en=connect_check_dic["error_msg_en"],
                suggestion_cn=connect_check_dic["suggestion_cn"],
                suggestion_en=connect_check_dic["suggestion_en"])
        check_results.append(result)
        logger.info("Finished checking karbor connection status.")
        return check_results

    def check_karbor_connect(self, node):
        cmds = "check_karbor_connect | sed 's/ //g'"
        ssh_client = self.ssh_client.get_sshclient_user_su_root(node)
        for i in range(CMD_EXECUTE_COUNT):
            result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                             cmds)
            try:
                if not self.ssh_client.is_ssh_cmd_executed(result):
                    raise Exception("Failed to execute cmd "
                                    "check_karbor_connect.")
            except Exception as e:
                logger.error("Failed to execute cmd check_karbor_connect, "
                             "retry for {}, error: {}".format(i, str(e)))
                continue
            if len(result) < 10:
                logger.error("check_karbor_connect is error "
                             "on node: {}".format(node.ip))
                return False

            self.ssh_client.ssh_close(ssh_client)

            for result in result:
                status = result.split("|")
                if len(status) == 7 and status[3] == "Error":
                    logger.error("Result of check_karbor_connect is error "
                                 "on node {}.".format(node.ip))
                    return False
            return True

    def check_nodes_sys_status(self, check_results, sys_check_list):
        logger.info("Start to check karbor node sys status.")
        for node in self.node_list:
            karbor_ip = node.ip
            ssh_client = self.ssh_client.get_sshclient_user_su_root(node)
            sys_list = self.check_sys_component(karbor_ip,
                                                ssh_client,
                                                sys_check_list)
            check_results.extend(sys_list)
            self.ssh_client.ssh_close(ssh_client)
        logger.info("Finished checking karbor node sys status.")
        return check_results

    def check_node_certs_status(self, check_results, certs_check_list):
        logger.info("Start to check karbor node certs status.")
        for node in self.node_list:
            karbor_ip = node.ip
            ssh_client = self.ssh_client.get_sshclient_user_su_root(node)
            cert_check_list = self.check_certs(karbor_ip,
                                               ssh_client,
                                               certs_check_list)
            check_results.extend(cert_check_list)
            self.ssh_client.ssh_close(ssh_client)
        logger.info("Finished checking karbor node certs status.")

        return check_results

    def check_certs(self, karbor_ip, ssh_client, certs_check_list):
        cert_list = list()
        for cert_ in certs_check_list:
            cert_cmd = "check_karbor_connect | grep {} | grep \"{}\" | " \
                       "wc -l".format(cert_.get("item"), cert_.get("contain"))
            logger.info("Check cmd: {}.".format(cert_cmd))
            cert_result = self.get_cert_check_result(karbor_ip, cert_cmd,
                                                     ssh_client, cert_)
            cert_list.append(self.build_check_result(cert_result, cert_))
        logger.info("Check certs on node:{} success.".format(karbor_ip))
        return cert_list

    def get_cert_check_result(self, karbor_ip, cert_cmd, ssh_client, cert_):
        for i in range(CMD_EXECUTE_COUNT):
            result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                             cert_cmd)
            try:
                logger.info("Cmd output:{}".format(result))
                if not self.ssh_client.is_ssh_cmd_executed(result):
                    raise Exception("Failed to execute cert cmd "
                                    "on node: {}.".format(karbor_ip))

                if int(result[0]) > cert_.get("thresholds"):
                    logger.error("Check {} cert failed, error is:{}, "
                                 "please check.".format(cert_.get("item"),
                                                        cert_.get("contain")))
                    return False
            except Exception as err:
                logger.error("Execute cert cmd err:{}, "
                             "retry for {}".format(err, i))
                continue
            return True

    def check_node_gaussdb_status(self, check_results, gauss_check_dic):
        logger.info("Start to check karbor node certs status.")
        gauss_status = self._check_gaussdb_status(self.node_list)
        check_results.append(
            self.build_check_result(gauss_status, gauss_check_dic))
        logger.info("Finished checking gaussdb status.")
        return check_results

    def _check_gaussdb_status(self, karbor_list):
        gaussdb_cmd = 'docker exec gaussdb /bin/bash ' \
                      '-c "source /home/gaussdba/.bashrc;gs_ctl query"'
        logger.info("Check cmd:{}.".format(gaussdb_cmd))
        gaussdb_status = list()
        for karbor_node in karbor_list:
            karbor_ip = karbor_node.ip
            ssh_client = self.ssh_client.get_sshclient_user_su_root(
                karbor_node)
            for i in range(CMD_EXECUTE_COUNT):
                result = \
                    self.ssh_client.ssh_exec_command_return(ssh_client,
                                                            gaussdb_cmd)
                try:
                    if not self.ssh_client.is_ssh_cmd_executed(result):
                        raise Exception("Failed to execute gaussdb cmd "
                                        "on node: {}.".format(karbor_ip))
                except Exception as e:
                    logger.error("Failed to execute gaussdb cmd, error: {} "
                                 "retry for {}".format(str(e), i))
                    continue
                for line in result:
                    if line.__contains__("LOCAL_ROLE"):
                        lr = line.split(':')
                        status = lr[1].strip()
                        if status not in DOUBLE_STATUS:
                            raise Exception(
                                "Gaussdb status {} on node {} "
                                "is abnormal.".format(status, karbor_ip))
                        logger.info("Get gaussdb status {} on node:{}"
                                    " success.".format(status, karbor_ip))
                        gaussdb_status.append(status)
                break
        if len(gaussdb_status) != 2:
            logger.error("Get all gaussdb status failed, please check.")
            return False
        if gaussdb_status[0] == gaussdb_status[1]:
            logger.error("Gaussdb status on karbor nodes is same: {}, "
                         "please check.".format(gaussdb_status))
            return False
        logger.info("Check gaussdb status success.")
        return True

    def check_sys_component(self, karbor_ip, ssh_client, sys_check_list):
        """check cpu, memory and diskfree

        """
        sys_list = list()
        for node_check in sys_check_list:
            check_cmd = "/usr/bin/sh /opt/huawei/dj/inspect/inspect.sh " \
                        "{} -c {}".format(karbor_ip, node_check.get("item"))
            logger.info("Check cmd: {}.".format(check_cmd))
            sys_result = self.get_sys_check_result(karbor_ip, check_cmd,
                                                   ssh_client, node_check)
            sys_list.append(self.build_check_result(sys_result, node_check))
        logger.info("Check system component on node:"
                    "{} success.".format(karbor_ip))
        return sys_list

    def get_sys_check_result(self, karbor_ip, check_cmd, ssh_client, node_):
        for i in range(CMD_EXECUTE_COUNT):
            result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                             check_cmd)
            logger.info(result)
            try:
                if not self.ssh_client.is_ssh_cmd_executed(result):
                    raise Exception("Failed to execute cmd "
                                    "on node:{}.".format(karbor_ip))
            except Exception as e:
                logger.error("Failed to execute sys_check cmd, retry "
                             "for {}, error: {}.".format(i, str(e)))
                continue
            if not self.parse_sys_check(result,
                                        node_.get("contain"),
                                        node_.get("thresholds")):
                logger.error("Check {} error on node "
                             "{}.".format(node_.get("item"), karbor_ip))
                return False
            return True

    @staticmethod
    def parse_sys_check(result, contain_str, check_limit):
        """parse check cmd result

        :param result: check cmd return
        :param contain_str: check item str
        :param check_limit: thresholds
        :return:
        """
        for line in result:
            if line.__contains__(contain_str):
                lr = line.replace("%", "").replace("MB", "").split(':')
                code = lr[1].strip()
                if int(code) < check_limit:
                    return False
                else:
                    return True

    @staticmethod
    def build_check_result(check_result, check_item):
        if check_result:
            result = CheckResult(itemname_ch=check_item.get("itemname_ch"),
                                 itemname_en=check_item.get("itemname_en"),
                                 status="success")
        else:
            result = CheckResult(itemname_ch=check_item.get("itemname_ch"),
                                 itemname_en=check_item.get("itemname_en"),
                                 status="failure",
                                 error_msg_cn=check_item.get("error_msg_cn"),
                                 error_msg_en=check_item.get("error_msg_en"),
                                 suggestion_cn=check_item.get("suggestion_cn"),
                                 suggestion_en=check_item.get("suggestion_en"))
        return result

    def change_operation_service(self, tar_status="stop"):
        if tar_status not in ["start", "stop"]:
            raise Exception("The tar_status attr should be 'start' or 'stop'.")
        cbs_node0 = self.karbor_params.cbs_node0
        ssh_client = self.ssh_client.get_sshclient_user_su_root(cbs_node0)
        cmd = "{}_service --service karbor-operation".format(tar_status)
        logger.info("Start change operation service, cmd: {}.".format(cmd))
        cmd_timeout = 60*10
        self.ssh_client.ssh_exec_command_return(ssh_client, cmd, cmd_timeout)
        self.ssh_client.ssh_close(ssh_client)
        logger.info("Change operation service successfully, "
                    "service status is {}.".format(tar_status))
        return True
