import json
import os
import zipfile

import utils.common.log as logger
import utils.common.software_package_util as FileUtil
from platforms.param.param_service import ParamService
from platforms.project.ProjectUtils import get_project_conditions
from plugins.CSBS_VBS.common import constant
from plugins.CSBS_VBS.common.ssh_client import SshClient
from plugins.CSBS_VBS.common.upgrade import models
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.dmk_util import DmkApi
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi

logger.init(constant.BACKUP_SERVICE_NAME)


class ParamsStore(object):
    def __init__(self, project_id, pod_id, regionid_list=None):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_id = regionid_list[0]
        self.params_tools = ParamsTools(project_id)
        self.db = BaseOps()
        self.service_name = constant.BACKUP_SERVICE_NAME
        self.dmk_api = DmkApi()
        self.ssh_client = SshClient()
        # get all params
        self.db_params_dict = self._get_params()

        # sub-package params based on category
        self.dmk_params = self.get_dmk_params()
        self.project_scene = self.get_project_scene_params()
        self.karbor_params = self.get_karbor_params()
        self.upgrade_path = self.get_upgrade_path_params()

    def get_karbor_params(self):
        cbs_nodes = self.get_karbor_nodes()
        cbs_node2 = cbs_nodes[2] if not self.params_tools.is_csha_scene \
            else None
        karbor_params = models.KarborParams(
            djmanager_account_name='djmanager',
            djmanager_account_pwd=self.db_params_dict['karbor_djmanager_pwd'],
            root_account_name='root',
            root_account_pwd=self.db_params_dict['karbor_root_pwd'],
            cbs_node0=cbs_nodes[0],
            cbs_node1=cbs_nodes[1],
            cbs_node2=cbs_node2
        )
        return karbor_params

    def get_karbor_nodes(self):
        karbor_ip_list = [
            self.db_params_dict['karbor_host0'],
            self.db_params_dict['karbor_host1'],
        ]
        if not self.params_tools.is_csha_scene:
            karbor_ip_list.append(self.db_params_dict['karbor_host2'])
        karbor_nodes_list = []
        for karbor_ip in karbor_ip_list:
            cbs_node = models.CBSNode(
                ip=karbor_ip,
                user="djmanager",
                user_pwd=self.db_params_dict['karbor_djmanager_pwd'],
                root_pwd=self.db_params_dict['karbor_root_pwd'],
                extra="")
            karbor_nodes_list.append(cbs_node)
        return karbor_nodes_list

    def get_dmk_params(self):
        dmk_params = models.DmkParams(
            account=self.db_params_dict['dmk_account'],
            tmp_account_pwd=self.db_params_dict['tmp_dmk_account_pwd'],
            account_pwd=self.db_params_dict['dmk_account_pwd'],
            account_group=self.db_params_dict['dmk_account_group'],
            float_ip=self.db_params_dict['DMK_dmk_floatIp'],
            ui_username=self.db_params_dict['DMK_dmk_ui_username'],
            ui_password=self.db_params_dict['DMK_dmk_ui_password'],
            os_business_username=self.db_params_dict[
                'DMK_dmk_os_business_username'],
            os_business_password=self.db_params_dict[
                'DMK_dmk_os_business_user_password']
        )
        return dmk_params

    def get_project_scene_params(self):
        project_scene = \
            models.ProjectScene(
                is_csha_scene=self.params_tools.is_csha_scene,
                is_x86_scene=self.params_tools.is_x86_scene,
                is_primary_region=self.params_tools.is_primary_region,
                is_global_dr_scene=self.params_tools.is_global_dr_scene,
                region_id=self.region_id)
        return project_scene

    def get_upgrade_path_params(self):
        ser_version_dict = self.get_upgrade_path_version()
        current_upgrade_path = self.get_current_upgrade_path()

        if current_upgrade_path in constant.UPGRADE_PATH["mo651_mo802"]:
            pkg_path, console_pkg_name = \
                self.params_tools.get_pkg_path_and_pkg_name(
                    "_CBS_Console_802.zip")
            console_pkg_version = \
                self.params_tools.get_pkg_version(pkg_path, console_pkg_name)
            karbor_pkg_name, karbor_pkg_version = None, None
        else:
            pkg_suffix = constant.PKG_SUFFIX_X86 \
                if self.params_tools.is_x86_scene else constant.PKG_SUFFIX_ARM
            pkg_path, karbor_pkg_name = \
                self.params_tools.get_pkg_path_and_pkg_name(pkg_suffix)
            karbor_pkg_version = \
                self.params_tools.get_pkg_version(pkg_path, karbor_pkg_name)

            pkg_path, console_pkg_name = \
                self.params_tools.get_pkg_path_and_pkg_name("_CBS_Console.zip")
            console_pkg_version = \
                self.params_tools.get_pkg_version(pkg_path, console_pkg_name)
        karbor_version = self.get_karbor_version()
        upgrade_path = \
            models.UpgradePath(
                current_upgrade_path=current_upgrade_path,
                current_version=karbor_version.get("short_version"),
                karbor_full_version=karbor_version.get("full_version"),
                original_version=ser_version_dict["original_version"],
                target_version=ser_version_dict["target_version"],
                pkg_path=pkg_path,
                karbor_pkg_name=karbor_pkg_name,
                karbor_pkg_version=karbor_pkg_version,
                console_pkg_name=console_pkg_name,
                console_pkg_version=console_pkg_version)
        return upgrade_path

    def _get_params(self):
        db_params_dict = {}
        service_items = self.db.get_service_cloud_param(self.pod_id,
                                                        self.service_name,
                                                        self.region_id)
        for key, value in service_items.items():
            db_params_dict[key] = value
        dmk_items = self.dmk_api.get_dmk_info(self.project_id)
        for key, value in dmk_items.items():
            db_params_dict[key] = value
        dmk_business_keys = ["DMK_dmk_os_business_username",
                             "DMK_dmk_os_business_user_password"]
        for key in dmk_business_keys:
            val = ParamUtil().get_param_value(self.pod_id, "DMK", key)
            if val:
                db_params_dict[key] = val

        db_params_dict.update(self.params_tools.get_karbor_vm_ips())
        return db_params_dict

    def get_current_upgrade_path(self):
        solution, version = \
            ProjectApi.get_solution_and_version(self.project_id)
        return '_'.join([solution, version])

    def get_karbor_version(self):
        ssh_client = self.ssh_client.get_sshclient_user_su_root(
            self.karbor_params.cbs_node0)
        karbor_version = dict()
        cmds_dict = {
            "short_version": "grep 'SHORT_VERSION' "
                             "/opt/huawei/dj/cfg/dj.version | cut -d '=' -f 2",
            "full_version": "grep 'FULL_VERSION' "
                            "/opt/huawei/dj/cfg/dj.version | cut -d ' ' -f 5"
        }
        for key, cmd in cmds_dict.items():
            result = \
                self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                raise Exception(
                    "Failed to execute cmd cat /opt/huawei/dj/cfg/dj.version.")
            karbor_version[key] = result[0]
        self.ssh_client.ssh_close(ssh_client)
        return karbor_version

    def get_upgrade_path_version(self, service=constant.BACKUP_SERVICE_NAME):
        solution, version = \
            ProjectApi.get_solution_and_version(self.project_id)
        sol_original_version, sol_target_version = version.strip().split("-")
        plugin_file_path = \
            "{}/{}".format(os.path.dirname(os.path.realpath(__file__)),
                           "../../plugin.json")
        ser_version_dict = dict()
        if os.path.exists(plugin_file_path):
            with open(plugin_file_path, 'r') as f:
                json_str = f.read()
                plugin_dic = json.loads(json_str)
                ser_version_dict["original_version"] = \
                    plugin_dic[solution][service]["VersionMatching"][
                        sol_original_version]
                ser_version_dict["target_version"] = \
                    plugin_dic[solution][service]["VersionMatching"][
                        sol_target_version]
                return ser_version_dict
        else:
            raise Exception("The 'plugin.json' file does not exist.")

    def is_last_region(self):
        region_list = self.get_cbs_regions()
        manage = ManageOneCmdbUtil(self.project_id)
        last_region = \
            manage.judge_last_upgrade_region('CSBS-VBS',
                                             self.get_upgrade_path_version(),
                                             self.region_id,
                                             region_list)
        if last_region:
            return True
        return False

    def get_cbs_regions(self):
        """get regions deploy csbs"""
        all_regions = BaseOps().get_all_regionid_by_projectid(self.project_id)
        mo_utils2 = ManageOneUtil2()
        iam_ip = mo_utils2.get_iam_ip(self.project_id)
        iam_token = mo_utils2.get_iam_token(iam_ip, self.project_id)
        cbs_region_list = []
        for region_id in all_regions:
            endpoint_info = mo_utils2.get_all_endpoints(iam_token, iam_ip,
                                                        self.region_id)
            for endpoint in endpoint_info['endpoints']:
                if endpoint['id'] == 'csbs':
                    cbs_region_list.append(region_id)
                    break
        return cbs_region_list

    @staticmethod
    def get_check_config(item):
        file_path = "{}/{}".format(os.path.dirname(os.path.realpath(__file__)),
                                   "../../config/upgrade_pre_check.json")
        if os.path.exists(file_path):
            with open(file_path, 'r', encoding='UTF-8') as f:
                json_str = f.read()
                check_dic = json.loads(json_str)
                return check_dic[item]
        else:
            raise Exception(
                "The 'upgrade_pre_check.json' file does not exist.")


class ParamsTools(object):
    def __init__(self, project_id):
        self.project_id = project_id

    def get_karbor_vm_names(self):
        vm_names = [
            constant.VMS_NAME['KARBOR1'],
            constant.VMS_NAME['KARBOR2'],
            constant.VMS_NAME['KARBOR3']
        ]
        vm_names_ha = [
            constant.VMS_NAME['KARBOR_HA1'],
            constant.VMS_NAME['KARBOR_HA2']
        ]
        return vm_names_ha if self.is_csha_scene else vm_names

    def get_karbor_vm_ips(self):
        vm_names = self.get_karbor_vm_names()
        vm_name_keys = ['karbor_host0', 'karbor_host1', 'karbor_host2']
        vm_ips = []
        for vm_name in vm_names:
            vm_info = ParamService().get_vm_info_by_vm_name(self.project_id,
                                                            vm_name)
            if not vm_info:
                raise Exception('Failed to getting the VM information based '
                                'on the VM name, vm name:{}.'.format(vm_name))
            vm_ips.append(vm_info[0].get('ip'))
        vm_ip_dict = dict(zip(vm_name_keys, vm_ips))
        return vm_ip_dict

    def get_console_vm_ips(self):
        vm_names = ['Console-CSBS01', 'Console-CSBS02']
        vm_names_ha = ['Console-CSBS', 'Console-CSBSdr']
        vm_names = vm_names_ha if self.is_csha_scene else vm_names

        vm_name_keys = ['console_host0', 'console_host1']
        vm_ips = []
        for vm_name in vm_names:
            vm_info = ParamService().get_vm_info_by_vm_name(self.project_id,
                                                            vm_name)
            if not vm_info:
                raise Exception('Failed to getting the VM information based '
                                'on the VM name, vm name:{}.'.format(vm_name))
            vm_ips.append(vm_info[0].get('ip'))
        vm_ip_dict = dict(zip(vm_name_keys, vm_ips))
        return vm_ip_dict

    @property
    def is_csha_scene(self):
        db_obj = ParamService()
        project_info = db_obj.get_project_info(self.project_id)
        base_options = project_info.get('base_options')
        if base_options.get('is_csha_scene'):
            return True
        return False

    @property
    def is_x86_scene(self):
        project_dict = get_project_conditions(self.project_id)
        if project_dict.get('is_x86_scene'):
            return True
        else:
            return False

    @property
    def is_global_dr_scene(self):
        project_dict = get_project_conditions(self.project_id)
        is_global_dr = project_dict.get("GlobalConDR")
        if is_global_dr:
            return True
        return False

    @property
    def is_primary_region(self):
        project_dict = get_project_conditions(self.project_id)
        is_primary = project_dict.get("PrimaryRegion")
        if is_primary:
            return True
        return False

    @staticmethod
    def get_pkg_version(pkg_path, pkg_name):
        pkg_path = os.path.join(pkg_path, pkg_name)
        file_fs = None
        try:
            file_fs = zipfile.ZipFile(r'{}'.format(pkg_path), 'r')
            for f in file_fs.namelist():
                if f.endswith(".json"):
                    return f.split("-").__getitem__(1)
        except Exception as e:
            raise Exception("Get package version failed, package name: {}, "
                            "err_msg: {}.".format(pkg_name, str(e)))
        finally:
            if file_fs:
                file_fs.close()

    @staticmethod
    def get_pkg_path_and_pkg_name(pkg_suffix):
        pkg_pre_name = "OceanStor BCManager"
        pkg_path, pkg_name = \
            FileUtil.find_software_package_by_name(pkg_pre_name=pkg_pre_name,
                                                   pkg_post_name=pkg_suffix,
                                                   pkg_version=None)
        if not pkg_name:
            raise Exception(
                "Can not find {}***{} pkg.".format(pkg_pre_name, pkg_suffix))
        return pkg_path, pkg_name

    def get_primary_region_id(self):
        return BaseOps().get_all_regionid_by_projectid(self.project_id)[0]
