import configparser
import os
import sys
from urllib.parse import quote

from platforms.param.param_manager import ParamService
from plugins.AcceptanceTest.basic_mo_test.common.login_sc import LoginSC
from plugins.CSBS_VBS.common.http_client import HttpClient
from plugins.CSBS_VBS.common.manageone import ManageOne
from utils.business.param_util import ParamUtil
from utils.common import log as logger
from utils.common.message import Message

logger.init("CSBS-VBS")
TASK_FILE = "{}/{}".format(os.path.dirname(os.path.realpath(__file__)),
                           "../../config/test_tasks.ini")


class SSOMoMixin(object):
    """Mainly used for login related operations."""

    def __init__(self, context, http_client):
        self.headers = {
            "Accept": "*/*",
            "X-Language": "zh-cn",
            "X-Requested-With": "XMLHttpRequest",
            "Connection": "keep-alive",
            "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                          "AppleWebKit/537.36 (KHTML, like Gecko) "
                          "Chrome/59.0.3071.104 Safari/537.36",
            "Content-Type": "application/json; charset=UTF-8"
        }
        self.http_client = http_client
        self.context = context
        self.mo = ManageOne(self.http_client, self.is_b2b, context)

        self.login_console()

    @property
    def is_b2b(self):
        return (self.context.basic_mo_test['manage_console_host'] ==
                self.context.basic_mo_test['console_host'] and 'bss_admin' in
                self.context.basic_mo_test['vdc1_vsm_user1'])

    @staticmethod
    def is_redirect(http_code):
        return http_code == 302

    def user_authentication(self):
        tenant_auth_url = "https://{host}/rest/mologin/v3.0/tenantAuth".format(
            host=self.context.basic_mo_test["iam_host"])

        req_body = (
            "userName={username}&password={password}&verifyCode=&"
            "locale=zh-cn&service=https%253A%252F%252F{host}%252F"
            "motenantconsolehomewebsite%252F".format(
                username=self.context.basic_mo_test['vdc1_vsm_user1'],
                password=quote(
                    self.context.basic_mo_test['sc_user_new_password']),
                host=self.context.basic_mo_test['console_host'])
        )
        _, rsp_body = self.http_client.post(
            tenant_auth_url, data=req_body, verify=False,
            headers={
                "Content-Type": "application/x-www-form-urlencoded; "
                                "charset=UTF-8",
                "Accept": "*/*"
            }
        )
        return rsp_body.get("url")

    def generate_st_tgt(self):
        third_login_url = self.user_authentication()
        if not third_login_url:
            raise Exception("Third login url is not found.")

        resp, _ = self.http_client.get(third_login_url,
                                       allow_redirects=False,
                                       headers=self.headers)
        if self.is_redirect(resp.status_code):
            # cookie: TGT id;
            # note: TGT(global session identifier) - Indicates that
            #       the user has logged in.
            self.headers['global_tgt'] = resp.headers.get('set-cookie')
            # location(redirect url): ST - Business system certification mark
            return resp.headers.get('Location')

    def generate_j_session_id(self):
        service_url = self.generate_st_tgt()
        if not service_url:
            raise Exception('Ticket url is not found.')

        resp, _ = self.http_client.get(service_url,
                                       allow_redirects=False,
                                       headers=self.headers)
        if not self.is_redirect(resp.status_code):
            raise Exception("The http request is not redirect.")

        ticket_cookie = resp.headers.get('set-cookie')

        if not self.is_b2b:
            self.headers['Cookie'] = ticket_cookie
            service_url = "https://{host}/motenantconsolehomewebsite/".format(
                host=self.context.basic_mo_test['console_host'])
            self.http_client.get(
                service_url, allow_redirects=False,
                headers={
                    'Cookie': ticket_cookie,
                    'Content-Type': 'application/x-www-form-urlencoded; '
                                    'charset=UTF-8',
                    'Accept': 'text/html,application/xhtml+xml,'
                              'application/xml;q=0.9,image/webp,*/*;q=0.8'
                }
            )
        else:
            agency_info = ticket_cookie.split('agencyID=')[1]
            agency_session = agency_info.split(';Path=/;Secure, J_SESSION_ID=')
            agency_id = agency_session[0]
            # J_SESSION_ID: sub service session identifier
            j_session_id = agency_session[1].split(
                ';Path=/;Secure;HttpOnly')[0]

            self.headers['AgencyId'] = agency_id

            self.headers['Cookie'] = (
                "theme=default; SID=Set2; "
                "agencyID={agencyid}; "
                "J_SESSION_ID={j_session_id}; "
                "locale=zh-cn; "
                "browserCheckResult=A".format(
                    agencyid=agency_id,
                    j_session_id=j_session_id)
            )

    def generate_cftk(self):
        self.generate_j_session_id()

        if not self.is_b2b:
            me_url = (
                "https://{host}/motenantconsolehomewebsite/rest/me?"
                "salt=0.4070987459363351".format(
                    host=self.context.basic_mo_test['console_host'])
            )
        else:
            me_url = (
                "https://{host}/moserviceaccesswebsite/rest/me?"
                "salt=0.4070987459363351".format(
                    host=self.context.basic_mo_test['console_host'])
            )
        _, rsp_body = self.http_client.get(me_url,
                                           allow_redirects=False,
                                           headers=self.headers)
        self.headers['cftk'] = rsp_body.get('cftk')
        self.headers['ProjectName'] = rsp_body.get('region')
        self.headers['AgencyId'] = rsp_body.get('id')

        region_name = self.mo.get_region_name()
        cloud_infras = self.mo.get_cloud_infras_by_region_name(
            self.context.basic_mo_test['console_host'], region_name,
            self.headers)
        if not cloud_infras:
            raise Exception('Cloud infra is not found.')
        self.headers["region"] = cloud_infras[0][0]

        user_info = {
            'project_id': rsp_body.get('projectId'),
            'user_id': rsp_body.get('userId'),
            'region_id': cloud_infras[0][0]
        }
        self.context.user_info = user_info

    def switch_project(self):
        if self.is_b2b:
            return

        projects_url = (
            "https://{host}/motenantconsolehomewebsite/rest/vdc/v3.1/users/"
            "{agencyid}/projects?start=0&limit=100&rel_projects=true".format(
                host=self.context.basic_mo_test['console_host'],
                agencyid=self.headers['AgencyId'])
        )
        _, rsp_body = self.http_client.get(projects_url, allow_redirects=False,
                                           headers=self.headers)
        projects = rsp_body.get('projects', [])
        for project in projects:
            region_temp = project.get('regions')[0]
            if region_temp.get('region_id') == self.headers['region']:
                self.context.user_info['project_id'] = project.get('id')
                self.headers['ProjectName'] = project.get('name')

    def login_console(self):
        tmp_retries = self.http_client.retries
        self.http_client.retries = 0
        try:
            self.generate_cftk()
            self.switch_project()
        except Exception as err:
            logger.trace()
            logger.error('Login cf2 console failed, msg: {}. '
                         'retry login vpc console.'.format(err))
            self.retry_login_vpc_console()

        self.headers['Referer'] = (
            "https://{console_host}/motenantconsolehomewebsite/".format(
                console_host=self.context.basic_mo_test['console_host'])
        )
        self.http_client.headers = self.headers
        self.http_client.retries = tmp_retries
        logger.info("Login mo console success.")

    def retry_login_vpc_console(self):
        self.headers['Cookie'] = self.headers['global_tgt']
        login_url = (
            "https://{host}/authui/login?service="
            "https%3A%2F%2F{service_host}%2Fvpc%2F%3FagencyId%3D"
            "{agencyid}%26region"
            "%3D{Region}%26locale%3Dzh-cn".format(
                host=self.context.basic_mo_test['iam_host'],
                service_host=self.context.basic_mo_test['console_host'],
                agencyid=self.headers['AgencyId'],
                Region=self.headers['ProjectName'])
        )
        rsp, _ = self.http_client.get(login_url, allow_redirects=False,
                                      headers=self.headers)
        vpc_location = rsp.headers.get('Location')

        rsp, _ = self.http_client.get(vpc_location, allow_redirects=False,
                                      headers=self.headers)
        if not self.is_b2b:
            vpc_j_session_id = rsp.headers.get('set-cookie').split(
                'J_SESSION_ID=')[1].split('; Version=1; Path=/; Secure;')[0]
            self.headers['Cookie'] += ';J_SESSION_ID=' + vpc_j_session_id
        else:
            vpc_j_session_id = rsp.headers.get('set-cookie').split(
                'Path=/; Secure, J_SESSION_ID=')[1].split(
                ';Path=/;Secure;HttpOnly')[0]
            self.headers['Cookie'] = (
                "theme=default; SID=Set2; agencyID={agencyid}; "
                "J_SESSION_ID={j_session_id}; locale=zh-cn; "
                "browserCheckResult=A".format(
                    agencyid=self.headers['AgencyId'],
                    j_session_id=vpc_j_session_id)
            )

        vpc_me_url = (
            "https://{host}/vpc/rest/me?salt=0.8809960674639905".format(
                host=self.context.basic_mo_test['console_host'])
        )
        _, rsp_body = self.http_client.get(vpc_me_url, allow_redirects=False,
                                           headers=self.headers)
        self.headers['cftk'] = rsp_body.get('cftk')


class ExecuteMixin(object):
    """Mainly used for task execution related operations."""

    def __init__(self, context, http_client):
        self.tasks = self.get_tasks()
        self.test_job_name = ""
        self.context = context
        self.http_client = http_client

    def get_tasks(self):
        conf = configparser.ConfigParser()
        conf.read(TASK_FILE)
        return dict(conf.items(self.test_job_name))

    def execute_test_job_tasks(self):
        if not self.tasks:
            logger.error('The job is not configured with tasks.')
            return

        for task_name, task_uri in self.tasks.items():
            logger.info("Start execute task: {task_name}.".format(
                task_name=task_name))
            module_name, _, class_fun_name = task_uri.rpartition('.')
            class_name = class_fun_name.split(':')[0]
            fun_name = class_fun_name.split(':')[1]

            __import__(module_name)
            cls = getattr(sys.modules[module_name], class_name, None)
            if not cls:
                raise Exception('Class({class_name}) cannot '
                                'be found.'.format(class_name=class_name))

            task_instance = cls(self.context, self.http_client)
            fun = getattr(task_instance, fun_name, None)
            if not fun:
                raise Exception('Fun({fun}) cannot be found.'.format(fun=fun))
            if not callable(fun):
                raise Exception('Fun({fun}) not is callable '
                                'object.'.format(fun=fun))
            fun()
            logger.info('Execute task({}) success.'.format(task_name))


class Context(object):
    """AB test context, mainly used to transfer parameters."""

    def __init__(self, engine_id):
        self.engine_id = engine_id
        self.engine_info = ParamService().get_project_info(engine_id)
        self.az_info = self.get_az_info(self.engine_info)
        self.az_id = self.az_info.get("azId")
        self.basic_mo_test = \
            self._wrap_get_service_cloud_param('Basic_MO_Test')
        self.basic_fs_test = \
            self._wrap_get_service_cloud_param('Basic_FS_Test')

    @staticmethod
    def get_az_info(engine_info):
        """This is special and requires the use of framework ab test methods

        :param engine_info:
        :return:
        """
        az_list = LoginSC(
            engine_info['project_id'],
            engine_info['region_id'],
            engine_info['region_type']).get_available_zone_name_v2()
        if az_list:
            return az_list[0]

    def _wrap_get_service_cloud_param(self, service_name):
        params = ParamUtil().get_service_cloud_param(
            self.engine_id, service_name, self.engine_info['region_id'])
        if not isinstance(params, dict):
            return params

        region_type = self.engine_info['region_type']
        return {key.replace(region_type + '_', '').replace(
            region_type.lower() + '_', ''
        ): value for key, value in params.items()}


class BaseTest(SSOMoMixin, ExecuteMixin):
    def __init__(self, engine_id):
        self.engine_id = engine_id
        self.context = Context(engine_id)
        self.http_client = HttpClient()
        self.http_client.login_fun = self.login_console
        SSOMoMixin.__init__(self, self.context, self.http_client)
        ExecuteMixin.__init__(self, self.context, self.http_client)

    def pre_check(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform resource pre-check before installation,
        this interface is called by the execute interface,
        The tool framework does not directly call this interface.
        """
        return Message(200)

    def execute(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform installation & configuration.
        """
        return Message(200)

    def rollback(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform job job failed rollback.
        """
        return Message(200)

    def retry(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform job failed retry.
        """
        self.rollback(engine_id, pod_id, regionid_list)
        return self.execute(engine_id, pod_id, regionid_list)

    def check(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Check before task execution.
        """
        return Message(200)
