import time
from datetime import datetime
from datetime import timedelta

import utils.common.log as logger
from plugins.CSBS_VBS.common import constant

logger.init("CSBS-VBS")


class PolicyTest(object):
    def __init__(self, context, http_client):
        self.context = context
        self.http_client = http_client
        self.console_host = self.context.basic_mo_test["console_host"]
        self.project_id = self.context.user_info['project_id']

    def csbs_create_policy(self):
        # assemble csbs resource_list
        server1_id = self._get_server_id_by_name(constant.VM_NAME)
        resource_list = [
            {
                "name": constant.VM_NAME,
                "id": server1_id,
                "type": "OS::Nova::Server"
            }
        ]
        policy_name = constant.CSBS_POLICY_NAME_PRE + time.strftime("%d%H%M")
        provider_id = constant.CSBS_PROVIDER_ID
        policy_info = self._create_policy(resource_list,
                                          policy_name,
                                          provider_id,
                                          is_enable_backup=True,
                                          is_enable_copy=False)
        setattr(self.context, 'csbs_policy_info', policy_info)

    def vbs_create_policy(self):
        # assemble vbs resource_list
        logger.info("Start to get volumes info by volume name, "
                    "volume name is {}.".format(constant.EVS_NAME))
        url = "https://{host}/ecm/rest/v2/detail/{project_id}" \
              "/volumes/detail?limit=100&offset=0" \
              "&name={vol_name}".format(host=self.console_host,
                                        project_id=self.project_id,
                                        vol_name=constant.EVS_NAME)
        _, volumes = self.http_client.get(url)
        volume1_id = volumes["volumes"][0]["id"]
        resource_list = [
            {
                "name": constant.EVS_NAME,
                "id": volume1_id,
                "type": "OS::Cinder::Volume"
            }
        ]
        policy_name = constant.VBS_POLICY_NAME_PRE + time.strftime("%d%H%M")
        provider_id = constant.VBS_PROVIDER_ID
        policy_info = self._create_policy(resource_list,
                                          policy_name,
                                          provider_id,
                                          is_enable_backup=True,
                                          is_enable_full_backup=False)
        setattr(self.context, 'vbs_policy_info', policy_info)

    def _create_policy(self, resource_list, policy_name, provider_id,
                       is_enable_backup=False, is_enable_copy=False,
                       is_enable_full_backup=False):
        expired_time = datetime.now() + timedelta(constant.POLICY_EXPIRED_TIME)
        expired_time_str = expired_time.strftime("%Y-%m-%dT%H:%M:%S")

        logger.info("Start to create a backup plan "
                    "that sets up a scheduling policy.")
        _, ret = self._post_create_policy_order(
            policy_name,
            resource_list,
            provider_id,
            expired_time=expired_time_str,
            is_enabled_backup=is_enable_backup,
            is_enabled_copy=is_enable_copy,
            is_enabled_full_backup=is_enable_full_backup)
        if ret["policy"]["name"] == policy_name and \
                ret["policy"]["resources"] == resource_list:
            logger.debug(
                "Backup plan has been created. plan name is: {policy_name}."
                "The name of the backup's resource: {resource_list},"
                "backup schedule expiration time: "
                "{expired_time}.".format(policy_name=policy_name,
                                         resource_list=resource_list,
                                         expired_time=expired_time_str))
        else:
            raise Exception("The backup plan created doesn't match the actual "
                            "backup plan information, please check.")
        policy_id = ret["policy"]["id"]

        resource_id_list = []
        for resource in resource_list:
            resource_id_list.append(resource["id"])

        policy_info = {
            "policy_id": policy_id,
            "resource_id_list": resource_id_list
        }

        return policy_info

    def _post_create_policy_order(self, plan_name, resources, provider_id,
                                  expired_time="", is_enabled_backup=False,
                                  is_enabled_copy=False,
                                  is_enabled_full_backup=False):

        url = "https://{console_host}/cbs/rest/karbor/v1/{project_id}/" \
              "policies".format(console_host=self.console_host,
                                project_id=self.project_id)

        scheduling_name_backup = "scheduling_name_backup"
        scheduling_name_copy = "scheduling_name_copy"
        scheduling_name_full_backup = "scheduling_name_full_backup"

        scheduled_operations = []
        if is_enabled_backup:
            scheduling_backup = \
                self._assemble_scheduled_operation(
                    scheduling_name=scheduling_name_backup,
                    is_enabled=is_enabled_backup,
                    operation_type='backup')
            scheduled_operations.append(scheduling_backup)
        if is_enabled_full_backup:
            scheduling_full_backup = \
                self._assemble_scheduled_operation(
                    scheduling_name=scheduling_name_full_backup,
                    is_enabled=is_enabled_full_backup,
                    operation_type='full_backup')
            scheduled_operations.append(scheduling_full_backup)
        if is_enabled_copy:
            scheduling_copy = \
                self._assemble_scheduled_operation(
                    scheduling_name=scheduling_name_copy,
                    is_enabled=is_enabled_copy,
                    operation_type='copy')
            scheduled_operations.append(scheduling_copy)

        body_dict = {
            "policy": {
                "name": plan_name,
                "description": "",
                "provider_id": provider_id,
                "parameters": {
                    "common": {
                        "az": self.context.az_id,
                        "expired_at": expired_time
                    }
                },
                "scheduled_operations": scheduled_operations,
                "resources": resources
            }
        }
        logger.debug(body_dict)

        return self.http_client.post(url, body=body_dict)

    @staticmethod
    def _assemble_scheduled_operation(scheduling_name, is_enabled,
                                      operation_type):
        max_backups = 20
        pattern_freq = "WEEKLY"
        pattern_by_day = "MO"
        pattern_by_hour = "10"
        pattern_by_minute = "20"

        trigger_pattern_str = "BEGIN:VCALENDAR\r\nBEGIN:VEVENT\r\n" \
                              "RRULE:FREQ={freq};BYDAY={by_day};" \
                              "BYHOUR={by_hour};BYMINUTE={by_minute}\r\n" \
                              "END:VEVENT\r\n" \
                              "END:VCALENDAR\r\n"
        trigger_pattern = \
            trigger_pattern_str.format(freq=pattern_freq,
                                       by_day=pattern_by_day,
                                       by_hour=pattern_by_hour,
                                       by_minute=pattern_by_minute)

        result_dict = {
            "name": scheduling_name,
            "description": "",
            "enabled": is_enabled,
            "operation_definition": {
                "max_backups": max_backups
            },
            "trigger": {
                "properties": {
                    "pattern": trigger_pattern
                }
            },
            "operation_type": operation_type
        }
        return result_dict

    def _get_server_id_by_name(self, server_name):
        server_id = None
        logger.info("Start to get all vms list information.")
        url = "https://{console_url}/ecm/rest/v1.0/servers?" \
              "offset=1&not-tags=__type_baremetal" \
              "&detail=3".format(console_url=self.console_host)
        _, server_list_info = self.http_client.get(url)
        logger.info("Succeed getting all vms list information.")

        for server_info in server_list_info["servers"]:
            if server_info["name"] == server_name:
                server_id = server_info["id"]
                break

        if not server_id:
            raise Exception("Failed to get tha vm's id with name, "
                            "vm name is {}.".format(server_name))
        return server_id

    def csbs_delete_policies(self):
        self._check_and_delete_policies(constant.CSBS_PROVIDER_ID)

    def vbs_delete_policies(self):
        self._check_and_delete_policies(constant.VBS_PROVIDER_ID)

    def _check_and_delete_policies(self, provider_id):
        policy_ids = self._get_policy_ids(provider_id)
        if policy_ids:
            logger.info("Start to delete the policy.")
            for policy_id in policy_ids:
                url = "https://{console_host}/cbs/rest/karbor" \
                      "/v1/{project_id}/policies" \
                      "/{policy_id}".format(console_host=self.console_host,
                                            project_id=self.project_id,
                                            policy_id=policy_id)
                self.http_client.delete(url)
            logger.info("Finished deleting the backup plan.")

    def _get_policy_ids(self, provider_id):
        logger.info("Get information of all policies under the tenant")
        url = "https://{console_host}/cbs/rest/karbor/" \
              "v1/{project_id}/policies?provider_id=" \
              "{provider_id}".format(console_host=self.console_host,
                                     project_id=self.project_id,
                                     provider_id=provider_id)
        _, ret = self.http_client.get(url)
        logger.info("Succeed getting information "
                    "of all policies:{}.".format(str(ret)))

        plan_id_list = []
        if not ret["policies"]:
            logger.info("After checking, there is no policy under the tenant.")
        else:
            for policies in ret["policies"]:
                plan_id_list.append(policies["id"])
            logger.info(
                "After checking, there are some policies under the tenant, "
                "plan ids: {}.".format(plan_id_list))
        return plan_id_list
