import json

import utils.common.log as logger
from plugins.CSBS_VBS.common import constant
from plugins.CSBS_VBS.job.ab_test.tasks.tenant import TenantApi


logger.init("CSBS-VBS")


class QuotaTest(object):
    def __init__(self, context, http_client):
        self.context = context
        self.http_client = http_client
        self.tenant_api = TenantApi(self.context, http_client)
        self.console_host = self.context.basic_mo_test["console_host"]
        self.project_id = self.context.user_info['project_id']

    def csbs_check_and_apply_quota(self):
        self._apply_for_quota(constant.ECS_PRODUCT_TYPE)

    def vbs_check_and_apply_quota(self):
        self._apply_for_quota(constant.EVS_PRODUCT_TYPE)

    def release_quota(self):
        logger.info("Start to get quota information.")
        quotas_info = self._get_quota_info()
        logger.info("Get quotas info successfully.")
        for quota in quotas_info["quotas"]["resources"]:
            quota_type = quota["type"]
            if quota_type in ["backup_capacity", "volume_backup_capacity"]:
                quota_num = quota["quota"]
                if quota_num != 0:
                    logger.info("Start to release quota, "
                                "quota type is {}.".format(quota_type))
                    quota_params = self._assemble_quota_params(quota_type,
                                                               int(-quota_num))
                    product_type = "ecs" if quota_type == "backup_capacity" \
                        else "evs"
                    self._modify_quota(product_type, quota_params)
                    logger.info("Succeed release quota, "
                                "quota type is {}.".format(quota_type))

    def _get_quota_info(self):
        url = "https://{console_host}/cbs/rest/karbor/v1/" \
              "{project_id}/quotas".format(console_host=self.console_host,
                                           project_id=self.project_id)
        _, quota_info = self.http_client.get(url)
        return quota_info

    def _modify_quota(self, product_type, quota_params):
        service_type = "csbs" if product_type == "ecs" else "vbs"
        product_id = self.tenant_api.get_product_id(product_type)
        url = "https://{console_host}/cbs/rest/karbor/" \
              "v1/{project_id}/products/{product_id}" \
              "/action".format(console_host=self.console_host,
                               project_id=self.project_id,
                               product_id=product_id)
        body_dict = {
            "apply": {
                "service_type": service_type,
                "region_id": self.context.engine_info['region_id'],
                "params": quota_params
            }
        }
        self.http_client.post(url, body=body_dict)

    def _apply_for_quota(self, product_type):
        if product_type not in ['ecs', 'evs']:
            raise Exception("The product_type should be ecs or evs.")

        logger.info("Start to get {product_type} quota "
                    "information.".format(product_type=product_type))
        quota_info = self._get_quota_info()

        is_need_add_quota = False
        cur_backup_quota = 0
        if product_type == 'ecs':
            quota_type = "backup_capacity"
        else:
            quota_type = "volume_backup_capacity"
        for quota in quota_info["quotas"]["resources"]:
            if quota["type"] == quota_type:
                cur_backup_quota = quota["quota"]
                if cur_backup_quota < constant.MINIMUM_QUOTA:
                    logger.info(
                        "The {product_type} backup capacity is {quota}M, "
                        "less than the established capacity.".format(
                            product_type=product_type,
                            quota=quota["quota"]))
                    is_need_add_quota = True
                    break
                else:
                    logger.info(
                        "The {product_type} backup capacity is {quota}M, "
                        "which can be used normally.".format(
                            product_type=product_type,
                            quota=quota["quota"]))
        if is_need_add_quota:
            logger.info("Start to apply for {product_type} "
                        "backup space.".format(product_type=product_type))
            quota_num = constant.MINIMUM_QUOTA - cur_backup_quota
            self._add_quota(product_type, quota_num)
            logger.info("Succeed in applying for {product_type} backup "
                        "space.".format(product_type=product_type))

    def _add_quota(self, product_type, quota_num):
        logger.info("Start to increase backup space.")
        if product_type == "ecs":
            capacity_type = "backup_capacity"
        else:
            capacity_type = "volume_backup_capacity"
        params = self._assemble_quota_params(capacity_type, quota_num)
        self._modify_quota(product_type, params)
        logger.info("Succeed in increasing backup space.")

    @staticmethod
    def _assemble_quota_params(capacity_type, var_quota_num):
        capacity_type_list = ['backup_capacity', 'copy_capacity',
                              'volume_backup_capacity', 'volume_copy_capacity']
        if capacity_type not in capacity_type_list:
            raise Exception("Capacity_type should be backup_capacity "
                            "or copy_capacity or volume_backup_capacity "
                            "or volume_copy_capacity.")

        if "backup_capacity" in capacity_type:
            display_en_label = "Backup"
            display_cn_label = "备份"
        else:
            display_en_label = "Replication"
            display_cn_label = "复制"

        params = {
            "count": 1,
            "param_items": [
                {
                    "type": capacity_type,
                    "quota": var_quota_num
                }
            ],
            "display": {
                "en_US": [
                    {
                        "label": display_en_label,
                        "type": "string",
                        "value": "xx"
                    }
                ],
                "zh_CN": [
                    {
                        "label": display_cn_label,
                        "type": "string",
                        "value": "xx"
                    }
                ]
            }
        }
        return json.dumps(params)
