import ipaddress

import utils.common.log as logger
from plugins.CSBS_VBS.common.ssh_client import SshClient
from plugins.CSBS_VBS.common.upgrade.params import ParamsTools
from plugins.CSBS_VBS.job.upgrade.base import BaseSubJob
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.common.exception import FCUException
from utils.common.message import Message

logger.init("CSBS-VBS")


class RegisterCMDB(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(RegisterCMDB, self).__init__(project_id, pod_id, regionid_list)
        self.ssh_client = SshClient()
        self.cmdb_util = ManageOneCmdbUtil(project_id)
        self.original_version = self.upgrade_path.original_version
        self.pkg_version = self.upgrade_path.karbor_pkg_version
        self.params_tools = ParamsTools(project_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            vm_names = self.params_tools.get_karbor_vm_names()
            vm_ips_dict = self.params_tools.get_karbor_vm_ips()
            zip_vm_name_and_ip = list(zip(vm_names, vm_ips_dict.values()))
            deploy_node_infos = [
                {"name": vm_name, "ipAddresses": [{"ip": vm_ip}]}
                for vm_name, vm_ip in zip_vm_name_and_ip
            ]
            # float ip to be registered should be the same with vim info in lld
            for deploy_node in deploy_node_infos:
                if deploy_node['name'].startswith('Service'):
                    float_ip_info = {
                        'floatIpAddresses': [
                            {
                                'floatIp': self.get_float_ip(),
                                'fixedIp': deploy_node['ipAddresses'][0]['ip']
                            }
                        ]
                    }
                    deploy_node.update(float_ip_info)
            # register CSBS-VBS service info
            self.set_csbs_cloud_service("CSBS-VBS", deploy_node_infos)
            # register CSBS-VBS-Console service info
            self.set_csbs_cloud_service("CSBS-VBS-Console")
        except Exception as e:
            logger.error("Failed executing register CMDB, "
                         "errmsg: {}.".format(e))
            return Message(500, FCUException(645022))
        logger.info("The backup service information has been "
                    "register succeed on CMDB.")
        return Message(200)

    def get_float_ip(self):
        cbs_node0 = self.karbor_params.cbs_node0
        logger.info("Get float_ip from the Karbor "
                    "node: {}.".format(cbs_node0.ip))
        ssh_client = self.ssh_client.get_sshclient_user_su_root(cbs_node0)
        cmds = 'get_info.py --tenant_float_ip'
        logger.info('Cmd: {}.'.format(cmds))
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmds)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            raise Exception("Failed to execute cmd: {}.".format(cmds))
        logger.info('Get karbor float ip info: {}.'.format(result))
        valid_ip_list = [item for item in result if self._is_valid_ip(item)]
        if valid_ip_list:
            karbor_float_ip = valid_ip_list[0].strip()
            logger.info('Karbor float ip is: {}.'.format(karbor_float_ip))
            return karbor_float_ip
        else:
            raise Exception("Get Karbor float ip failed.")

    @staticmethod
    def _is_valid_ip(ip):
        try:
            ipaddress.ip_address(ip)
            return True
        except Exception:
            logger.info('{} is not an IPv4 or IPv6 address.'.format(ip))
            return False

    def set_csbs_cloud_service(self, index_name, deploy_node_infos=None):
        """Register cloud services on CMDB

        """
        csbs_cloud_service = {
            "indexName": index_name,
            "version": self.pkg_version,
            "originalVersion": self.original_version,
            "extendInfos": [
                {"key": "dmk_floatIp",
                 "value": self.dmk_params.float_ip}
            ]
        }
        if deploy_node_infos:
            csbs_cloud_service["deploy_node_infos"] = deploy_node_infos

        logger.info('Current {} cloud service '
                    'infos: {}.'.format(index_name, csbs_cloud_service))
        logger.info("Start calling 'cmdb_util.set_cloud_service_info' method "
                    "to execute register CMDB.")

        primary_region_id = self.params_tools.get_primary_region_id()

        self.cmdb_util.set_cloud_service_info(
            self.regionid_list[0],
            csbs_cloud_service,
            input_master_region_code=primary_region_id)
        result = \
            self.cmdb_util.get_cloud_service_info(self.regionid_list[0],
                                                  index_name,
                                                  self.pkg_version)
        logger.info("Calling 'get_cloud_service_info' method to "
                    "get {} info, result:{}.".format(index_name, result))
        if not result:
            raise Exception("Failed calling 'get_cloud_service_info' "
                            "to get {} infos.".format(index_name))
